/**
 * AI Discovery Files — Admin JavaScript.
 *
 * Handles FAQ repeater, service repeater, key people repeater,
 * conditional fields, file preview, validation, and clipboard.
 *
 * @package AIDF
 * @since   1.0.0
 */

/* global jQuery, aidfAdmin */

(function ($) {
	'use strict';

	/**
	 * Escape a string for safe insertion into an HTML attribute.
	 *
	 * @param {string} str Raw string.
	 * @return {string} Escaped string.
	 */
	function escAttr(str) {
		var div = document.createElement('div');
		div.appendChild(document.createTextNode(str));
		return div.innerHTML;
	}

	/* -------------------------------------------------------
	   Initialise on DOM ready
	   ------------------------------------------------------- */
	$(function () {

		/* -------------------------------------------------------
		   FAQ Repeater
		   ------------------------------------------------------- */
		var faqList = $('#aidf-faq-list');
		var faqEmpty = $('#aidf-faq-empty');

		function getFaqCount() {
			return faqList.find('.aidf-faq-item').length;
		}

		function renumberFaqs() {
			faqList.find('.aidf-faq-item').each(function (index) {
				var $item = $(this);
				$item.attr('data-index', index);
				$item.find('.aidf-faq-item-number').text('Q' + (index + 1));
				$item.find('.aidf-faq-question').attr('name', 'aidf_settings[faqs][' + index + '][q]');
				$item.find('.aidf-faq-answer').attr('name', 'aidf_settings[faqs][' + index + '][a]');
			});

			if (getFaqCount() === 0) {
				faqEmpty.show();
			} else {
				faqEmpty.hide();
			}
		}

		$('#aidf-add-faq').on('click', function () {
			var index = getFaqCount();

			var html =
				'<div class="aidf-faq-item" data-index="' + index + '">' +
					'<div class="aidf-faq-item-header">' +
						'<span class="aidf-faq-item-number">Q' + (index + 1) + '</span>' +
						'<button type="button" class="aidf-faq-remove" title="' + escAttr(aidfAdmin.i18n.remove) + '">' +
							'<span class="dashicons dashicons-no-alt"></span>' +
						'</button>' +
					'</div>' +
					'<input type="text" name="aidf_settings[faqs][' + index + '][q]" class="aidf-faq-question" placeholder="' + escAttr(aidfAdmin.i18n.questionPlaceholder) + '">' +
					'<textarea name="aidf_settings[faqs][' + index + '][a]" class="aidf-faq-answer" placeholder="' + escAttr(aidfAdmin.i18n.answerPlaceholder) + '"></textarea>' +
				'</div>';

			faqList.append(html);
			faqEmpty.hide();

			// Focus the new question field.
			faqList.find('.aidf-faq-item').last().find('.aidf-faq-question').focus();
		});

		faqList.on('click', '.aidf-faq-remove', function () {
			$(this).closest('.aidf-faq-item').remove();
			renumberFaqs();
		});

		/* -------------------------------------------------------
		   Services Repeater (card layout)
		   ------------------------------------------------------- */
		var servicesList = $('#aidf-services-list');
		var servicesEmpty = $('#aidf-services-empty');

		function getServiceCount() {
			return servicesList.find('.aidf-service-item').length;
		}

		function renumberServices() {
			servicesList.find('.aidf-service-item').each(function (index) {
				var $item = $(this);
				$item.attr('data-index', index);
				$item.find('.aidf-service-item-number').text('S' + (index + 1));
				$item.find('.aidf-service-name').attr('name', 'aidf_settings[services][' + index + '][name]');
				$item.find('.aidf-service-url').attr('name', 'aidf_settings[services][' + index + '][url]');
			});

			if (getServiceCount() === 0) {
				servicesEmpty.show();
			} else {
				servicesEmpty.hide();
			}
		}

		$('#aidf-add-service').on('click', function () {
			var index = getServiceCount();

			var html =
				'<div class="aidf-service-item" data-index="' + index + '">' +
					'<div class="aidf-service-item-header">' +
						'<span class="aidf-service-item-number">S' + (index + 1) + '</span>' +
						'<button type="button" class="aidf-service-remove" title="' + escAttr(aidfAdmin.i18n.remove) + '">' +
							'<span class="dashicons dashicons-no-alt"></span>' +
						'</button>' +
					'</div>' +
					'<input type="text" name="aidf_settings[services][' + index + '][name]" class="aidf-service-name" placeholder="' + escAttr(aidfAdmin.i18n.servicePlaceholder) + '">' +
					'<input type="url" name="aidf_settings[services][' + index + '][url]" class="aidf-service-url" placeholder="' + escAttr(aidfAdmin.i18n.serviceUrlPlaceholder) + '">' +
				'</div>';

			servicesList.append(html);
			servicesEmpty.hide();

			servicesList.find('.aidf-service-item').last().find('.aidf-service-name').focus();
		});

		servicesList.on('click', '.aidf-service-remove', function () {
			$(this).closest('.aidf-service-item').remove();
			renumberServices();
		});

		/* -------------------------------------------------------
		   Key People Repeater (card layout)
		   ------------------------------------------------------- */
		var peopleList = $('#aidf-people-list');
		var peopleEmpty = $('#aidf-people-empty');

		function getPeopleCount() {
			return peopleList.find('.aidf-person-item').length;
		}

		function renumberPeople() {
			peopleList.find('.aidf-person-item').each(function (index) {
				var $item = $(this);
				$item.attr('data-index', index);
				$item.find('.aidf-person-item-number').text('P' + (index + 1));
				$item.find('.aidf-person-name').attr('name', 'aidf_settings[key_people][' + index + '][name]');
				$item.find('.aidf-person-role').attr('name', 'aidf_settings[key_people][' + index + '][role]');
				$item.find('.aidf-person-url').attr('name', 'aidf_settings[key_people][' + index + '][url]');
			});

			if (getPeopleCount() === 0) {
				peopleEmpty.show();
			} else {
				peopleEmpty.hide();
			}
		}

		$('#aidf-add-person').on('click', function () {
			var index = getPeopleCount();

			var html =
				'<div class="aidf-person-item" data-index="' + index + '">' +
					'<div class="aidf-person-item-header">' +
						'<span class="aidf-person-item-number">P' + (index + 1) + '</span>' +
						'<button type="button" class="aidf-person-remove" title="' + escAttr(aidfAdmin.i18n.remove) + '">' +
							'<span class="dashicons dashicons-no-alt"></span>' +
						'</button>' +
					'</div>' +
					'<input type="text" name="aidf_settings[key_people][' + index + '][name]" class="aidf-person-name" placeholder="' + escAttr(aidfAdmin.i18n.namePlaceholder) + '">' +
					'<input type="text" name="aidf_settings[key_people][' + index + '][role]" class="aidf-person-role" placeholder="' + escAttr(aidfAdmin.i18n.rolePlaceholder) + '">' +
				'<input type="url" name="aidf_settings[key_people][' + index + '][url]" class="aidf-person-url" placeholder="' + escAttr(aidfAdmin.i18n.personUrlPlaceholder) + '">' +
				'</div>';

			peopleList.append(html);
			peopleEmpty.hide();

			peopleList.find('.aidf-person-item').last().find('.aidf-person-name').focus();
		});

		peopleList.on('click', '.aidf-person-remove', function () {
			$(this).closest('.aidf-person-item').remove();
			renumberPeople();
		});

		/* -------------------------------------------------------
		   Conditional Fields
		   ------------------------------------------------------- */

		// Business Type → Company Number / Jurisdiction fields.
		var registeredTypes = ['Limited Company', 'Limited Liability Partnership', 'Public Limited Company', 'Corporation', 'LLC'];

		$('#aidf-business-type').on('change', function () {
			var val = $(this).val();
			if (registeredTypes.indexOf(val) !== -1) {
				$('#aidf-company-fields').slideDown(200);
			} else {
				$('#aidf-company-fields').slideUp(200);
			}
		});

		// VAT Registered toggle → VAT Number field.
		$('#aidf-vat-registered').on('change', function () {
			if (this.checked) {
				$('#aidf-vat-fields').slideDown(200);
			} else {
				$('#aidf-vat-fields').slideUp(200);
			}
		});

		// E-commerce toggle → Products URL field.
		$('#aidf-is-ecommerce').on('change', function () {
			if (this.checked) {
				$('#aidf-ecommerce-fields').slideDown(200);
			} else {
				$('#aidf-ecommerce-fields').slideUp(200);
			}
		});

		// Operating Hours Preset → Custom hours field.
		$('#aidf-hours-preset').on('change', function () {
			if ($(this).val() === 'custom') {
				$('#aidf-custom-hours-field').slideDown(200);
			} else {
				$('#aidf-custom-hours-field').slideUp(200);
			}
		});

		/* -------------------------------------------------------
		   File Preview
		   ------------------------------------------------------- */
		var previewPicker = $('#aidf-preview-picker');
		var previewOutput = $('#aidf-preview-output');
		var viewLiveLink = $('#aidf-view-live');

		function loadPreview(slug) {
			previewOutput.val(aidfAdmin.i18n.generating);

			$.post(aidfAdmin.ajaxUrl, {
				action: 'aidf_preview_file',
				nonce: aidfAdmin.nonce,
				file_slug: slug
			}, function (response) {
				if (response.success) {
					previewOutput.val(response.data.content);
					viewLiveLink.attr('href', response.data.url);
				} else {
					previewOutput.val(aidfAdmin.i18n.errorPrefix + (response.data || aidfAdmin.i18n.unknownError));
				}
			}).fail(function () {
				previewOutput.val(aidfAdmin.i18n.previewError);
			});
		}

		previewPicker.on('change', function () {
			loadPreview($(this).val());
		});

		// Set initial live link.
		if (previewPicker.length) {
			var firstSlug = previewPicker.val();

			if (firstSlug) {
				var fileTypes = {
					'llms-txt': 'llms.txt',
					'ai-txt': 'ai.txt',
					'ai-json': 'ai.json',
					'identity-json': 'identity.json',
					'brand-txt': 'brand.txt',
					'faq-ai-txt': 'faq-ai.txt',
					'developer-ai-txt': 'developer-ai.txt',
					'robots-ai-txt': 'robots-ai.txt',
					'llms-html': 'llms.html',
					'llm-txt': 'llm.txt'
				};
				var filename = fileTypes[firstSlug] || '';

				if (filename) {
					viewLiveLink.attr('href', window.location.origin + '/' + filename);
				}
			}
		}

		/* -------------------------------------------------------
		   Copy to Clipboard
		   ------------------------------------------------------- */
		$('#aidf-copy-preview').on('click', function () {
			var content = previewOutput.val();
			var $btn = $(this);

			if (navigator.clipboard && navigator.clipboard.writeText) {
				navigator.clipboard.writeText(content).then(function () {
					$btn.find('.dashicons').removeClass('dashicons-admin-page').addClass('dashicons-yes');
					setTimeout(function () {
						$btn.find('.dashicons').removeClass('dashicons-yes').addClass('dashicons-admin-page');
					}, 1500);
				});
			} else {
				previewOutput.select();
				document.execCommand('copy');
			}
		});

		/* -------------------------------------------------------
		   Validation
		   ------------------------------------------------------- */
		var validationResults = $('#aidf-validation-results');
		var validationOutput = $('#aidf-validation-output');

		$('#aidf-validate-btn').on('click', function () {
			var $btn = $(this);
			$btn.prop('disabled', true).find('.dashicons').removeClass('dashicons-yes-alt').addClass('dashicons-update');

			$.post(aidfAdmin.ajaxUrl, {
				action: 'aidf_validate_files',
				nonce: aidfAdmin.nonce
			}, function (response) {
				$btn.prop('disabled', false).find('.dashicons').removeClass('dashicons-update').addClass('dashicons-yes-alt');

				if (!response.success) {
					return;
				}

				var html = '';
				var allValid = true;

				$.each(response.data, function (slug, result) {
					if (slug === '_consistency') {
						// Consistency warnings.
						if (result.warnings && result.warnings.length) {
							html += '<h3 style="margin-top: 16px; font-size: 14px;">' + escAttr(aidfAdmin.i18n.crossFileConsistency) + '</h3>';
							$.each(result.warnings, function (i, msg) {
								html += '<div class="aidf-validation-item aidf-validation-item--warning">';
								html += '<span class="dashicons dashicons-warning"></span> ' + escAttr(msg);
								html += '</div>';
							});
							allValid = false;
						}
						return;
					}

					var icon = result.valid ? 'dashicons-yes' : 'dashicons-no';
					var cls = result.valid ? 'ok' : 'error';

					html += '<div class="aidf-validation-item aidf-validation-item--' + cls + '">';
					html += '<span class="dashicons ' + icon + '"></span> ';
					html += '<strong>' + escAttr(slug) + '</strong>';

					if (result.valid) {
						html += ' \u2014 ' + escAttr(aidfAdmin.i18n.valid);
					}

					html += '</div>';

					if (result.errors && result.errors.length) {
						$.each(result.errors, function (i, msg) {
							html += '<div class="aidf-validation-item aidf-validation-item--error" style="padding-left: 24px;">';
							html += '<span class="dashicons dashicons-no"></span> ' + escAttr(msg);
							html += '</div>';
						});
						allValid = false;
					}

					if (result.warnings && result.warnings.length) {
						$.each(result.warnings, function (i, msg) {
							html += '<div class="aidf-validation-item aidf-validation-item--warning" style="padding-left: 24px;">';
							html += '<span class="dashicons dashicons-warning"></span> ' + escAttr(msg);
							html += '</div>';
						});
					}
				});

				if (allValid) {
					html = '<div class="aidf-notice aidf-notice--success"><span class="dashicons dashicons-yes-alt"></span><div>' + escAttr(aidfAdmin.i18n.allValid) + '</div></div>' + html;
				}

				validationOutput.html(html);
				validationResults.show();

				// Scroll to results.
				$('html, body').animate({
					scrollTop: validationResults.offset().top - 80
				}, 300);
			}).fail(function () {
				$btn.prop('disabled', false).find('.dashicons').removeClass('dashicons-update').addClass('dashicons-yes-alt');
			});
		});

		/* -------------------------------------------------------
		   Sync attribution toggle to hidden field on other tabs
		   ------------------------------------------------------- */
		$('.aidf-attribution-toggle').on('change', function () {
			$('.aidf-hidden-attribution').val(this.checked ? '1' : '0');
		});

		/* -------------------------------------------------------
		   Page Picker — populate URL input from dropdown
		   ------------------------------------------------------- */
		$('.aidf-page-picker').on('change', function () {
			var $select = $(this);
			var url = $select.val();

			if (url) {
				$('#' + $select.data('target')).val(url);
				$select.val('');
			}
		});

		/* -------------------------------------------------------
		   Logo Media Picker
		   ------------------------------------------------------- */
		var logoFrame;
		var logoSelectBtn = $('#aidf-logo-select');
		var logoRemoveBtn = $('#aidf-logo-remove');
		var logoInput = $('#aidf-logo-url');
		var logoPreview = $('#aidf-logo-preview');

		logoSelectBtn.on('click', function (e) {
			e.preventDefault();

			if (logoFrame) {
				logoFrame.open();
				return;
			}

			logoFrame = wp.media({
				title: aidfAdmin.i18n.chooseLogo,
				button: { text: aidfAdmin.i18n.useThisImage },
				multiple: false,
				library: { type: 'image' }
			});

			logoFrame.on('select', function () {
				var attachment = logoFrame.state().get('selection').first().toJSON();
				var url = attachment.url;

				logoInput.val(url);
				logoPreview.find('img').attr('src', url);
				logoPreview.show();
				logoRemoveBtn.show();
				logoSelectBtn.find('.dashicons').next().addBack().last().text(' ' + aidfAdmin.i18n.changeLogo);
			});

			logoFrame.open();
		});

		logoRemoveBtn.on('click', function (e) {
			e.preventDefault();
			logoInput.val('');
			logoPreview.hide();
			logoRemoveBtn.hide();

			// Reset button text.
			logoSelectBtn.html('<span class="dashicons dashicons-format-image"></span> ' + escAttr(aidfAdmin.i18n.selectLogo));
		});

		/* -------------------------------------------------------
		   Domain Verification Modal
		   ------------------------------------------------------- */
		var verifyModal = $('#aidf-verify-modal');

		function openVerifyModal() {
			verifyModal.show();
			$('body').css('overflow', 'hidden');
			verifyModal.find('#aidf-verify-code').focus();
		}

		function closeVerifyModal() {
			verifyModal.hide();
			$('body').css('overflow', '');
		}

		// Open from CTA button or Status tab panel button.
		$(document).on('click', '#aidf-open-verify-cta, #aidf-open-verify-panel', function (e) {
			e.preventDefault();
			openVerifyModal();
		});

		// Close: X button, backdrop, Cancel button.
		verifyModal.on('click', '[data-aidf-close-modal]', function () {
			closeVerifyModal();
		});

		// Close: Escape key.
		$(document).on('keydown', function (e) {
			if (e.key === 'Escape' && verifyModal.is(':visible')) {
				closeVerifyModal();
			}
		});

		// Save verification code via AJAX.
		$('#aidf-verify-save').on('click', function () {
			var $btn = $(this);
			var code = $('#aidf-verify-code').val().trim();
			var $status = $('#aidf-verify-status');

			$btn.prop('disabled', true).find('.dashicons').removeClass('dashicons-yes').addClass('dashicons-update');

			$.post(aidfAdmin.ajaxUrl, {
				action: 'aidf_save_verify_code',
				nonce: aidfAdmin.nonce,
				verify_code: code
			}, function (response) {
				$btn.prop('disabled', false).find('.dashicons').removeClass('dashicons-update').addClass('dashicons-yes');

				if (!response.success) {
					$status
						.removeClass('aidf-verify-modal__status--active')
						.html('<span class="dashicons dashicons-warning"></span> <span>' + escAttr(response.data) + '</span>');
					return;
				}

				var data = response.data;

				if (data.active) {
					$status
						.addClass('aidf-verify-modal__status--active')
						.html(
							'<span class="dashicons dashicons-yes-alt"></span> ' +
							'<span>' + escAttr(aidfAdmin.i18n.verifyFileActive) + ' ' +
							'<a href="' + escAttr(data.url) + '" target="_blank" rel="noopener">' + escAttr(data.url) + '</a></span>'
						);
				} else {
					$status
						.removeClass('aidf-verify-modal__status--active')
						.html('<span class="dashicons dashicons-info-outline"></span> <span>' + escAttr(aidfAdmin.i18n.verifyFileInactive) + '</span>');
				}

				// Flash the save button green briefly.
				$btn.addClass('aidf-btn--success');
				setTimeout(function () {
					$btn.removeClass('aidf-btn--success');
				}, 1500);
			}).fail(function () {
				$btn.prop('disabled', false).find('.dashicons').removeClass('dashicons-update').addClass('dashicons-yes');
			});
		});

	}); // end DOM ready

})(jQuery);
