<?php
/*
Plugin Name: AI Alt Text Builder
Description: Generate SEO-friendly ALT text for Media Library images in one click, with scoring, bulk generation and language options.
Version:     1.0.7
Author:      RankPilotAI
Author URI:  https://rankpilotai.com
Text Domain: ai-alt-text-builder
Domain Path: /languages
Requires at least: 5.8
Tested up to:      6.9
Requires PHP:      7.4
License:           GPLv2 or later
License URI:       https://www.gnu.org/licenses/gpl-2.0.html
*/
if ( ! defined( 'ABSPATH' ) ) exit;

define( 'AATB_VERSION',       '1.0.7' );
define( 'AATB_PATH',          plugin_dir_path( __FILE__ ) );
define( 'AATB_URL',           plugin_dir_url ( __FILE__ ) );
define( 'AATB_SETTINGS_SLUG', 'aatb_settings' );

require_once AATB_PATH . 'includes/rp-http.php';   // HTTP helper (CA bundle + retries)
require_once AATB_PATH . 'includes/helper-functions.php';
require_once AATB_PATH . 'admin/settings-page.php';

/**
 * Activation defaults
 */
register_activation_hook( __FILE__, function () {
	$opt = get_option( AATB_SETTINGS_SLUG, [] );
	if ( ! is_array( $opt ) ) {
		$opt = [];
	}

	if ( ! array_key_exists( 'site_token', $opt ) ) {
		$opt['site_token'] = '';
	}

	if ( empty( $opt['lang_choice'] ) ) {
		$opt['lang_choice'] = 'en';
	}
	if ( empty( $opt['model_choice'] ) ) {
		$opt['model_choice'] = 'gpt-4o-mini';
	}

	update_option( AATB_SETTINGS_SLUG, $opt );
} );

/*───────────────────────────────────────────────────────────────
 *  Alt length score helper
 *───────────────────────────────────────────────────────────────*/
function aatb_length_score( string $alt ): int {
	$l = mb_strlen( trim( $alt ) );
	return match ( true ) {
		$l === 0  => 0,
		$l <= 125 => 100,
		$l <= 150 => 80,
		$l <= 175 => 60,
		$l <= 200 => 20,
		default   => 0,
	};
}

// === Model cost & remaining helpers (Token Guard) ===
if ( ! function_exists('aatb_model_cost') ) {
    function aatb_model_cost(string $model): int {
        $m = strtolower(trim($model));
        // Same mapping as Snippet: 4o=5, 4.1=3, mini=1
        return match ($m) {
            'gpt-4o'      => 5,
            'gpt-4.1'     => 3,
            'gpt-4o-mini' => 1,
            default       => 1, // safe default
        };
    }
}

if ( ! function_exists('aatb_model_tier') ) {
    function aatb_model_tier(string $model): string {
        $m = strtolower(trim($model));
        // Tiering consistent with Snippet
        return match ($m) {
            'gpt-4o'      => 'premium',
            'gpt-4.1'     => 'balanced',
            default => 'economical',
        };
    }
}

if ( ! function_exists('aatb_model_alias_pack') ) {
    function aatb_model_alias_pack(string $model): array {
        $m = strtolower(trim($model));
        switch ($m) {
            // BALANCED
            case 'gpt-4.1':
            case 'gpt-4-1':
            case 'gpt_4_1':
            case 'gpt41':
            case 'gpt4.1':
                return [
                    'canonical' => 'gpt-4.1',
                    'engine'    => 'gpt-4.1',
                    'family'    => 'gpt41',
                    'aliases'   => ['gpt-4.1','gpt-4-1','gpt_4_1','gpt41','gpt4.1'],
                ];

            // PREMIUM
            case 'gpt-4o':
            case 'gpt4o':
            case 'o4':
            case '4o':
                return [
                    'canonical' => 'gpt-4o',
                    'engine'    => 'gpt-4o',
                    'family'    => 'gpt4o',
                    'aliases'   => ['gpt-4o','gpt4o','o4','4o'],
                ];

            // ECONOMY (DEFAULT)
            default:
                return [
                    'canonical' => 'gpt-4o-mini',
                    'engine'    => 'gpt-4o-mini',
                    'family'    => 'gpt4omini',
                    'aliases'   => ['gpt-4o-mini','gpt4o-mini','4o-mini','mini'],
                ];
        }
    }
}

/** Vision-capable models */
function aatb_vision_models(): array {
    return [ 'gpt-4o-mini', 'gpt-4o', 'gpt-4.1' ];
}

/** Base prompt (custom text is appended if provided) */
function aatb_base_prompt( string $lang ): string {
	return 'You are an assistant that writes short, SEO-friendly ALT text for images in **'.$lang.'**.
Rules:
- Max 125 characters (hard limit).
- Focus: object (≈70%), material/texture (≈20%), dominant color (≈10%).
- No AI disclaimers. Avoid camera/brand names unless custom directives explicitly require them.
- If custom directives are provided, they OVERRIDE the above rules (except the 125-character limit).
Output:
- Return ONLY JSON: {"alt_text":"..."}';
}

final class AATB_Plugin {

	private static $ins;
	public static function get_instance() {
		return self::$ins ?: self::$ins = new self;
	}
	private function __construct() { $this->hooks(); }

	/* ……………………… HOOKS ……………………… */
	private function hooks() {
		add_action( 'admin_menu',            [ $this, 'menu'   ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'assets' ] );
		add_filter( 'attachment_fields_to_edit', [ $this, 'ui'   ], 10, 2 );
		add_filter( 'attachment_fields_to_save', [ $this, 'save' ], 10, 2 );
		add_filter( 'manage_upload_columns',      [ $this, 'add_col' ] );
		add_action( 'manage_media_custom_column', [ $this, 'col'     ], 10, 2 );
		add_action( 'wp_ajax_aatb_generate_alt',  [ $this, 'ajax' ] );
		add_filter( 'bulk_actions-upload',        [ $this, 'bulk_items'  ] );
		add_filter( 'handle_bulk_actions-upload', [ $this, 'bulk_handle' ], 10, 3 );
		add_action( 'admin_notices',              [ $this, 'bulk_notice' ] );
		add_action( 'restrict_manage_posts',      [ $this, 'filter_dd'   ] );
		add_filter( 'posts_clauses',              [ $this, 'filter_query' ], 10, 2 );
		add_filter( 'plugin_action_links_' . plugin_basename(__FILE__), [ $this, 'plugin_links' ] );
        add_action( 'wp_ajax_aatb_bulk_step', [ $this, 'bulk_step_ajax' ] );
        add_action( 'wp_ajax_aatb_bulk_cancel', [ $this, 'bulk_cancel_ajax' ] );
	}

	/* ……………………… MENU ……………………… */
	public function menu() {
    add_menu_page(
        'AI Alt Text Builder',
        'AI Alt Text Builder',
        'manage_options',
        'ai-alt-text-builder',
        'aatb_render_settings_page',
        plugin_dir_url(__FILE__) . 'admin/assets/img/ai-alt-text-builder.svg',
        25
    );
}

	/* ……………………… ASSETS ……………………… */
	public function assets( $hook ) {
		$need = in_array( $hook, [ 'upload.php', 'media.php' ], true )
      || ( strpos( (string) $hook, 'ai-alt-text-builder' ) !== false )
      || ( $hook === 'post.php' && function_exists('get_current_screen') && ( get_current_screen()->post_type ?? '' ) === 'attachment' );
		if ( ! $need ) return;

		wp_enqueue_style ( 'aatb-css', AATB_URL . 'admin/assets/css/admin.css', [], AATB_VERSION );
		// Dynamic menu icon color fix without admin_head <style>
$path = AATB_PATH . 'admin/assets/img/ai-alt-text-builder.svg';
$url  = AATB_URL  . 'admin/assets/img/ai-alt-text-builder.svg';
$ver  = file_exists($path) ? filemtime($path) : AATB_VERSION;
$icon = esc_url( $url . '?v=' . $ver );

$menu_css = '
#adminmenu #toplevel_page_ai-alt-text-builder .wp-menu-image img{display:none!important;}
#adminmenu #toplevel_page_ai-alt-text-builder .wp-menu-image{
  color:#a7aaad!important;background-color:currentColor!important;
  -webkit-mask:url("'.$icon.'") center/18px 18px no-repeat;
          mask:url("'.$icon.'") center/18px 18px no-repeat;
}
#adminmenu li#toplevel_page_ai-alt-text-builder:hover>a .wp-menu-image,
#adminmenu li#toplevel_page_ai-alt-text-builder.wp-has-current-submenu>a .wp-menu-image,
#adminmenu li#toplevel_page_ai-alt-text-builder.current>a .wp-menu-image{color:inherit!important;}
';
wp_add_inline_style( 'aatb-css', $menu_css );
		wp_enqueue_script( 'aatb-js',  AATB_URL . 'admin/assets/js/admin.js',
		                    [ 'jquery' ], AATB_VERSION, true );
		wp_localize_script( 'aatb-js', 'AATB_Ajax', [
			'url'   => admin_url( 'admin-ajax.php' ),
			'nonce' => wp_create_nonce( 'aatb_ajax' ),
		] );
	}

	/* ……………………… UI ……………………… */
	public function ui( $fields, $post ) {
		if ( ! isset( $fields['image_alt'] ) && ! isset( $fields['alt'] ) )
			return $fields;

		$alt  = get_post_meta( $post->ID, '_wp_attachment_image_alt', true );
		$sc   = aatb_length_score( $alt );
		$col  = $sc >= 100 ? '#0091e0' : ( $sc >= 67 ? '#ffab00' : '#dc3232' );
$btn_label = esc_html__( 'Generate with AI Alt Text Builder', 'ai-alt-text-builder' );
$html = '<div style="margin-top:6px">
  <button type="button" class="button button-secondary aatb-gen"
    data-id="'.esc_attr( (string) $post->ID ).'">'.$btn_label.'</button>
  <span id="aatb-status-'.esc_attr( (string) $post->ID ).'" style="margin-left:8px"></span></div>
  <div style="margin-top:6px">
  <span style="background:'.esc_attr($col).';color:#fff;font-weight:600;border-radius:3px;
  padding:3px 6px;font-size:11px">'.esc_html( (string) $sc ).' / 100</span></div>';

		$key = isset( $fields['image_alt'] ) ? 'image_alt' : 'alt';
		$fields[ $key ]['html'] .= $html;
		return $fields;
	}

	/* ……………………… SAVE ……………………… */
	public function save( $post, $att ) {
		if ( isset( $att['alt'] ) ) {
	if ( ! current_user_can( 'edit_post', (int) $post['ID'] ) ) {
		return $post;
	}
	$alt = sanitize_text_field( wp_unslash( $att['alt'] ) );
	update_post_meta( $post['ID'], '_wp_attachment_image_alt', $alt );
	update_post_meta( $post['ID'], '_aatb_score', aatb_length_score( $alt ) );
}
		return $post;
	}

	/* ……………………… LIST COLUMN ……………………… */
	public function add_col( $c ) {
	$c['aatb_score'] = esc_html__( 'Alt Score', 'ai-alt-text-builder' );
	return $c;
}

	public function col( $col, $id ) {
		if ( $col !== 'aatb_score' ) return;
		$alt = get_post_meta( $id, '_wp_attachment_image_alt', true );
		$s   = (int) get_post_meta( $id, '_aatb_score',          true );
		if ( $alt === '' ) $s = 0;
		$clr = $s >= 100 ? '#0091e0' : ( $s >= 67 ? '#ffab00' : '#dc3232' );
		echo '<span style="background:'.esc_attr($clr).';padding:4px 7px;border-radius:3px;color:#fff;font-size:11px;font-weight:600;">'
   . esc_html( (string) $s ) . ' / 100</span>';
	}

	/* ……………………… AJAX ……………………… */
	public function ajax() {
		if ( ! current_user_can( 'upload_files' ) )
			wp_send_json_error( [ 'msg' => 'permission' ], 403 );

		check_ajax_referer( 'aatb_ajax', 'nonce' );

		$id = absint( wp_unslash( $_POST['id'] ?? 0 ) );
		if ( ! $id )
			wp_send_json_error( [ 'msg' => 'invalid id' ] );

		try {
			$data = $this->generate_via_rankpilot( $id );
			wp_send_json_success( $data );
		} catch ( \Exception $e ) {
			wp_send_json_error( [ 'msg' => $e->getMessage() ] );
		}
	}

	/* ———————————————————————————————————————————
	 *  RankPilotAI API bridge
	 * ——————————————————————————————————————————— */
	private function generate_via_rankpilot( int $attach_id ): array {

		$opt   = get_option( AATB_SETTINGS_SLUG, [] );
		$token = $opt['site_token'] ?? '';
		if ( ! $token )
			throw new \Exception( 'No site token (Settings ▸ Site Key tab)' );

		$stat = aatb_check_token_status( $token );
		if ( ! empty( $stat['error'] ) ) {
			throw new \Exception( 'Token check: '.$stat['error'] );
		}

		$model     = $opt['model_choice'] ?? 'gpt-4o-mini';
		$lang      = $opt['lang_choice']  ?? 'en';
		$need      = aatb_model_cost( $model );
		$remaining = aatb_status_remaining( $stat );

		// Token guard
		if ( $remaining <= 0 ) {
			throw new \Exception( __( 'No tokens left. Please top up.', 'ai-alt-text-builder' ) );
		}
		if ( $remaining < $need ) {
			throw new \Exception( __( 'Not enough tokens for the selected model.', 'ai-alt-text-builder' ) );
		}

		// Vision check (UI only offers supported models, but keep this defensive)
		if ( ! in_array( strtolower($model), aatb_vision_models(), true ) ) {
			throw new \Exception( sprintf(
	__( 'Selected model (%s) does not support image analysis.', 'ai-alt-text-builder' ),
	esc_html( $model )
) );
		}

		$file     = wp_get_attachment_metadata( $attach_id );
		$filename = basename( $file['file'] ?? get_attached_file( $attach_id ) );
		$curr_alt = get_post_meta( $attach_id, '_wp_attachment_image_alt', true );

		$img_url = wp_get_attachment_url( $attach_id );
		if ( ! $img_url )
			throw new \Exception( __( 'Cannot retrieve image URL.', 'ai-alt-text-builder' ) );

		$base = aatb_base_prompt( $lang );
$cust = trim( (string) ( $opt['custom_prompt'] ?? '' ) );

$prompt = $base;
if ( $cust !== '' ) {
    $prompt .= "\n\nCustom Directives (OVERRIDE base rules except the 125-character limit):\n{$cust}\n\n"
             . "If a directive asks to append a specific suffix/prefix, ensure the returned alt_text includes it "
             . "exactly; shorten earlier words first to keep ≤125 characters.";
}

$payload = [
    'image_url'   => esc_url_raw( $img_url ),
    'file_name'   => $filename,
    'alt_current' => $curr_alt,
    'model'       => $model,
    'language'    => $lang,
    'lang'        => $lang,   // b/c
    'prompt'        => $prompt,
    'custom_prompt' => $cust,
];

		$res = aatb_call_rankpilot_api( $token, $payload, '/alt-text', 'POST' );

		if ( empty( $res['alt_text'] ) )
			throw new \Exception( 'RankPilot response empty' );

		$alt = sanitize_text_field( $res['alt_text'] );

		update_post_meta( $attach_id, '_wp_attachment_image_alt', $alt );
		update_post_meta( $attach_id, '_aatb_score', aatb_length_score( $alt ) );

		return [ 'alt' => $alt ];
	}

	/* ………………… BULK + LIST FILTERS ……………… */
	public function bulk_items( $a ) {
  $a['aatb_bulk_generate'] = esc_html__( 'Generate with AI Alt Text Builder', 'ai-alt-text-builder' );
  return $a;
}

	public function bulk_handle( $redirect, $action, $ids ) {
		if ( $action !== 'aatb_bulk_generate' ) return $redirect;

		$opt   = get_option( AATB_SETTINGS_SLUG, [] );
		$token = $opt['site_token'] ?? '';
		if ( ! $token ) return add_query_arg( 'aatb_berror', 'notoken', $redirect );

		$stat = aatb_check_token_status( $token );
		if ( ! empty( $stat['error'] ) ) {
			return add_query_arg( 'aatb_berror', 'status', $redirect );
		}

		$model     = $opt['model_choice'] ?? 'gpt-4o-mini';
		if ( ! in_array(strtolower($model), aatb_vision_models(), true) ) {
			return add_query_arg( 'aatb_binsupported', 1, $redirect );
		}
		$need      = aatb_model_cost( $model );
		$remaining = aatb_status_remaining( $stat );

		if ( $remaining <= 0 ) {
			return add_query_arg( 'aatb_bzerotok', 1, $redirect );
		}
		if ( $remaining < $need ) {
			return add_query_arg( 'aatb_binsufficient', 1, $redirect );
		}

		$job = [
			'ids'   => array_values( array_map( 'intval', (array) $ids ) ),
			'done'  => 0,
			'fail'  => 0,
			'model' => $model,
			'lang'  => $opt['lang_choice']  ?? 'en',
			'total' => count( $ids ),
		];
		$job_id = wp_generate_password( 12, false, false );
		set_transient( 'aatb_job_' . $job_id, $job, HOUR_IN_SECONDS );

		return add_query_arg( [ 'aatb_job' => $job_id ], $redirect );
	}

	public function bulk_step_ajax() {
		if ( ! current_user_can( 'upload_files' ) )
			wp_send_json_error( [ 'msg' => 'permission' ], 403 );

		check_ajax_referer( 'aatb_ajax', 'nonce' );

		$job_id = sanitize_text_field( $_POST['job'] ?? '' );
		$job    = get_transient( 'aatb_job_' . $job_id );
		if ( ! $job || empty( $job['ids'] ) )
			wp_send_json_error( [ 'msg' => 'no_job' ], 404 );

		if ( ! empty( $job['cancel'] ) ) {
			delete_transient( 'aatb_job_' . $job_id );
			wp_send_json_success( [
				'job'       => $job_id,
				'done'      => (int) $job['done'],
				'fail'      => (int) $job['fail'],
				'remaining' => 0,
				'processed' => (int) ($job['done'] + $job['fail']),
				'total'     => (int) ($job['total'] ?? ($job['done'] + $job['fail'])),
				'cancelled' => 1,
			] );
		}

		$chunk = (int) apply_filters( 'aatb_bulk_chunk', 6 );
		$take  = array_splice( $job['ids'], 0, $chunk );

		$urls = [];
		foreach ( $take as $aid ) {
			$u = wp_get_attachment_url( $aid );
			if ( $u ) { $urls[$aid] = $u; } else { $job['fail']++; }
		}

		if ( $urls ) {
			try {
				$payload = [
					'images'     => array_values( $urls ),
					'language'   => $job['lang'],
					'lang'       => $job['lang'],
					'model'      => $job['model'],
					'pool'       => 3,
				];
				$opt   = get_option( AATB_SETTINGS_SLUG, [] );
$lang  = $job['lang'];
$base  = aatb_base_prompt( $lang );
$cust  = trim( (string) ( $opt['custom_prompt'] ?? '' ) );

$prompt = $base;
if ( $cust !== '' ) {
    $prompt .= "\n\nCustom Directives (OVERRIDE base rules except the 125-character limit):\n{$cust}\n\n"
             . "If a directive asks to append a specific suffix/prefix, ensure the returned alt_text includes it "
             . "exactly; shorten earlier words first to keep ≤125 characters.";
}

$payload['prompt']        = $prompt;
$payload['custom_prompt'] = $cust;
				$res = aatb_call_rankpilot_api(
					( get_option( AATB_SETTINGS_SLUG, [] )['site_token'] ?? '' ),
					$payload, '/alt-batch', 'POST'
				);

				if ( ! empty( $res['results'] ) && is_array( $res['results'] ) ) {
					$rev = array_flip( $urls );
					foreach ( $res['results'] as $row ) {
						$url = $row['image_url'] ?? '';
						$id  = $rev[ $url ] ?? 0;
						if ( $id && ! empty( $row['ok'] ) && ! empty( $row['alt_text'] ) ) {
							$alt = sanitize_text_field( $row['alt_text'] );
							update_post_meta( $id, '_wp_attachment_image_alt', $alt );
							update_post_meta( $id, '_aatb_score', aatb_length_score( $alt ) );
							$job['done']++;
						} else {
							$job['fail']++;
						}
					}
				} else {
					$job['fail'] += count( $urls );
				}
			} catch ( \Exception $e ) {
				$job['fail'] += count( $urls );
			}
		}

		$remain = count( $job['ids'] );
		if ( $remain > 0 ) {
			set_transient( 'aatb_job_' . $job_id, $job, HOUR_IN_SECONDS );
		} else {
			delete_transient( 'aatb_job_' . $job_id );
		}

		$remain    = count( $job['ids'] );
		$total     = (int) ($job['total'] ?? ($remain + $job['done'] + $job['fail']));
		$processed = (int) ($job['done'] + $job['fail']);

		wp_send_json_success( [
			'job'       => $job_id,
			'done'      => (int) $job['done'],
			'fail'      => (int) $job['fail'],
			'remaining' => (int) $remain,
			'processed' => $processed,
			'total'     => $total,
		] );
	}

	public function bulk_cancel_ajax() {
		if ( ! current_user_can( 'upload_files' ) )
			wp_send_json_error( [ 'msg' => 'permission' ], 403 );
		check_ajax_referer( 'aatb_ajax', 'nonce' );

		$job_id = sanitize_text_field( $_POST['job'] ?? '' );
		$job    = get_transient( 'aatb_job_' . $job_id );
		if ( ! $job ) wp_send_json_error( [ 'msg' => 'no_job' ], 404 );

		$job['cancel'] = 1;
		set_transient( 'aatb_job_' . $job_id, $job, HOUR_IN_SECONDS );
		wp_send_json_success( ['ok'=>true] );
	}

	/* ………………… BULK NOTICES ………………… */
	public function bulk_notice() {

		if ( isset($_GET['aatb_job']) ) {
    $job   = sanitize_text_field( $_GET['aatb_job'] );
    $nonce = wp_create_nonce('aatb_ajax');

    echo '<div class="notice notice-info"><p><strong>AI Alt Text Builder:</strong> Running bulk job… 
            <span id="aatb-bulk-p">processed 0/0 - success 0, failed 0</span>
            <button type="button" class="button" id="aatb-bulk-cancel" style="margin-left:10px">Cancel</button>
          </p></div>';
}

		if ( isset($_GET['aatb_berror']) && $_GET['aatb_berror'] === 'notoken' ) {
			echo '<div class="notice notice-error is-dismissible"><p><strong>AI Alt Text Builder:</strong> No Site Key configured. '
			   . '<a href="' . esc_url( admin_url('admin.php?page=ai-alt-text-builder&tab=site-key') ) . '">Add your Site Key</a> to continue.</p></div>';
		}

		if ( ! empty($_GET['aatb_bzerotok']) ) {
			echo '<div class="notice notice-error is-dismissible"><p><strong>AI Alt Text Builder:</strong> Failed: no tokens left. '
			   . '<a href="' . esc_url( home_url('/tokens-1/?plugin=ai-alt-text-builder') ) . '" target="_blank" rel="noopener">Buy Tokens</a> to continue.</p></div>';
		}

		if ( ! empty($_GET['aatb_binsufficient']) ) {
			echo '<div class="notice notice-warning is-dismissible"><p><strong>AI Alt Text Builder:</strong> Not enough tokens for the selected model. '
			   . 'Please switch to a cheaper model from '
			   . '<a href="' . esc_url( admin_url('admin.php?page=ai-alt-text-builder&tab=alt-settings') ) . '">Alt-Text Settings</a> '
			   . 'or <a href="' . esc_url( home_url('/tokens-1/?plugin=ai-alt-text-builder') ) . '" target="_blank" rel="noopener">Buy Tokens</a>.</p></div>';
		}

		if ( ! empty($_GET['aatb_binsupported']) ) {
			echo '<div class="notice notice-warning is-dismissible"><p><strong>AI Alt Text Builder:</strong> '
			   . 'The selected model does not support image analysis. '
			   . 'Please choose <strong>GPT-4o-mini</strong> or <strong>GPT-4o</strong> or <strong>GPT-4.1</strong> from '
			   . '<a href="' . esc_url( admin_url('admin.php?page=ai-alt-text-builder&tab=alt-settings') ) . '">Alt-Text Settings</a>.</p></div>';
		}

				if ( isset( $_GET['aatb_done'] ) || isset( $_GET['aatb_fail'] ) ) {
			$d = isset( $_GET['aatb_done'] ) ? (int) $_GET['aatb_done'] : 0;
			$f = isset( $_GET['aatb_fail'] ) ? (int) $_GET['aatb_fail'] : 0;

			if ( $d > 0 ) {
				printf(
					'<div class="notice notice-success is-dismissible"><p><strong>%s:</strong> %s</p></div>',
					esc_html__( 'AI Alt Text Builder', 'ai-alt-text-builder' ),
					sprintf(
						esc_html__( 'Generated for %d image(s).', 'ai-alt-text-builder' ),
						$d
					)
				);
			}

			if ( $f > 0 && empty( $_GET['aatb_bzerotok'] ) ) {
				printf(
					'<div class="notice notice-error is-dismissible"><p><strong>%s:</strong> %s</p></div>',
					esc_html__( 'AI Alt Text Builder', 'ai-alt-text-builder' ),
					sprintf(
						esc_html__( 'Failed for %d image(s).', 'ai-alt-text-builder' ),
						$f
					)
				);
			}
		}
	}

	/* ………………… MEDIA LIST FILTER ……………… */
	public function filter_dd() {
		if ( get_current_screen()->post_type !== 'attachment' ) return;
		$sel = isset($_GET['aatb_alt_filter']) ? sanitize_key( wp_unslash( $_GET['aatb_alt_filter'] ) ) : '';
		echo '<select name="aatb_alt_filter">
		<option value="">'.esc_html__( 'AI Alt Text Builder', 'ai-alt-text-builder' ).'</option>
		<option value="with"     '.selected( $sel, 'with', false ).'>'.esc_html__( 'With Alt Text', 'ai-alt-text-builder' ).'</option>
		<option value="without"  '.selected( $sel, 'without', false ).'>'.esc_html__( 'Without Alt Text', 'ai-alt-text-builder' ).'</option>
	  </select>';
	}
	public function filter_query( $clauses, $query ) {
		global $wpdb;
		if ( $query->get( 'post_type' ) !== 'attachment' ) return $clauses;
		$f = isset($_GET['aatb_alt_filter']) ? sanitize_key( wp_unslash( $_GET['aatb_alt_filter'] ) ) : '';
		if ( $f === 'with' ) {
			$clauses['where'] .=" AND EXISTS(SELECT 1 FROM {$wpdb->postmeta}
				WHERE post_id={$wpdb->posts}.ID AND meta_key='_wp_attachment_image_alt' AND meta_value!='')";
		} elseif ( $f === 'without' ) {
			$clauses['where'] .=" AND NOT EXISTS(SELECT 1 FROM {$wpdb->postmeta}
				WHERE post_id={$wpdb->posts}.ID AND meta_key='_wp_attachment_image_alt' AND meta_value!='')";
		}
		return $clauses;
	}

	/* ………………… PLUGIN ACTION LINK ……………… */
	public function plugin_links( $links ) {
	$settings_link = '<a href="admin.php?page=ai-alt-text-builder">' . esc_html__( 'Settings', 'ai-alt-text-builder' ) . '</a>';
	array_unshift( $links, $settings_link );
	return $links;
   }
}
AATB_Plugin::get_instance();

