<?php
/**
 * Main Plugin Class with OAuth Authentication
 *
 * @package Agents24x7
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Agents24x7_Plugin
 *
 * Main plugin class using OAuth authentication
 */
class Agents24x7_Plugin {
    
    /**
     * Auth handler
     *
     * @var Agents24x7_Auth
     */
    private $auth;
    
    /**
     * Integration handler
     *
     * @var Agents24x7_Integration
     */
    private $integration;
    
    /**
     * API Proxy handler
     *
     * @var Agents24x7_API_Proxy
     */
    private $api_proxy;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->auth = new Agents24x7_Auth();
        $this->integration = new Agents24x7_Integration();
        $this->api_proxy = new Agents24x7_API_Proxy( $this->auth );
        
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        
        // AJAX handlers
        add_action( 'wp_ajax_agents24x7_get_wp_credentials', array( $this, 'ajax_get_wp_credentials' ) );
        add_action( 'wp_ajax_agents24x7_get_woocommerce_credentials', array( $this, 'ajax_get_woocommerce_credentials' ) );
        add_action( 'wp_ajax_agents24x7_disconnect', array( $this, 'ajax_disconnect' ) );
        add_action( 'wp_ajax_agents24x7_get_auth_status', array( $this, 'ajax_get_auth_status' ) );
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            'Agents24x7',
            'Agents24x7',
            'edit_posts',
            'agents24x7',
            array( $this, 'admin_page' ),
            'dashicons-lightbulb',
            25
        );
    }
    
    /**
     * Enqueue scripts
     *
     * @param string $hook Current admin page
     */
    public function enqueue_scripts( $hook ) {
        if ( $hook !== 'toplevel_page_agents24x7' ) {
            return;
        }
        
        // Check if development mode is active (webpack dev server running)
        $dev_server_url = 'http://localhost:3001/admin.js';
        $is_dev_mode = defined( 'WP_DEBUG' ) && WP_DEBUG && $this->is_dev_server_running();
        
        if ( $is_dev_mode ) {
            // Development mode - load from webpack dev server
            wp_enqueue_script(
                'agents24x7-admin',
                $dev_server_url,
                array( 'wp-element' ),
                time(), // Use timestamp to prevent caching
                true
            );
        } else {
            // Production mode - load built assets
            wp_enqueue_script(
                'agents24x7-admin',
                AGENTS24X7_PLUGIN_URL . 'admin/js/dist/admin.js',
                array( 'wp-element' ),
                AGENTS24X7_VERSION,
                true
            );
        }
        
        wp_enqueue_style(
            'agents24x7-admin',
            AGENTS24X7_PLUGIN_URL . 'admin/css/admin.css',
            array(),
            AGENTS24X7_VERSION
        );
        
        // Localize script with data
        wp_localize_script( 'agents24x7-admin', 'agents24x7_data', array(
            'ajaxUrl'         => admin_url( 'admin-ajax.php' ),
            'restUrl'         => rest_url( 'agents24x7/v1' ),
            'nonce'           => wp_create_nonce( 'agents24x7_nonce' ),
            'wpNonce'         => wp_create_nonce( 'wp_rest' ),
            'isAuthenticated' => $this->auth->is_authenticated(),
            'connectUrl'      => $this->auth->get_connect_url(),
            'convexCloudUrl'  => $this->get_convex_cloud_url(),
            'convexSiteUrl'   => $this->get_convex_site_url(),
            'appUrl'          => $this->get_app_url(),
            'wpSiteInfo'      => array(
                'name'        => get_bloginfo( 'name' ),
                'url'         => home_url(),
                'description' => get_bloginfo( 'description' ),
            ),
            'woocommerceInfo' => $this->integration->get_woocommerce_info(),
            'userId'          => get_current_user_id(),
        ) );
    }
    
    /**
     * Check if webpack dev server is running
     *
     * @return bool
     */
    private function is_dev_server_running() {
        $dev_server_url = 'http://localhost:3001';
        $context = stream_context_create( array(
            'http' => array(
                'timeout'        => 1,
                'ignore_errors'  => true,
            ),
        ) );
        
        $result = @file_get_contents( $dev_server_url, false, $context );
        return $result !== false;
    }
    
    /**
     * Admin page output
     */
    public function admin_page() {
        // Check for OAuth success/error messages from transient storage
        $message = get_transient( 'agents24x7_message' );
        if ( $message ) {
            // Delete transient after reading (single-use message)
            delete_transient( 'agents24x7_message' );
            
            if ( $message['type'] === 'success' ) {
                echo '<div class="notice notice-success is-dismissible"><p>' . 
                     esc_html( $message['text'] ) . 
                     '</p></div>';
            } elseif ( $message['type'] === 'error' ) {
                $error_message = $message['text'];
                $error_parts = explode( ' | Details: ', $error_message );
                $main_error = $error_parts[0];
                $details = isset( $error_parts[1] ) ? $error_parts[1] : '';
                
                echo '<div class="notice notice-error" style="border-left: 4px solid #dc3232; padding: 15px; margin: 20px 0; background: #fef7f7; position: relative; z-index: 999;">';
                echo '<h3 style="margin: 0 0 10px 0; color: #d54e21;">⚠️ OAuth Connection Failed</h3>';
                echo '<p style="margin: 0 0 10px 0; font-weight: bold; color: #333;">' . esc_html( $main_error ) . '</p>';
                
                if ( $details ) {
                    echo '<details style="margin-top: 10px;">';
                    echo '<summary style="cursor: pointer; font-weight: bold; color: #666;">Technical Details (click to expand)</summary>';
                    echo '<div style="margin-top: 10px; padding: 10px; background: #f9f9f9; border: 1px solid #ddd; font-family: monospace; font-size: 12px; white-space: pre-wrap;">';
                    echo esc_html( str_replace( ' | ', "\n", $details ) );
                    echo '</div>';
                    echo '</details>';
                }
                
                echo '<p style="margin: 10px 0 0 0;"><strong>Next Steps:</strong></p>';
                echo '<ul style="margin: 5px 0 0 20px;">';
                echo '<li>Check your site\'s error logs for more details</li>';
                echo '<li>Verify your Convex deployment is running correctly</li>';
                echo '<li>Try the connection again in a few minutes</li>';
                echo '<li>Contact support if the issue persists</li>';
                echo '</ul>';
                echo '</div>';
            }
        }
        ?>
        <div id="agents24x7-admin-root"></div>
        <?php
    }
    
    /**
     * AJAX handler for getting WordPress credentials
     */
    public function ajax_get_wp_credentials() {
        check_ajax_referer( 'agents24x7_nonce', 'nonce' );
        
        // Note: Detailed capability checking is now handled in get_wordpress_credentials()
        // which calls get_or_create_app_password() with full validation
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'You need administrative access to use this plugin.' );
            return;
        }
        
        $credentials = $this->integration->get_wordpress_credentials();
        
        if ( is_wp_error( $credentials ) ) {
            wp_send_json_error( $credentials->get_error_message() );
            return;
        }
        
        wp_send_json_success( $credentials );
    }
    
    /**
     * AJAX handler for getting WooCommerce credentials
     */
    public function ajax_get_woocommerce_credentials() {
        check_ajax_referer( 'agents24x7_nonce', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'You need administrative access to use this plugin.' );
            return;
        }
        
        $credentials = $this->integration->get_woocommerce_credentials();
        
        if ( is_wp_error( $credentials ) ) {
            wp_send_json_error( $credentials->get_error_message() );
            return;
        }
        
        wp_send_json_success( $credentials );
    }
    
    /**
     * AJAX handler for disconnecting site
     */
    public function ajax_disconnect() {
        check_ajax_referer( 'agents24x7_nonce', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
            return;
        }
        
        // First, call Convex API to disconnect from cloud
        $site_url = home_url();
        $convex_result = $this->api_proxy->disconnect_from_plugin( $site_url );
        
        if ( is_wp_error( $convex_result ) ) {
            wp_send_json_error( 'Failed to disconnect from Agents24x7: ' . $convex_result->get_error_message() );
            return;
        }
        
        // Then disconnect OAuth and clean up local data
        $this->auth->disconnect();
        
        // Clean up WordPress integration
        $result = $this->integration->disconnect_site();
        
        if ( is_wp_error( $result ) ) {
            // Log the error but don't fail the disconnect process
            Agents24x7_Logger::error( 'Failed to clean up WordPress integration: ' . $result->get_error_message() );
        }
        
        wp_send_json_success( array(
            'message' => 'Site disconnected successfully',
            'refresh' => true
        ) );
    }
    
    /**
     * AJAX handler for getting auth status
     */
    public function ajax_get_auth_status() {
        check_ajax_referer( 'agents24x7_nonce', 'nonce' );
        
        wp_send_json_success( array(
            'isAuthenticated' => $this->auth->is_authenticated(),
            'hasToken'        => ! is_null( $this->auth->get_token() ),
        ) );
    }
    
    /**
     * Get Convex cloud URL
     *
     * @return string
     */
    private function get_convex_cloud_url() {
        // Load constants from config file
        require_once AGENTS24X7_PLUGIN_DIR . 'config.php';
        return defined( 'AGENTS24X7_CONVEX_CLOUD_URL' ) 
            ? AGENTS24X7_CONVEX_CLOUD_URL 
            : '';
    }

    /**
     * Get Convex site URL
     *
     * @return string
     */
    private function get_convex_site_url() {
        // Load constants from config file
        require_once AGENTS24X7_PLUGIN_DIR . 'config.php';
        return defined( 'AGENTS24X7_CONVEX_SITE_URL' ) 
            ? AGENTS24X7_CONVEX_SITE_URL 
            : '';
    }
    
    /**
     * Get app URL
     *
     * @return string
     */
    private function get_app_url() {
        // Load constants from config file
        require_once AGENTS24X7_PLUGIN_DIR . 'config.php';
        return defined( 'AGENTS24X7_APP_URL' ) 
            ? AGENTS24X7_APP_URL 
            : '';
    }
}