<?php
/**
 * Logger Class for Agents24x7 Plugin
 *
 * Provides centralized logging with WordPress.org compliance
 * All logging is conditional on WP_DEBUG to prevent production logs
 *
 * @package Agents24x7
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Agents24x7_Logger
 *
 * Centralized logging system that respects WordPress debug settings
 */
class Agents24x7_Logger {
    
    /**
     * Log prefix for all messages
     */
    const PREFIX = 'AGENTS24X7';
    
    /**
     * Log a debug message
     * 
     * @param string $message Debug message to log
     */
    public static function debug( $message ) {
        self::log( 'DEBUG', $message );
    }
    
    /**
     * Log an info message
     * 
     * @param string $message Info message to log
     */
    public static function info( $message ) {
        self::log( 'INFO', $message );
    }
    
    /**
     * Log an error message
     * 
     * @param string $message Error message to log
     */
    public static function error( $message ) {
        self::log( 'ERROR', $message );
    }
    
    /**
     * Log a success message
     * 
     * @param string $message Success message to log
     */
    public static function success( $message ) {
        self::log( 'SUCCESS', $message );
    }
    
    /**
     * Internal logging method
     * 
     * Only logs when WP_DEBUG is enabled to ensure WordPress.org compliance
     * 
     * @param string $level   Log level (DEBUG, INFO, ERROR, SUCCESS)
     * @param string $message Message to log
     */
    private static function log( $level, $message ) {
        // Only log when WordPress debug mode is enabled
        if ( ! defined( 'WP_DEBUG' ) || ! WP_DEBUG ) {
            return;
        }
        
        // Format: AGENTS24X7 LEVEL - message
        $formatted_message = sprintf(
            '%s %s - %s',
            self::PREFIX,
            $level,
            $message
        );
        
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debug mode only
        error_log( $formatted_message );
    }
}