<?php

class Agents24x7_Integration {
    
    private $app_name = 'Agents24x7 Connector';
    
    /**
     * Check if current user has all required capabilities
     * 
     * @return array|true Returns true if all capabilities present, or array of missing capabilities
     */
    private function has_required_capabilities() {
        $required_caps = [
            'manage_options' => 'Administrative plugin access',
            'edit_posts' => 'Create and edit blog posts',
            'upload_files' => 'Upload images and media',
            'publish_posts' => 'Publish posts to website'
        ];
        
        $missing_caps = [];
        foreach ($required_caps as $cap => $description) {
            if (!current_user_can($cap)) {
                $missing_caps[$cap] = $description;
            }
        }
        
        return empty($missing_caps) ? true : $missing_caps;
    }
    
    /**
     * Generate detailed error message for missing capabilities
     * 
     * @param array $missing_caps Array of missing capabilities
     * @return string Formatted error message
     */
    private function format_capability_error($missing_caps) {
        $all_caps = [
            'edit_posts' => 'Create and edit blog posts',
            'upload_files' => 'Upload images and media',
            'publish_posts' => 'Publish posts to website'
        ];
        
        $message = esc_html__('Insufficient permissions. You need the following capabilities to use Agents24x7:', 'agents24x7') . "\n\n";
        
        foreach ($all_caps as $cap => $description) {
            $status = isset($missing_caps[$cap]) ? '✗ MISSING' : '✓';
            $message .= "• {$description} ({$cap}) - {$status}\n";
        }
        
        $message .= "\n" . esc_html__('Please contact your site administrator to grant the missing permissions.', 'agents24x7');
        
        return $message;
    }
    
    
    /**
     * Check if application password exists
     */
    public function has_app_password() {
        $user_id = get_current_user_id();
        if (!$user_id) {
            return false;
        }
        
        $existing_passwords = WP_Application_Passwords::get_user_application_passwords($user_id);
        foreach ($existing_passwords as $password) {
            if ($password['name'] === $this->app_name) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Create or get Application Password for current user
     */
    public function get_or_create_app_password() {
        $user_id = get_current_user_id();
        
        if (!$user_id) {
            return new WP_Error('no_user', 'No authenticated user');
        }
        
        // Check if user has all required capabilities
        $capability_check = $this->has_required_capabilities();
        if ($capability_check !== true) {
            $error_message = $this->format_capability_error($capability_check);
            return new WP_Error('insufficient_capabilities', $error_message);
        }
        
        // Check if WordPress supports Application Passwords
        if (!class_exists('WP_Application_Passwords')) {
            return new WP_Error('no_app_passwords', 'WordPress Application Passwords not available');
        }

        // Delete any existing application passwords with the same name to allow reconnection
        $existing_passwords = WP_Application_Passwords::get_user_application_passwords($user_id);
        foreach ($existing_passwords as $password) {
            if ($password['name'] === $this->app_name) {
                WP_Application_Passwords::delete_application_password($user_id, $password['uuid']);
            }
        }
        
        // Create new application password
        $app_password = WP_Application_Passwords::create_new_application_password($user_id, [
            'name' => $this->app_name,
        ]);
        
        if (is_wp_error($app_password)) {
            return $app_password;
        }
        
        // Return the plain text password (only available during creation)
        return [
            'password' => $app_password[0],
            'username' => wp_get_current_user()->user_login,
            'site_url' => home_url(),
        ];
    }
    
    /**
     * Check if WooCommerce is installed and active
     */
    public function check_for_woocommerce() {
        if (!defined('WC_VERSION')) {
            return false;
        } else {
            if (version_compare(WC_VERSION, '3.0', '>=')) {
                return true;
            }
            return false;
        }
    }
    
    /**
     * Get WooCommerce information
     */
    public function get_woocommerce_info() {
        if (!$this->check_for_woocommerce()) {
            return [
                'available' => false,
                'version' => null,
            ];
        }
        
        return [
            'available' => true,
            'version' => WC_VERSION,
            'store_currency' => get_woocommerce_currency(),
            'store_country' => WC()->countries->get_base_country(),
        ];
    }
    
    /**
     * Get WooCommerce credentials for Convex integration
     */
    public function get_woocommerce_credentials() {
        if (!$this->check_for_woocommerce()) {
            return new WP_Error('no_woocommerce', 'WooCommerce is not installed or active');
        }
        
        // Add check for WooCommerce REST API availability
        if (!class_exists('WC_REST_Authentication')) {
            return new WP_Error('wc_api_disabled', 'WooCommerce REST API is not available');
        }
        
        $app_password_data = $this->get_or_create_app_password();
        
        if (is_wp_error($app_password_data)) {
            return $app_password_data;
        }
        
        return [
            'baseUrl' => home_url(),
            'username' => $app_password_data['username'],
            'password' => $app_password_data['password'],
            'woocommerce' => [
                'version' => WC_VERSION,
                'currency' => get_woocommerce_currency(),
                'country' => WC()->countries->get_base_country(),
            ],
        ];
    }
    
    /**
     * Get WordPress credentials for Convex integration
     */
    public function get_wordpress_credentials() {
        $app_password_data = $this->get_or_create_app_password();
        
        if (is_wp_error($app_password_data)) {
            return $app_password_data;
        }
        
        return [
            'baseUrl' => home_url(),
            'username' => $app_password_data['username'],
            'password' => $app_password_data['password'],
        ];
    }
    
    
    /**
     * Disconnect entire site from Agents24x7 (site-level operation)
     * Removes ALL Application Passwords for this plugin across all users
     */
    public function disconnect_site() {
        if (!class_exists('WP_Application_Passwords')) {
            return true;
        }
        
        // Get all users and remove plugin's Application Passwords
        $users = get_users(['fields' => 'ID']);
        
        foreach ($users as $user_id) {
            $existing_passwords = WP_Application_Passwords::get_user_application_passwords($user_id);
            
            foreach ($existing_passwords as $password) {
                if ($password['name'] === $this->app_name) {
                    WP_Application_Passwords::delete_application_password($user_id, $password['uuid']);
                }
            }
        }
        
        return true;
    }
}