<?php
/**
 * OAuth Authentication Handler for Agents24x7 Plugin
 *
 * @package Agents24x7
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Agents24x7_Auth
 *
 * Handles OAuth 2.0 authentication flow
 */
class Agents24x7_Auth {
    /**
     * OAuth configuration
     */
    const CLIENT_ID = 'wordpress-plugin';
    
    /**
     * Get base app URL from configuration
     */
    private function get_app_url() {
        return defined('AGENTS24X7_APP_URL') ? AGENTS24X7_APP_URL : 'https://agents24x7.com';
    }
    
    /**
     * Get OAuth authorize URL
     */
    private function get_oauth_authorize_url() {
        return $this->get_app_url() . '/api/v1/oauth/authorize';
    }
    
    /**
     * Get OAuth token URL  
     */
    private function get_oauth_token_url() {
        return defined('AGENTS24X7_CONVEX_SITE_URL') 
            ? AGENTS24X7_CONVEX_SITE_URL . '/api/v1/oauth/token'
            : 'https://api.agents24x7.com/api/v1/oauth/token';
    }
    
    /**
     * Option names
     */
    const OPTION_API_TOKEN    = 'agents24x7_api_token';
    const OPTION_USER_DATA    = 'agents24x7_user_data';
    
    /**
     * Constructor
     */
    public function __construct() {
        // Load configuration file
        require_once AGENTS24X7_PLUGIN_DIR . 'config.php';
        add_action( 'admin_init', array( $this, 'handle_oauth_callback' ) );
    }
    
    /**
     * Get the OAuth authorization URL
     *
     * @return string
     */
    public function get_connect_url() {
        // Create a composite state parameter containing both OAuth state nonce and WordPress nonce
        // This satisfies WordPress plugin review requirements while maintaining OAuth security
        $state_data = array(
            'oauth_nonce' => wp_create_nonce( 'agents24x7_oauth' ),
            'wp_nonce'    => wp_create_nonce( 'agents24x7_callback' ),
        );
        $state = base64_encode( wp_json_encode( $state_data ) );
        
        $callback_url = admin_url( 'admin.php?page=agents24x7&action=oauth_callback' );
        
        $auth_url = $this->get_oauth_authorize_url();
        
        // Debug: Log the URLs being used
        Agents24x7_Logger::debug( 'APP_URL: ' . ( defined('AGENTS24X7_APP_URL') ? AGENTS24X7_APP_URL : 'NOT DEFINED' ) );
        Agents24x7_Logger::debug( 'Auth URL: ' . $auth_url );
        
        return $auth_url . '?' . http_build_query( array(
            'client_id'     => self::CLIENT_ID,
            'redirect_uri'  => $callback_url,
            'response_type' => 'code',
            'state'         => $state,
            'scope'         => 'wordpress:all',
        ) );
    }
    
    /**
     * Handle OAuth callback from Agents24x7.
     *
     * This method implements dual nonce verification:
     * 1. WordPress nonce check (satisfies WordPress plugin requirements)
     * 2. OAuth state nonce check (standard OAuth security)
     * Both are embedded in the state parameter to maintain OAuth compatibility.
     */
    public function handle_oauth_callback() {
        // Only proceed on our plugin's settings page and OAuth action.
        if (
            empty( $_GET['page'] ) || $_GET['page'] !== 'agents24x7' ||
            empty( $_GET['action'] ) || $_GET['action'] !== 'oauth_callback'
        ) {
            return;
        }

        // Only administrators can complete this OAuth flow.
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Unauthorized', 'agents24x7' ) );
        }

        // Sanitize external inputs - nonce verification happens below after decoding state
        $code  = isset( $_GET['code'] )  ? sanitize_text_field( wp_unslash( $_GET['code'] ) )  : '';
        $state = isset( $_GET['state'] ) ? sanitize_text_field( wp_unslash( $_GET['state'] ) ) : '';

        Agents24x7_Logger::debug( 'OAuth code=' . $code . ', state=' . $state );

        if ( ! $code || ! $state ) {
            Agents24x7_Logger::debug( 'Missing code or state' );
            $this->redirect_with_error( esc_html__( 'Invalid OAuth response', 'agents24x7' ) );
            return;
        }

        // Decode the composite state parameter
        $state_json = base64_decode( $state );
        if ( $state_json === false ) {
            Agents24x7_Logger::debug( 'Failed to decode state parameter' );
            $this->redirect_with_error( esc_html__( 'Invalid state parameter', 'agents24x7' ) );
            return;
        }

        $state_data = json_decode( $state_json, true );
        if ( ! is_array( $state_data ) || ! isset( $state_data['oauth_nonce'] ) || ! isset( $state_data['wp_nonce'] ) ) {
            Agents24x7_Logger::debug( 'Invalid state parameter structure' );
            $this->redirect_with_error( esc_html__( 'Invalid state parameter', 'agents24x7' ) );
            return;
        }

        // Verify WordPress nonce (for plugin review compliance)
        if ( ! wp_verify_nonce( $state_data['wp_nonce'], 'agents24x7_callback' ) ) {
            Agents24x7_Logger::debug( 'Invalid WordPress nonce' );
            $this->redirect_with_error( esc_html__( 'Security check failed', 'agents24x7' ) );
            return;
        }

        // Verify OAuth state nonce (standard OAuth security)
        if ( ! wp_verify_nonce( $state_data['oauth_nonce'], 'agents24x7_oauth' ) ) {
            Agents24x7_Logger::debug( 'Invalid OAuth state nonce' );
            $this->redirect_with_error( esc_html__( 'Invalid state parameter', 'agents24x7' ) );
            return;
        }

        // All checks passed—exchange the code for an access token.
        Agents24x7_Logger::debug( 'Exchanging OAuth code for token' );
        $this->exchange_code_for_token( $code );
    }
    
    /**
     * Exchange authorization code for access token
     *
     * @param string $code Authorization code
     */
    private function exchange_code_for_token( $code ) {
        $callback_url = admin_url( 'admin.php?page=agents24x7&action=oauth_callback' );
        $token_url = $this->get_oauth_token_url();
        
        // Debug: Log the token exchange attempt
        Agents24x7_Logger::debug( 'Token URL: ' . $token_url );
        Agents24x7_Logger::debug( 'Callback URL: ' . $callback_url );
        Agents24x7_Logger::debug( 'Code: ' . $code );
        
        $response = wp_remote_post( $token_url, array(
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
            ),
            'body' => array(
                'grant_type'    => 'authorization_code',
                'code'          => $code,
                'redirect_uri'  => $callback_url,
                'client_id'     => self::CLIENT_ID,
            ),
            'timeout' => 30,
        ) );
        
        // Always log response details for debugging (not just WP_DEBUG)
        $response_code = 0;
        $response_body = '';
        $error_details = array();
        
        if ( is_wp_error( $response ) ) {
            $error_message = 'Network Error: ' . $response->get_error_message();
            $error_details[] = 'Token URL: ' . $token_url;
            $error_details[] = 'WP Error Code: ' . $response->get_error_code();
            
            Agents24x7_Logger::error( $error_message );
            foreach ( $error_details as $detail ) {
                Agents24x7_Logger::error( $detail );
            }
            
            $this->redirect_with_error( $error_message . ' | Details: ' . implode( ' | ', $error_details ) );
            return;
        }
        
        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        
        // Always log response for debugging
        Agents24x7_Logger::info( 'Token exchange response code: ' . $response_code );
        Agents24x7_Logger::info( 'Token exchange response body: ' . $response_body );
        
        // Check for HTTP error status codes
        if ( $response_code >= 400 ) {
            $error_details[] = 'HTTP Status: ' . $response_code;
            $error_details[] = 'Token URL: ' . $token_url;
            $error_details[] = 'Response: ' . substr( $response_body, 0, 200 );
            
            $error_message = 'HTTP Error ' . $response_code . ' from OAuth server';
            
            Agents24x7_Logger::error( $error_message );
            foreach ( $error_details as $detail ) {
                Agents24x7_Logger::error( $detail );
            }
            
            $this->redirect_with_error( $error_message . ' | Details: ' . implode( ' | ', $error_details ) );
            return;
        }
        
        $data = json_decode( $response_body, true );
        
        if ( ! $data ) {
            $error_message = 'Invalid JSON response from OAuth server';
            $error_details[] = 'Response Code: ' . $response_code;
            $error_details[] = 'Raw Response: ' . substr( $response_body, 0, 200 );
            
            Agents24x7_Logger::error( $error_message );
            $this->redirect_with_error( $error_message . ' | Details: ' . implode( ' | ', $error_details ) );
            return;
        }
        
        if ( isset( $data['error'] ) ) {
            $error_message = isset( $data['error_description'] ) 
                ? $data['error_description'] 
                : ( 'OAuth Error: ' . $data['error'] );
            
            $error_details[] = 'Error Code: ' . $data['error'];
            $error_details[] = 'Response Code: ' . $response_code;
            if ( isset( $data['error_description'] ) ) {
                $error_details[] = 'Description: ' . $data['error_description'];
            }
            
            Agents24x7_Logger::error( 'OAuth server returned error: ' . $error_message );
            foreach ( $error_details as $detail ) {
                Agents24x7_Logger::error( $detail );
            }
            
            $this->redirect_with_error( $error_message . ' | Details: ' . implode( ' | ', $error_details ) );
            return;
        }
        
        if ( ! isset( $data['access_token'] ) ) {
            $error_message = 'No access token received from OAuth server';
            $error_details[] = 'Response Code: ' . $response_code;
            $error_details[] = 'Available Fields: ' . implode( ', ', array_keys( $data ) );
            $error_details[] = 'Response Data: ' . substr( json_encode( $data ), 0, 200 );
            
            Agents24x7_Logger::error( $error_message );
            foreach ( $error_details as $detail ) {
                Agents24x7_Logger::error( $detail );
            }
            
            $this->redirect_with_error( $error_message . ' | Details: ' . implode( ' | ', $error_details ) );
            return;
        }
        
        // Log successful token exchange
        Agents24x7_Logger::success( 'Successfully received access token from OAuth server' );
        
        // Store the token
        $this->store_token( $data );
        
        // Redirect with success
        $this->redirect_with_success();
    }
    
    /**
     * Store the API token
     *
     * @param array $token_data Token response data
     */
    private function store_token( $token_data ) {
        update_option( self::OPTION_API_TOKEN, array(
            'token'      => $token_data['access_token'],
            'type'       => $token_data['token_type'],
            'scope'      => isset( $token_data['scope'] ) ? $token_data['scope'] : 'wordpress:all',
            'created_at' => time(),
        ) );
    }
    
    /**
     * Get the stored API token
     *
     * @return string|null
     */
    public function get_token() {
        $token_data = get_option( self::OPTION_API_TOKEN );
        
        if ( ! $token_data || ! isset( $token_data['token'] ) ) {
            return null;
        }
        
        return $token_data['token'];
    }
    
    /**
     * Check if authenticated
     *
     * @return bool
     */
    public function is_authenticated() {
        return ! is_null( $this->get_token() );
    }
    
    /**
     * Disconnect (remove token)
     */
    public function disconnect() {
        delete_option( self::OPTION_API_TOKEN );
        delete_option( self::OPTION_USER_DATA );
    }
    
    /**
     * Redirect with error message
     *
     * @param string $message Error message
     */
    private function redirect_with_error( $message ) {
        // Store error message in transient (expires in 30 seconds)
        set_transient( 'agents24x7_message', array(
            'type' => 'error',
            'text' => $message,
        ), 30 );
        
        wp_redirect( admin_url( 'admin.php?page=agents24x7' ) );
        exit;
    }
    
    /**
     * Redirect with success message
     */
    private function redirect_with_success() {
        // Store success message in transient (expires in 30 seconds)
        set_transient( 'agents24x7_message', array(
            'type' => 'success',
            'text' => 'Successfully connected to Agents24x7!',
        ), 30 );
        
        wp_redirect( admin_url( 'admin.php?page=agents24x7' ) );
        exit;
    }
    
    /**
     * Get authorization header for API requests
     *
     * @return array
     */
    public function get_auth_headers() {
        $token = $this->get_token();
        
        if ( ! $token ) {
            return array();
        }
        
        return array(
            'Authorization' => 'Bearer ' . $token,
        );
    }
}