<?php
/**
 * API Proxy for Agents24x7 Plugin
 *
 * @package Agents24x7
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Agents24x7_API_Proxy
 *
 * Proxies API requests from React frontend to Convex backend
 */
class Agents24x7_API_Proxy {
    /**
     * Auth handler
     *
     * @var Agents24x7_Auth
     */
    private $auth;
    
    /**
     * Constructor
     *
     * @param Agents24x7_Auth $auth Auth handler
     */
    public function __construct( $auth ) {
        $this->auth = $auth;
        add_action( 'rest_api_init', array( $this, 'register_routes' ) );
    }
    
    /**
     * Register REST API routes
     */
    public function register_routes() {
        register_rest_route( 'agents24x7/v1', '/query', array(
            'methods'             => 'POST',
            'callback'            => array( $this, 'proxy_query' ),
            'permission_callback' => array( $this, 'check_permissions' ),
        ) );
        
        register_rest_route( 'agents24x7/v1', '/mutation', array(
            'methods'             => 'POST',
            'callback'            => array( $this, 'proxy_mutation' ),
            'permission_callback' => array( $this, 'check_permissions' ),
        ) );
        
        register_rest_route( 'agents24x7/v1', '/action', array(
            'methods'             => 'POST',
            'callback'            => array( $this, 'proxy_action' ),
            'permission_callback' => array( $this, 'check_permissions' ),
        ) );
    }
    
    /**
     * Check permissions
     *
     * @return bool
     */
    public function check_permissions() {
        return current_user_can( 'edit_posts' );
    }
    
    /**
     * Proxy query request
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function proxy_query( $request ) {
        return $this->proxy_convex_request( 'query', $request );
    }
    
    /**
     * Proxy mutation request
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function proxy_mutation( $request ) {
        return $this->proxy_convex_request( 'mutation', $request );
    }
    
    /**
     * Proxy action request
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function proxy_action( $request ) {
        return $this->proxy_convex_request( 'action', $request );
    }
    
    /**
     * Proxy request to Convex
     *
     * @param string          $type    Request type (query, mutation, action)
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    private function proxy_convex_request( $type, $request ) {
        try {
            // Check if we have an API token
            if ( ! $this->auth->is_authenticated() ) {
                return new WP_Error( 
                    'not_authenticated', 
                    esc_html__( 'Plugin not connected to Agents24x7', 'agents24x7' ),
                    array( 'status' => 401 )
                );
            }
            
            $params = $request->get_json_params();
            
            if ( ! isset( $params['function'] ) || ! isset( $params['args'] ) ) {
                return new WP_Error(
                    'invalid_request',
                    esc_html__( 'Missing required parameters: function, args', 'agents24x7' ),
                    array( 'status' => 400 )
                );
            }
        
        // Get the Convex URL (use SITE_URL for HTTP endpoints, not CLOUD_URL)
        $convex_url = defined( 'AGENTS24X7_CONVEX_SITE_URL' ) 
            ? AGENTS24X7_CONVEX_SITE_URL 
            : 'https://precious-loris-550.convex.site';
            
        $api_url = trailingslashit( $convex_url ) . 'wordpress/api';
        
        // Prepare headers
        $headers = array_merge(
            $this->auth->get_auth_headers(),
            array(
                'Content-Type'     => 'application/json',
                'X-WordPress-Site' => home_url(),
                'X-WordPress-User' => get_current_user_id(),
            )
        );
        
        // Make the request
        $response = wp_remote_post( $api_url, array(
            'headers' => $headers,
            'body'    => wp_json_encode( array(
                'type'         => $type,
                'functionName' => $params['function'],
                'args'         => $params['args'],
            ) ),
            'timeout' => 30,
        ) );
        
        if ( is_wp_error( $response ) ) {
            return new WP_Error(
                'api_error',
                $response->get_error_message(),
                array( 'status' => 500 )
            );
        }
        
        $status_code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        
        // Log API usage for debugging
        $this->log_api_usage( $type, $params['function'], $status_code );
        
        // Handle 401 authentication errors specifically
        if ( $status_code === 401 ) {
            // Clear the invalid/expired token
            $this->auth->disconnect();
            
            return new WP_Error(
                'authentication_expired',
                esc_html__( 'Your connection to Agents24x7 has expired. Please reconnect.', 'agents24x7' ),
                array( 
                    'status' => 401,
                    'requires_reauthentication' => true 
                )
            );
        }
        
        if ( $status_code >= 400 ) {
            $error_message = isset( $data['error'] ) 
                ? $data['error'] 
                : esc_html__( 'API request failed', 'agents24x7' );
                
            return new WP_Error(
                'api_error',
                $error_message,
                array( 'status' => $status_code )
            );
        }
        
        return new WP_REST_Response( $data, $status_code );
        
        } catch ( Exception $e ) {
            return new WP_Error(
                'server_error',
                'Internal server error: ' . $e->getMessage(),
                array( 'status' => 500 )
            );
        }
    }
    
    /**
     * Call disconnectFromPlugin action
     *
     * @param string $site_url Site URL to disconnect
     * @return array|WP_Error API response or error
     */
    public function disconnect_from_plugin( $site_url ) {
        try {
            // Check if we have an API token
            if ( ! $this->auth->is_authenticated() ) {
                return new WP_Error( 
                    'not_authenticated', 
                    esc_html__( 'Plugin not connected to Agents24x7', 'agents24x7' )
                );
            }
            
            // Get the Convex URL (use SITE_URL for HTTP endpoints, not CLOUD_URL)
            $convex_url = defined( 'AGENTS24X7_CONVEX_SITE_URL' ) 
                ? AGENTS24X7_CONVEX_SITE_URL 
                : 'https://precious-loris-550.convex.site';
                
            $api_url = trailingslashit( $convex_url ) . 'wordpress/api';
            
            // Prepare headers
            $headers = array_merge(
                $this->auth->get_auth_headers(),
                array(
                    'Content-Type'     => 'application/json',
                    'X-WordPress-Site' => home_url(),
                    'X-WordPress-User' => get_current_user_id(),
                )
            );
            
            // Make the request
            $response = wp_remote_post( $api_url, array(
                'headers' => $headers,
                'body'    => wp_json_encode( array(
                    'type'         => 'action',
                    'functionName' => 'wordpress/plugin:disconnectFromPlugin',
                    'args'         => array( 'siteUrl' => $site_url ),
                ) ),
                'timeout' => 30,
            ) );
            
            if ( is_wp_error( $response ) ) {
                return new WP_Error(
                    'api_error',
                    $response->get_error_message()
                );
            }
            
            $status_code = wp_remote_retrieve_response_code( $response );
            $body = wp_remote_retrieve_body( $response );
            $data = json_decode( $body, true );
            
            // Log API usage for debugging
            $this->log_api_usage( 'action', 'wordpress/plugin:disconnectFromPlugin', $status_code );
            
            // Handle 401 authentication errors specifically
            if ( $status_code === 401 ) {
                // Token is already invalid, just return specific error
                return new WP_Error(
                    'authentication_expired',
                    esc_html__( 'Your connection to Agents24x7 has expired. Please reconnect.', 'agents24x7' )
                );
            }
            
            if ( $status_code >= 400 ) {
                $error_message = isset( $data['error'] ) 
                    ? $data['error'] 
                    : esc_html__( 'API request failed', 'agents24x7' );
                    
                return new WP_Error(
                    'api_error',
                    $error_message
                );
            }
            
            return $data;
            
        } catch ( Exception $e ) {
            return new WP_Error(
                'server_error',
                'Internal server error: ' . $e->getMessage()
            );
        }
    }

    /**
     * Log API usage
     *
     * @param string $type         Request type
     * @param string $function     Function name
     * @param int    $status_code  Response status code
     */
    private function log_api_usage( $type, $function, $status_code ) {
        Agents24x7_Logger::info( sprintf(
            '[Agents24x7 API] %s %s - Status: %d',
            strtoupper( $type ),
            $function,
            $status_code
        ) );
    }
}