<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

affl_import(AFFILILABS_SERVICE_PRODUCT_FILE);
affl_import(AFFILILABS_CPT_PRODUCT_FILE);
affl_import(AFFILILABS_SERVICE_CRITERIA_FILE);

function affililabs_product_fields_shortcode_assets() {
    wp_enqueue_style(
        'affl-product-fields',
        AFFILILABS_SHORTCODE_PRODUCT_FIELDS_CSS_URL,
        array(),
        AFFILILABS_VERSION
    );
}

/**
 * Helper function to retrieve product data for shortcodes.
 * It's context-aware: uses the current product ID on a product page,
 * or a specified ID from attributes.
 *
 * @param array $atts Shortcode attributes.
 * @param string $shortcode_name The name of the shortcode for error messages.
 * @return array|string Product data on success, error message string on failure.
 */
function _affl_get_product_data_for_shortcode(array $atts, string $shortcode_name) {
    $product_id = 0;

    // Case 1: ID is provided directly in the shortcode, e.g., [affl_price id="123"]
    if (!empty($atts['id'])) {
        $product_id = intval($atts['id']);
    } 
    // Case 2: No ID provided, try to get it from the current post if it's a product page
    else {
        $current_post_id = get_the_ID();
        if ($current_post_id && get_post_type($current_post_id) === AFFILILABS_CPT_PRODUCT) {
            $product_id = $current_post_id;
        }
    }

    if (!$product_id) {
        /* translators: %s: shortcode name */
        return sprintf(__('Error: Product ID not found. Please provide an ID attribute (e.g., [%s id="123"]) or use the shortcode on a product page.', 'affililabs'), $shortcode_name);
    }

    $product_data = AFFLProductService::get_product_by_id($product_id);

    if (!$product_data) {
        /* translators: %d: product ID */
        return sprintf(__('Error: Product not found for ID %d.', 'affililabs'), $product_id);
    }

    return $product_data;
}


/**
 * [affl_name id="123"]
 */
function affl_name_shortcode($atts) {
    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_name');
    if (is_string($product_data)) {
        return $product_data;
    }
    return esc_html($product_data['name']);
}
add_shortcode('affl_name', 'affl_name_shortcode');

/**
 * [affl_price id="123"]
 */
function affl_price_shortcode($atts) {
    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_price');
    if (is_string($product_data)) {
        return $product_data;
    }
    return esc_html($product_data['price_currency'] . number_format_i18n($product_data['price'], 2));
}
add_shortcode('affl_price', 'affl_price_shortcode');

/**
 * [affl_rating id="123"]
 */
function affl_rating_shortcode($atts) {
    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_rating');
    if (is_string($product_data)) {
        return $product_data;
    }
    if (empty($product_data['rating'])) {
        return '';
    }
    return AFFLProductService::get_rating_stars_html(floatval($product_data['rating']));
}
add_shortcode('affl_rating', 'affl_rating_shortcode');

/**
 * [affl_description id="123"]
 */
function affl_description_shortcode($atts) {
    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_description');
    if (is_string($product_data)) {
        return $product_data;
    }
    return wp_kses_post($product_data['description']);
}
add_shortcode('affl_description', 'affl_description_shortcode');

/**
 * [affl_features id="123"]
 */
function affl_features_shortcode($atts) {
    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_features');
    if (is_string($product_data)) {
        return $product_data;
    }
    if (empty($product_data['short_features'])) {
        return '';
    }
    $items = AFFLProductService::get_list_items_as_array($product_data['short_features']);
    $output = '<ul class="affililabs-product-list affl-product-list--features">';
    foreach ($items as $item) {
        if (!empty($item)) {
            $output .= '<li>' . esc_html($item) . '</li>';
        }
    }
    $output .= '</ul>';
    return $output;
}
add_shortcode('affl_features', 'affl_features_shortcode');

/**
 * [affl_pros id="123"]
 */
function affl_pros_shortcode($atts) {
    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_pros');
    if (is_string($product_data)) {
        return $product_data;
    }
    if (empty($product_data['pros'])) {
        return '';
    }
    $items = AFFLProductService::get_list_items_as_array($product_data['pros']);
    $output = '<ul class="affililabs-product-list affl-product-list--pros">';
    foreach ($items as $item) {
        if (!empty($item)) {
            $output .= '<li><img src="' . esc_url(AFFILILABS_ASSETS_CHECK_FILLED_SVG) . '" alt="Pro" class="affl-list-icon"> ' . esc_html($item) . '</li>';
        }
    }
    $output .= '</ul>';
    return $output;
}
add_shortcode('affl_pros', 'affl_pros_shortcode');

/**
 * [affl_cons id="123"]
 */
function affl_cons_shortcode($atts) {
    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_cons');
    if (is_string($product_data)) {
        return $product_data;
    }
    if (empty($product_data['cons'])) {
        return '';
    }
    $items = AFFLProductService::get_list_items_as_array($product_data['cons']);
    $output = '<ul class="affililabs-product-list affl-product-list--cons">';
    foreach ($items as $item) {
        if (!empty($item)) {
            $output .= '<li><img src="' . esc_url(AFFILILABS_ASSETS_CLOSE_SQUARE_FILLED_RED_SVG) . '" alt="Con" class="affl-list-icon"> ' . esc_html($item) . '</li>';
        }
    }
    $output .= '</ul>';
    return $output;
}
add_shortcode('affl_cons', 'affl_cons_shortcode');

/**
 * [affl_category id="123"]
 */
function affl_category_shortcode($atts) {
    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_category');
    if (is_string($product_data)) {
        return $product_data;
    }
    if (empty($product_data['category_names'])) {
        return '';
    }
    return esc_html(implode(', ', $product_data['category_names']));
}
add_shortcode('affl_category', 'affl_category_shortcode');

/**
 * [affl_marketplace id="123"]
 */
function affl_marketplace_shortcode($atts) {
    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_marketplace');
    if (is_string($product_data)) {
        return $product_data;
    }
    return esc_html($product_data['marketplace_name']);
}
add_shortcode('affl_marketplace', 'affl_marketplace_shortcode');

/**
 * [affl_image id="123" size="medium"]
 */
function affl_image_shortcode($atts) {
    $atts = shortcode_atts(array('id' => 0, 'size' => 'medium'), $atts);
    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_image');
    if (is_string($product_data)) {
        return $product_data;
    }
    $image_size = sanitize_text_field($atts['size']);
    $image_url = get_the_post_thumbnail_url($product_data['id'], $image_size) ?: AFFILILABS_DEFAULT_PRODUCT_IMAGE;
    return '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($product_data['name']) . '">';
}
add_shortcode('affl_image', 'affl_image_shortcode');

/**
 * [affl_button id="123" text="Buy Now" color="#0073aa" radius="8"]
 */
function affl_button_shortcode($atts) {
    $default_atts = array(
        'id' => 0,
        'text' => __('Check Product*', 'affililabs'),
        'color' => '#0073aa',
        'radius' => '8',
        'text_color' => '#ffffff',
    );
    $atts = shortcode_atts($default_atts, $atts);

    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_button');
    if (is_string($product_data)) {
        return $product_data;
    }

    if (empty($product_data['affiliate_link'])) {
        return '';
    }

    $button_text = sanitize_text_field($atts['text']);
    $button_color = sanitize_hex_color($atts['color']) ?: '#0073aa';
    $border_radius = intval($atts['radius']);
    $text_color = sanitize_hex_color($atts['text_color']) ?: '#ffffff';

    return '<a href="' . esc_url($product_data['affiliate_link']) . '"
               class="affililabs-product-fields__button"
               target="_blank"
               rel="nofollow"
               style="background-color: ' . esc_attr($button_color). '; border-radius: ' . esc_attr($border_radius) . 'px; color: ' . esc_attr($text_color) . ';">'
           . esc_html($button_text) .
           '</a>';
}
add_shortcode('affl_button', 'affl_button_shortcode');

/**
 * [affl_criteria id="123"]
 * Outputs all criteria and their fields for the current product.
 */
function affl_criteria_shortcode($atts) {
    $default_atts = array(
        'id' => 0,
    );
    $atts = shortcode_atts($default_atts, $atts);

    $product_data = _affl_get_product_data_for_shortcode($atts, 'affl_criteria');
    if (is_string($product_data)) {
        return $product_data;
    }

    $product_id = $product_data['id'];
    $all_criterion_values = get_post_meta($product_id, AFFILILABS_META_KEY_PRODUCT_CRITERION_VALUES, true);

    if (!is_array($all_criterion_values)) {
        $all_criterion_values = [];
    }

    return AFFLCriteriaService::render_product_criteria_html($all_criterion_values);
}
add_shortcode('affl_criteria', 'affl_criteria_shortcode');