<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

affl_import(AFFILILABS_SERVICE_PRODUCT_FILE);

function affililabs_product_box_shortcode_assets() {
    wp_enqueue_style(
        'affl-product-box',
        AFFILILABS_SHORTCODE_PRODUCT_BOX_CSS_URL,
        array(),
        AFFILILABS_VERSION
    );
}

/**
 * Shortcode for displaying a product box
 * 
 * Usage: [affl_product_box id="123"]
 * 
 * Special ID values:
 * - id="random" or id="0": Display a random product
 * - id="category": Display a random product from the same category as the current post
 * - No id provided: If current post is not a product, display a random product
 */
function affililabs_product_box_shortcode($atts) {
    $default_atts = array(
        'id' => 0,
        'image_size' => 'medium',
        'template' => 'default',
        'show_button' => 'true',
        'button_text' => __('Check Product*', 'affililabs'),
        'button_color' => '#0073aa',
        'show_rating' => 'true',
        'show_features' => 'true',
        'show_pros_cons' => 'true',
        'show_description' => 'true',
        'show_category' => 'true',
        'show_marketplace' => 'true',
        'show_badge' => 'true',
        'badge_text' => __('Top Pick', 'affililabs'),
        'badge_background_color' => '#ffc107',
        'badge_text_color' => '#000000',
        'border_radius' => '8',
        'enable_shadow' => 'true',
    );

    
    if (isset($atts['template']) && $atts['template'] === 'compact') {
      $default_atts['show_features'] = isset($atts['show_features']) ? $atts['show_features'] : 'false';
      $default_atts['show_pros_cons'] = isset($atts['show_pros_cons']) ? $atts['show_pros_cons'] : 'false';
      $default_atts['show_description'] = isset($atts['show_description']) ? $atts['show_description'] : 'false';
      $default_atts['show_category'] = isset($atts['show_category']) ? $atts['show_category'] : 'false';
      $default_atts['show_marketplace'] = isset($atts['show_marketplace']) ? $atts['show_marketplace'] : 'false';
      $default_atts['show_badge'] = isset($atts['show_badge']) ? $atts['show_badge'] : 'false';
      $default_atts['enable_shadow'] = isset($atts['enable_shadow']) ? $atts['enable_shadow'] : 'false';
    }
    
    $atts = shortcode_atts($default_atts, $atts);
    
    // Handle random product selection
    $product_data = null;
    $product_id = 0;
    
    if (isset($atts['id']) && (strtolower($atts['id']) === 'random' || $atts['id'] === '0')) {
        // Get a random product
        $product_data = AFFLProductService::get_random_product('publish');
        if ($product_data) {
            $product_id = $product_data['id'];
        }
    } elseif (isset($atts['id']) && strtolower($atts['id']) === 'category') {
        // Get a random product from the same category as the current post
        $current_post_id = get_the_ID();
        $current_post = get_post($current_post_id);
        
        if ($current_post && $current_post->post_type === AFFILILABS_CPT_PRODUCT) {
            $categories = wp_get_post_terms($current_post_id, AFFILILABS_TAXONOMY_PRODUCT_CATEGORY);
            
            if (!is_wp_error($categories) && !empty($categories)) {
                $category_id = $categories[0]->term_id;
                $product_data = AFFLProductService::get_random_product_by_category($category_id, 'publish');
                if ($product_data) {
                    $product_id = $product_data['id'];
                }
            }
        }
        
        // Fall back to random
        if (!$product_data) {
            $product_data = AFFLProductService::get_random_product('publish');
            if ($product_data) {
                $product_id = $product_data['id'];
            }
        }
    } else {
        // Standard behavior: use the provided ID or try to get from current post
        $product_id = intval($atts['id']);
        
        // If no ID provided and current post is not a product, get a random product
        if (!$product_id) {
            $current_post_id = get_the_ID();
            $current_post = get_post($current_post_id);
            
            if (!$current_post || $current_post->post_type !== AFFILILABS_CPT_PRODUCT) {
                $product_data = AFFLProductService::get_random_product('publish');
                if ($product_data) {
                    $product_id = $product_data['id'];
                }
            } else {
                $product_id = $current_post_id;
            }
        }
    }
    
    // If we still don't have a product ID, return error
    if (!$product_id) {
        return __('Error: No product available', 'affililabs');
    }
    
    // The assets are enqueued only when the shortcode is used
    affililabs_product_box_shortcode_assets();

    // Get product data if we haven't already
    if (!$product_data) {
        $product_data = AFFLProductService::get_product_by_id($product_id);
    }

    if (!$product_data || get_post_status($product_id) !== 'publish') {
        return __('Error: Product not found or not published', 'affililabs');
    }

    $image_size = sanitize_text_field($atts['image_size']);
    $show_button = filter_var($atts['show_button'], FILTER_VALIDATE_BOOLEAN);
    $button_text = sanitize_text_field($atts['button_text']);
    $button_color = sanitize_hex_color($atts['button_color']) ?: '#0073aa';
    $show_rating = filter_var($atts['show_rating'], FILTER_VALIDATE_BOOLEAN);
    $show_features = filter_var($atts['show_features'], FILTER_VALIDATE_BOOLEAN);
    $show_pros_cons = filter_var($atts['show_pros_cons'], FILTER_VALIDATE_BOOLEAN);
    $show_description = filter_var($atts['show_description'], FILTER_VALIDATE_BOOLEAN);
    $show_category = filter_var($atts['show_category'], FILTER_VALIDATE_BOOLEAN);
    $show_marketplace = filter_var($atts['show_marketplace'], FILTER_VALIDATE_BOOLEAN);
    $show_badge = filter_var($atts['show_badge'], FILTER_VALIDATE_BOOLEAN);
    $badge_text = sanitize_text_field($atts['badge_text']);
    $badge_background_color = sanitize_hex_color($atts['badge_background_color']) ?: '#ffc107';
    $badge_text_color = sanitize_hex_color($atts['badge_text_color']) ?: '#000000';
    $border_radius = intval($atts['border_radius']);
    $enable_shadow = filter_var($atts['enable_shadow'], FILTER_VALIDATE_BOOLEAN);
    $image_url = get_the_post_thumbnail_url($product_id, $image_size) ?: AFFILILABS_DEFAULT_PRODUCT_IMAGE;

    $box_classes = 'affililabs-product-box';
    if ($enable_shadow) {
        $box_classes .= ' affililabs-product-box--shadow';
    }
    $box_style = 'border-radius: ' . $border_radius . 'px;';
    $badge_style = 'background-color:' . $badge_background_color . '; color:' . $badge_text_color . ';';

    ob_start();
    ?>
    <div class="<?php affl_print(esc_attr($box_classes)); ?>" style="<?php affl_print(esc_attr($box_style)); ?>">
        <?php if ($show_badge && !empty($badge_text)): ?>
            <div class="affililabs-product-box__badge" style="<?php affl_print(esc_attr($badge_style)); ?>">
                <?php affl_print(esc_html($badge_text)); ?>
            </div>
        <?php endif; ?>

        <div class="affililabs-product-box__image">
            <img src="<?php affl_print(esc_url($image_url)); ?>" alt="<?php affl_print(esc_attr($product_data['name'])); ?>">
        </div>

        <div class="affililabs-product-box__content">
            <h3 class="affililabs-product-box__title"><?php affl_print(esc_html($product_data['name'])); ?></h3>
            
            <?php if ($show_category && !empty($product_data['category_names'])): ?>
                <div class="affililabs-product-box__category"><?php affl_print(esc_html(implode(', ', $product_data['category_names']))); ?></div>
            <?php endif; ?>

            <?php if ($show_marketplace && !empty($product_data['marketplace_name'])): ?>
                <div class="affililabs-product-box__marketplace"><?php affl_print(esc_html($product_data['marketplace_name'])); ?></div>
            <?php endif; ?>

            <?php if ($show_rating && !empty($product_data['rating'])): ?>
                <div class="affililabs-product-box__rating">
                    <?php affl_print(AFFLProductService::get_rating_stars_html(floatval($product_data['rating']))); ?>
                </div>
            <?php endif; ?>

            <?php if ($show_description && !empty($product_data['description'])): ?>
                <div class="affililabs-product-box__description"><?php affl_print(wp_kses_post($product_data['description'])); ?></div>
            <?php endif; ?>

            <?php if ($show_features && !empty($product_data['short_features'])): ?>
                <div class="affililabs-product-box__features">
                    <ul class="affililabs-product-list affl-product-list--features">
                        <?php
                        affl_import(AFFILILABS_SERVICE_PRODUCT_FILE);
                        $feature_items = AFFLProductService::get_list_items_as_array( $product_data['short_features'] );
                        foreach ( $feature_items as $item ) {
                            if ( ! empty( $item ) ) {
                                affl_print( '<li>' . esc_html( $item ) . '</li>' );
                            }
                        }
                        ?>
                    </ul>
                </div>
            <?php endif; ?>

            <?php if ($show_pros_cons): ?>
                <?php if (!empty($product_data['pros'])): ?>
                    <div class="affililabs-product-box__pros">
                        <h4><?php affl_print(esc_html__('Pros', 'affililabs')); ?></h4>
                        <ul class="affililabs-product-list affl-product-list--pros">
                            <?php
                            $pro_items = AFFLProductService::get_list_items_as_array( $product_data['pros'] );
                            foreach ( $pro_items as $item ) {
                                if ( ! empty( $item ) ) {
                                    affl_print( '<li><img src="' . esc_url( AFFILILABS_ASSETS_CHECK_FILLED_SVG ) . '" alt="Pro" class="affl-list-icon"> ' . esc_html( $item ) . '</li>' );
                                }
                            }
                            ?>
                        </ul>
                    </div>
                <?php endif; ?>
                <?php if (!empty($product_data['cons'])): ?>
                    <div class="affililabs-product-box__cons">
                        <h4><?php affl_print(esc_html__('Cons', 'affililabs')); ?></h4>
                        <ul class="affililabs-product-list affl-product-list--cons">
                            <?php
                            $con_items = AFFLProductService::get_list_items_as_array( $product_data['cons'] );
                            foreach ( $con_items as $item ) {
                                if ( ! empty( $item ) ) {
                                    affl_print( '<li><img src="' . esc_url( AFFILILABS_ASSETS_CLOSE_SQUARE_FILLED_RED_SVG ) . '" alt="Con" class="affl-list-icon"> ' . esc_html( $item ) . '</li>' );
                                }
                            }
                            ?>
                        </ul>
                    </div>
                <?php endif; ?>
            <?php endif; ?>

            <p class="affililabs-product-box__price"><?php affl_print(esc_html($product_data['price_currency'] . number_format_i18n($product_data['price'], 2))); ?></p>

            <?php if ($show_button && !empty($product_data['affiliate_link'])): ?>
                <a href="<?php affl_print(esc_url($product_data['affiliate_link'])); ?>"
                   class="affililabs-product-box__button" 
                   target="_blank" 
                   rel="nofollow"
                   style="background-color: <?php affl_print(esc_attr($button_color)); ?>; border-radius: <?php affl_print(esc_attr($border_radius)); ?>px;">
                    <?php affl_print(esc_html($button_text)); ?>
                </a>
            <?php endif; ?>
        </div>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('affl_product_box', 'affililabs_product_box_shortcode');
