(function() {
      document.addEventListener("DOMContentLoaded", function() {

            const tables = document.querySelectorAll(".affl-comparison-table");
            tables.forEach(function(table) {
                  const best_price_color = table.getAttribute("data-best-price-color");
                  const best_quality_color = table.getAttribute("data-best-quality-color");
                  const best_cost_benefit_color = table.getAttribute("data-best-cost-benefit-color");

                  function hex_to_rgba(hex, alpha) {
                        if (!hex) return null;
                        const r = parseInt(hex.slice(1, 3), 16),
                              g = parseInt(hex.slice(3, 5), 16),
                              b = parseInt(hex.slice(5, 7), 16);
                        return "rgba(" + r + ", " + g + ", " + b + ", " + alpha + ")";
                  }

                  if (best_price_color) {
                        table.style.setProperty("--affl-tier-best-price-color", best_price_color);
                        const best_price_cells = table.querySelectorAll(".affl-comparison-table__product-cell--best-price, .affl-comparison-table__header-cell--best-price");
                        best_price_cells.forEach(function(cell) {
                              cell.style.backgroundColor = hex_to_rgba(best_price_color, 0.05);
                        });
                  }
                  if (best_quality_color) {
                        table.style.setProperty("--affl-tier-best-quality-color", best_quality_color);
                        const best_quality_cells = table.querySelectorAll(".affl-comparison-table__product-cell--best-quality, .affl-comparison-table__header-cell--best-quality");
                        best_quality_cells.forEach(function(cell) {
                              cell.style.backgroundColor = hex_to_rgba(best_quality_color, 0.05);
                        });
                  }
                  if (best_cost_benefit_color) {
                        table.style.setProperty("--affl-tier-best-cost-benefit-color", best_cost_benefit_color);
                        const best_cost_benefit_cells = table.querySelectorAll(".affl-comparison-table__product-cell--best-cost-benefit, .affl-comparison-table__header-cell--best-cost-benefit");
                        best_cost_benefit_cells.forEach(function(cell) {
                              cell.style.backgroundColor = hex_to_rgba(best_cost_benefit_color, 0.05);
                        });
                  }

                  const scroll_container = table.querySelector(".affl-comparison-table__scroll-container");
                  const main_table_grid = table.querySelector(".affl-comparison-table__grid");
                  const left_arrow = table.querySelector(".affl-comparison-table__nav-arrow--left");
                  const right_arrow = table.querySelector(".affl-comparison-table__nav-arrow--right");

                  const MOBILE_BREAKPOINT = 768;
                  const DESKTOP_PRODUCT_COLUMN_WIDTH = 250;
                  const MOBILE_PRODUCT_COLUMN_WIDTH = 150;
                  const CRITERIA_COLUMN_WIDTH = 250;
                  const TABLE_OUTER_BORDER_WIDTH = 2;

                  function get_current_product_column_width() {
                        return window.innerWidth <= MOBILE_BREAKPOINT ? MOBILE_PRODUCT_COLUMN_WIDTH : DESKTOP_PRODUCT_COLUMN_WIDTH;
                  }

                  if (scroll_container && main_table_grid && left_arrow && right_arrow) {
                        left_arrow.addEventListener("click", function() {
                              const scroll_amount = get_current_product_column_width();
                              if (window.innerWidth <= MOBILE_BREAKPOINT) {
                                    const new_translate_x = Math.min(0, current_translate_x + scroll_amount);
                                    const all_product_cells = table.querySelectorAll(".affl-comparison-table__header-cell--product, .affl-comparison-table__product-cell, .affl-comparison-table__sticky-product-item");
                                    all_product_cells.forEach(el => {
                                          el.style.transition = 'transform 0.3s ease-out';
                                          el.style.transform = `translateX(${new_translate_x}px)`;
                                    });
                                    current_translate_x = new_translate_x;
                                    update_arrow_position_and_visibility();
                              } else {
                                    scroll_container.scrollBy({ left: -scroll_amount, behavior: 'smooth' });
                              }
                        });

                        right_arrow.addEventListener("click", function() {
                              const scroll_amount = get_current_product_column_width();
                              if (window.innerWidth <= MOBILE_BREAKPOINT) {
                                    const new_translate_x = Math.max(max_translate_x, current_translate_x - scroll_amount);
                                    const all_product_cells = table.querySelectorAll(".affl-comparison-table__header-cell--product, .affl-comparison-table__product-cell, .affl-comparison-table__sticky-product-item");
                                    all_product_cells.forEach(el => {
                                          el.style.transition = 'transform 0.3s ease-out';
                                          el.style.transform = `translateX(${new_translate_x}px)`;
                                    });
                                    current_translate_x = new_translate_x;
                                    update_arrow_position_and_visibility();
                              } else {
                                    scroll_container.scrollBy({ left: scroll_amount, behavior: 'smooth' });
                              }
                        });
                  }

                  const sticky_product_header = table.querySelector(".affl-comparison-table__sticky-product-header");
                  const original_header_row = table.querySelector(".affl-comparison-table__header-row");
                  const sticky_products_wrapper = table.querySelector(".affl-comparison-table__sticky-products-wrapper");

                  let header_offset_top;

                  function get_admin_bar_height() {
                        const admin_bar = document.getElementById('wpadminbar');
                        return admin_bar ? admin_bar.offsetHeight : 0;
                  }

                  function calculate_header_offset() {
                        if (window.innerWidth <= MOBILE_BREAKPOINT) {
                              const mobile_header_wrapper = table.querySelector(".affl-comparison-table__mobile-product-headers-wrapper");
                              if (mobile_header_wrapper) {
                                    const rect = mobile_header_wrapper.getBoundingClientRect();
                                    header_offset_top = rect.bottom + window.scrollY;
                              } else {
                                    header_offset_top = 0;
                              }
                        } else {
                              if (original_header_row) {
                                    const rect = original_header_row.getBoundingClientRect();
                                    header_offset_top = rect.bottom + window.scrollY;
                              } else {
                                    header_offset_top = 0;
                              }
                        }
                  }

                  function update_sticky_header_visibility() {
                        if (!sticky_product_header || !original_header_row) return;

                        const current_scroll_top = window.scrollY;
                        const admin_bar_height = get_admin_bar_height();
                        const table_rect = table.getBoundingClientRect();
                        const table_bottom_offset = table_rect.bottom + window.scrollY;

                        document.documentElement.style.setProperty('--affililabs-get-wp-admin-bar-height', `${admin_bar_height}px`);

                        sticky_product_header.style.width = `${table_rect.width}px`;
                        sticky_product_header.style.left = `${table_rect.left}px`;


                        const scroll_offset = window.innerWidth <= MOBILE_BREAKPOINT ? 0 : -100;

                        const sticky_header_top_threshold = header_offset_top - admin_bar_height + scroll_offset;
                        const sticky_header_bottom_threshold = table_bottom_offset - sticky_product_header.offsetHeight;

                        const is_inside_table = current_scroll_top > sticky_header_top_threshold && current_scroll_top < sticky_header_bottom_threshold;


                        if (is_inside_table) {
                              sticky_product_header.classList.add("affl-comparison-table__sticky-product-header--visible");
                        } else {
                              sticky_product_header.classList.remove("affl-comparison-table__sticky-product-header--visible");
                        }
                  }


                  function synchronize_sticky_header_scroll() {
                        const scroll_container = table.querySelector(".affl-comparison-table__scroll-container");
                        if (sticky_products_wrapper && scroll_container && window.innerWidth > MOBILE_BREAKPOINT) {
                              sticky_products_wrapper.scrollLeft = scroll_container.scrollLeft;
                        }
                  }

                  function update_arrow_position_and_visibility() {
                        const table_rect = table.getBoundingClientRect();
                        const left_arrow = table.querySelector(".affl-comparison-table__nav-arrow--left");
                        const right_arrow = table.querySelector(".affl-comparison-table__nav-arrow--right");

                        if (!left_arrow || !right_arrow) return;
                        const is_table_in_view = table_rect.top < window.innerHeight && table_rect.bottom > 100;

                        const product_columns = table.querySelectorAll(".affl-comparison-table__header-cell--product");
                        const num_products = product_columns.length;
                        const current_product_column_width = get_current_product_column_width();
                        const total_products_width = num_products * current_product_column_width;
                        const table_visible_width = table.clientWidth;
                        const can_scroll = total_products_width > table_visible_width;

                        const arrow_absolute_top_position_doc = table_rect.top + window.scrollY + (table_rect.height * 0.10);
                        
                        const fixed_top_offset = get_admin_bar_height() + 20;

                        const should_be_fixed = (window.scrollY + fixed_top_offset) >= arrow_absolute_top_position_doc;

                        if (should_be_fixed) {
                              left_arrow.style.position = 'fixed';
                              right_arrow.style.position = 'fixed';
                              left_arrow.style.top = `${fixed_top_offset}px`;
                              right_arrow.style.top = `${fixed_top_offset}px`;

                              left_arrow.style.left = `${table_rect.left - 50}px`;
                              right_arrow.style.right = `${window.innerWidth - table_rect.right - 50}px`;
                        } else {
                              left_arrow.style.position = 'absolute';
                              right_arrow.style.position = 'absolute';
                              left_arrow.style.top = `10%`;
                              right_arrow.style.top = `10%`;

                              left_arrow.style.left = `-50px`;
                              right_arrow.style.right = `-50px`;
                        }

                        const active_scroll_element = window.innerWidth <= MOBILE_BREAKPOINT ?
                              table.querySelector(".affl-comparison-table__mobile-product-headers-wrapper") :
                              table.querySelector(".affl-comparison-table__scroll-container");

                        if (!active_scroll_element) return;

                        if (window.innerWidth <= MOBILE_BREAKPOINT) {
                              left_arrow.classList.add("affl-comparison-table__nav-arrow--hidden");
                              right_arrow.classList.add("affl-comparison-table__nav-arrow--hidden");
                              left_arrow.style.opacity = '0';
                              left_arrow.style.pointerEvents = 'none';
                              right_arrow.style.opacity = '0';
                              right_arrow.style.pointerEvents = 'none';

                        } else if (is_table_in_view && can_scroll) {
                              const desktop_scroll_container = table.querySelector(".affl-comparison-table__scroll-container");
                              if (desktop_scroll_container) {
                                    if (desktop_scroll_container.scrollLeft <= 0) {
                                          left_arrow.classList.add("affl-comparison-table__nav-arrow--hidden");
                                          left_arrow.style.opacity = '';
                                          left_arrow.style.pointerEvents = '';
                                    } else {
                                          left_arrow.classList.remove("affl-comparison-table__nav-arrow--hidden");
                                          left_arrow.style.opacity = '1';
                                          left_arrow.style.pointerEvents = 'auto';
                                    }

                                    // Use a small tolerance for floating point inaccuracies
                                    if (desktop_scroll_container.scrollLeft >= desktop_scroll_container.scrollWidth - desktop_scroll_container.clientWidth - 5) {
                                          right_arrow.classList.add("affl-comparison-table__nav-arrow--hidden");
                                          right_arrow.style.opacity = '';
                                          right_arrow.style.pointerEvents = '';
                                    } else {
                                          right_arrow.classList.remove("affl-comparison-table__nav-arrow--hidden");
                                          right_arrow.style.opacity = '1';
                                          right_arrow.style.pointerEvents = 'auto';
                                    }
                              }
                        } else {
                              left_arrow.classList.add("affl-comparison-table__nav-arrow--hidden");
                              left_arrow.style.opacity = '0';
                              left_arrow.style.pointerEvents = 'none';
                              right_arrow.classList.add("affl-comparison-table__nav-arrow--hidden");
                              right_arrow.style.opacity = '0';
                              right_arrow.style.pointerEvents = 'none';
                        }
                  }

                  window.addEventListener("scroll", update_sticky_header_visibility);
                  const desktop_scroll_container = table.querySelector(".affl-comparison-table__scroll-container");
                  if (desktop_scroll_container) {
                        desktop_scroll_container.addEventListener("scroll", synchronize_sticky_header_scroll);
                        desktop_scroll_container.addEventListener("scroll", update_arrow_position_and_visibility);
                  }

                  let start_x = 0;
                  let start_y = 0;
                  let mobile_product_headers_wrapper_ref = null;
                  let is_horizontal_swipe_detected_and_dominant = false;
                  let current_translate_x = 0;
                  let initial_translate_x_on_touch_start = 0;
                  let max_translate_x = 0;
                  const MOVEMENT_THRESHOLD = 50;
                  const DOMINANCE_THRESHOLD = 5;

                  function handle_touch_start(e) {
                        if (window.innerWidth > MOBILE_BREAKPOINT) return;
                        
                        start_x = e.touches[0].clientX;
                        start_y = e.touches[0].clientY;
                        is_horizontal_swipe_detected_and_dominant = false;

                        const first_product_cell = table.querySelector(".affl-comparison-table__header-cell--product");
                        if (first_product_cell) {
                              const transform_style = window.getComputedStyle(first_product_cell).transform;
                              if (transform_style && transform_style !== 'none') {
                                    const matrix = transform_style.match(/matrix.*\((.+)\)/)[1].split(', ');
                                    initial_translate_x_on_touch_start = parseFloat(matrix[4]);
                              } else {
                                    initial_translate_x_on_touch_start = 0;
                              }
                        } else {
                              initial_translate_x_on_touch_start = 0;
                        }
                        current_translate_x = initial_translate_x_on_touch_start;

                        const all_product_cells = table.querySelectorAll(".affl-comparison-table__header-cell--product, .affl-comparison-table__product-cell, .affl-comparison-table__sticky-product-item");
                        all_product_cells.forEach(el => {
                              el.style.transition = 'none';
                        });
                  }

                  function handle_touch_move(e) {
                        if (window.innerWidth > MOBILE_BREAKPOINT) return;

                        const touch_x = e.touches[0].clientX;
                        const touch_y = e.touches[0].clientY;
                        const delta_x_from_start = touch_x - start_x;
                        const delta_y_from_start = touch_y - start_y;

                        if (!is_horizontal_swipe_detected_and_dominant) {
                              if (Math.abs(delta_x_from_start) > DOMINANCE_THRESHOLD || Math.abs(delta_y_from_start) > DOMINANCE_THRESHOLD) { // Initial small threshold to detect any movement
                                    if (Math.abs(delta_x_from_start) > Math.abs(delta_y_from_start)) { // Horizontal is dominant
                                          if (Math.abs(delta_x_from_start) >= MOVEMENT_THRESHOLD) { // Horizontal movement exceeds 50px threshold
                                                is_horizontal_swipe_detected_and_dominant = true;
                                                e.preventDefault(); // Prevent native scrolling
                                          } else {
                                                const all_product_cells = table.querySelectorAll(".affl-comparison-table__header-cell--product, .affl-comparison-table__product-cell, .affl-comparison-table__sticky-product-item");
                                                all_product_cells.forEach(el => {
                                                      el.style.transform = `translateX(${initial_translate_x_on_touch_start}px)`;
                                                });
                                          }
                                    }
                              }
                        }

                        if (is_horizontal_swipe_detected_and_dominant) {
                              e.preventDefault(); // Continue preventing native scroll
                              
                        }
                  }

                  function handle_touch_end(e) {
                        if (window.innerWidth > MOBILE_BREAKPOINT) return;

                        const all_product_cells = table.querySelectorAll(".affl-comparison-table__header-cell--product, .affl-comparison-table__product-cell, .affl-comparison-table__sticky-product-item");
                        all_product_cells.forEach(el => {
                              el.style.transition = 'transform 0.3s ease-out';
                        });

                        let final_translate_x = current_translate_x; // Start with the current position from touchmove

                        if (is_horizontal_swipe_detected_and_dominant) {
                              const end_x = e.changedTouches[0].clientX;
                              const swipe_distance = end_x - start_x;

                              if (Math.abs(swipe_distance) >= MOVEMENT_THRESHOLD) {
                                    let current_column_index = Math.round(current_translate_x / MOBILE_PRODUCT_COLUMN_WIDTH);
                                    
                                    if (swipe_distance > 0) { // Swiped right (moving towards less negative, or 0)
                                          final_translate_x = (current_column_index + 1) * MOBILE_PRODUCT_COLUMN_WIDTH;
                                    } else { // Swiped left (moving towards more negative)
                                          final_translate_x = (current_column_index - 1) * MOBILE_PRODUCT_COLUMN_WIDTH;
                                    }
                              }
                        }
                        
                        final_translate_x = Math.max(max_translate_x, Math.min(0, final_translate_x));


                        all_product_cells.forEach(el => {
                              el.style.transform = `translateX(${final_translate_x}px)`;
                        });
                        current_translate_x = final_translate_x;

                        update_arrow_position_and_visibility();
                  }

                  window.addEventListener("scroll", update_sticky_header_visibility);
                  window.addEventListener("scroll", update_arrow_position_and_visibility);
                  window.addEventListener("resize", function() {
                        calculate_header_offset();
                        update_sticky_header_visibility();
                        update_arrow_position_and_visibility();
                        set_optimal_desktop_table_width();
                  });

                  calculate_header_offset();
                  update_sticky_header_visibility();
                  update_arrow_position_and_visibility();
                  set_optimal_desktop_table_width();

                  function set_mobile_table_width() {
                        if (window.innerWidth > MOBILE_BREAKPOINT) return;

                        const table_element = document.querySelector(".affl-comparison-table");
                        if (!table_element) return;

                        const available_width = table_element.clientWidth;
                        
                        let num_columns_to_fit = Math.floor(available_width / MOBILE_PRODUCT_COLUMN_WIDTH);
                        num_columns_to_fit = Math.max(2, Math.min(10, num_columns_to_fit));

                        const optimal_table_width = (num_columns_to_fit * MOBILE_PRODUCT_COLUMN_WIDTH) + num_columns_to_fit;

                        table_element.style.width = `${optimal_table_width}px`;
                        table_element.style.margin = `0 auto`;
                  }

                  function set_optimal_desktop_table_width() {
                        if (window.innerWidth <= MOBILE_BREAKPOINT) return;

                        const table_element = document.querySelector(".affl-comparison-table");
                        if (!table_element) return;

                        const parent_element = table_element.offsetParent || table_element.parentElement;
                        if (!parent_element) return;

                        const available_space_in_parent = parent_element.clientWidth;
                        const num_total_product_columns = table_element.querySelectorAll(".affl-comparison-table__header-cell--product").length;

                        const available_space_for_columns = available_space_in_parent - CRITERIA_COLUMN_WIDTH - TABLE_OUTER_BORDER_WIDTH;
                        let num_product_columns_to_display = Math.floor(available_space_for_columns / DESKTOP_PRODUCT_COLUMN_WIDTH);
                        
                        num_product_columns_to_display = Math.max(1, num_product_columns_to_display);
                        num_product_columns_to_display = Math.min(num_total_product_columns, num_product_columns_to_display);

                        const final_table_width = CRITERIA_COLUMN_WIDTH + (num_product_columns_to_display * DESKTOP_PRODUCT_COLUMN_WIDTH) + TABLE_OUTER_BORDER_WIDTH;

                        table_element.style.width = `${final_table_width}px`;
                        table_element.style.margin = `0 auto`;
                  }

                  function debounce(func, delay) {
                        let timeout;
                        return function(...args) {
                              const context = this;
                              clearTimeout(timeout);
                              timeout = setTimeout(() => func.apply(context, args), delay);
                        };
                  }

                  const debounced_set_mobile_table_width = debounce(set_mobile_table_width, 100);

                  function apply_mobile_criteria_transposition() {
                        if (window.innerWidth > MOBILE_BREAKPOINT) return;

                        const table = document.querySelector(".affl-comparison-table");
                        if (!table) return;

                        const scroll_container = table.querySelector(".affl-comparison-table__scroll-container");
                        const original_grid = table.querySelector(".affl-comparison-table__grid");

                        if (!scroll_container || !original_grid) return;

                        original_grid.style.display = 'none';

                        let transposed_content_container = table.querySelector(".affl-comparison-table__mobile-transposed-content");
                        if (!transposed_content_container) {
                              transposed_content_container = document.createElement("div");
                              transposed_content_container.classList.add("affl-comparison-table__mobile-transposed-content");
                              scroll_container.parentNode.insertBefore(transposed_content_container, scroll_container.nextSibling);
                        } else {
                              transposed_content_container.innerHTML = '';
                        }

                        const header_row = original_grid.querySelector(".affl-comparison-table__header-row");
                        if (header_row) {
                              const product_header_cells = Array.from(header_row.querySelectorAll(".affl-comparison-table__header-cell--product"));
                              
                              let mobile_product_headers_wrapper = scroll_container.querySelector(".affl-comparison-table__mobile-product-headers-wrapper");
                              if (!mobile_product_headers_wrapper) {
                                    mobile_product_headers_wrapper = document.createElement("div");
                                    mobile_product_headers_wrapper.classList.add("affl-comparison-table__mobile-product-headers-wrapper");
                                    scroll_container.prepend(mobile_product_headers_wrapper); 
                              } else {
                                    mobile_product_headers_wrapper.innerHTML = '';
                              }

                              product_header_cells.forEach(cell => {
                                    const cloned_cell = cell.cloneNode(true);
                                    mobile_product_headers_wrapper.appendChild(cloned_cell);
                              });
                        }

                        const criteria_rows = Array.from(original_grid.querySelectorAll(".affl-comparison-table__row:not(.affl-comparison-table__header-row)"));
                        criteria_rows.forEach(row => {
                              const criteria_cell = row.querySelector(".affl-comparison-table__criteria-cell");
                              if (criteria_cell) {
                                    const criteria_label = criteria_cell.textContent.trim();

                                    const new_criteria_label_row = document.createElement("div");
                                    new_criteria_label_row.classList.add("affl-comparison-table__mobile-criteria-full-row");
                                    new_criteria_label_row.textContent = criteria_label;
                                    transposed_content_container.appendChild(new_criteria_label_row);

                                    const mobile_product_data_row = document.createElement("div");
                                    mobile_product_data_row.classList.add("affl-comparison-table__mobile-product-data-row");
                                    
                                    const product_cells = Array.from(row.querySelectorAll(".affl-comparison-table__product-cell"));
                                    product_cells.forEach(cell => {
                                          const cloned_cell = cell.cloneNode(true);
                                          mobile_product_data_row.appendChild(cloned_cell);
                                    });
                                    transposed_content_container.appendChild(mobile_product_data_row);

                              }
                        });

                        set_mobile_table_width();

                        const num_products = table.querySelectorAll(".affl-comparison-table__mobile-product-headers-wrapper .affl-comparison-table__header-cell--product").length;
                        const total_products_width = num_products * MOBILE_PRODUCT_COLUMN_WIDTH;
                        
                        const table_visible_width = table.clientWidth;
                        const num_visible_columns = Math.floor(table_visible_width / MOBILE_PRODUCT_COLUMN_WIDTH);
                        
                        max_translate_x = Math.min(0, -(num_products - num_visible_columns) * MOBILE_PRODUCT_COLUMN_WIDTH); // max_translate_x is negative or 0

                        const all_product_cells = table.querySelectorAll(".affl-comparison-table__header-cell--product, .affl-comparison-table__product-cell, .affl-comparison-table__sticky-product-item");
                        all_product_cells.forEach(el => {
                              el.style.transform = `translateX(0px)`;
                        });
                        current_translate_x = 0;
                  }

                  function revert_mobile_criteria_transposition() {
                        const table = document.querySelector(".affl-comparison-table");
                        if (!table) return;

                        const original_grid = table.querySelector(".affl-comparison-table__grid");
                        if (original_grid) {
                              original_grid.style.display = '';
                        }

                        const transposed_content_container = table.querySelector(".affl-comparison-table__mobile-transposed-content");
                        if (transposed_content_container) {
                              transposed_content_container.remove();
                        }

                        const mobile_product_headers_wrapper = table.querySelector(".affl-comparison-table__mobile-product-headers-wrapper");
                        if (mobile_product_headers_wrapper) {
                              mobile_product_headers_wrapper.remove();
                        }


                  }

                  const media_query_list = window.matchMedia(`(max-width: ${MOBILE_BREAKPOINT}px)`);

                  function handle_mobile_layout_change(event) {
                        if (event.matches) {
                              table.classList.add("affl-comparison-table--mobile-active");
                              apply_mobile_criteria_transposition();
                              calculate_header_offset();

                              table.addEventListener("touchstart", handle_touch_start);
                              table.addEventListener("touchmove", handle_touch_move);
                              table.addEventListener("touchend", handle_touch_end);

                        } else {
                              table.classList.remove("affl-comparison-table--mobile-active");
                              revert_mobile_criteria_transposition();
                              table.removeEventListener("touchstart", handle_touch_start);
                              table.removeEventListener("touchmove", handle_touch_move);
                              table.removeEventListener("touchend", handle_touch_end);
                        }
                        update_sticky_header_visibility();
                        update_arrow_position_and_visibility();
                  }

                  handle_mobile_layout_change(media_query_list);
            });
      });
})();
