<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

affl_import(AFFILILABS_SERVICE_CRYPT_SERVICE_FILE);
affl_import(AFFILILABS_SERVICES_LOGGER_FILE); 


class AFFLSettingsService {

    private const INTERNAL_API_KEY_OPTION = 'affl_internal_api_key';

    /**
     * Saves the internal AffiliLabs API key securely.
     *
     * @param string $api_key The raw API key to save.
     * @return bool True on success, throws an exception on failure.
     * @throws Exception If the API key is invalid or encryption fails.
     */
    public static function save_internal_api_key($api_key) {


        $sanitized_key = sanitize_text_field(trim($api_key));

        if (strlen($sanitized_key) < 10) {
            throw new Exception(esc_html__('API key must be at least 10 characters long.', 'affililabs'));
        }

        $salt = AFFLCryptographyService::get_or_generate_salt(self::INTERNAL_API_KEY_OPTION);
        if (empty($salt)) {
            throw new Exception(esc_html__('Failed to generate or retrieve salt for API key encryption.', 'affililabs'));
        }

        $encrypted_key = AFFLCryptographyService::encrypt_api_key($sanitized_key, $salt);
        if (empty($encrypted_key)) {
            AFFLCryptographyService::delete_salt(self::INTERNAL_API_KEY_OPTION);
            throw new Exception(esc_html__('Failed to encrypt API key.', 'affililabs'));
        }

        return update_option(self::INTERNAL_API_KEY_OPTION, $encrypted_key, 'no'); // Autoload 'no'
    }

    /**
     * Retrieves and decrypts the internal AffiliLabs API key.
     *
     * @return string The decrypted API key, or an empty string if not set or decryption fails.
     */
    public static function get_internal_api_key() {


        $encrypted_key = get_option(self::INTERNAL_API_KEY_OPTION);
        if (empty($encrypted_key)) {
            return ''; // No key saved
        }

        $salt = AFFLCryptographyService::get_or_generate_salt(self::INTERNAL_API_KEY_OPTION);
        if (empty($salt)) {
            AFFL_LoggerService::log('Salt missing for internal API key decryption.');
            return ''; // Cannot decrypt without salt
        }

        $decrypted_key = AFFLCryptographyService::decrypt_api_key($encrypted_key, $salt);
        if ($decrypted_key === false) {
            AFFL_LoggerService::log('Failed to decrypt internal API key.');
            return '';
        }

        return $decrypted_key;
    }

     /**
     * Deletes the internal AffiliLabs API key and its salt.
     *
     * @return bool True if the option was successfully deleted, false otherwise.
     */
    public static function delete_internal_api_key() {


        AFFLCryptographyService::delete_salt(self::INTERNAL_API_KEY_OPTION);
        
        return delete_option(self::INTERNAL_API_KEY_OPTION);
    }

    /**
     * Saves the debug mode setting.
     *
     * @param int $enable_debug 1 to enable debug mode, 0 to disable.
     * @return bool True on success, false on failure.
     */
    public static function save_debug_mode($enable_debug): bool {


        $current_value = get_option(AFFILILABS_ENABLE_DEBUG_OPTION, 0);

        if ($current_value !== $enable_debug) {
            update_option(AFFILILABS_ENABLE_DEBUG_OPTION, $enable_debug, 'no');
        }
        return true;
    }

    /**
     * Retrieves the debug mode setting.
     *
     * @return int 1 if debug mode is enabled, 0 if it is disabled.
     */
    public static function get_debug_mode(): int {

        return intval(get_option(AFFILILABS_ENABLE_DEBUG_OPTION, 0));
    }

    /**
     * Saves the AI language setting.
     *
     * @param string $language The language code or custom language string.
     * @return bool True on success, false on failure.
     */
    public static function save_ai_language($language): bool {

        $sanitized_language = sanitize_text_field(trim($language));
        $current_language = get_option(AFFILILABS_AI_LANGUAGE_OPTION, 'english');

        if ($current_language !== $sanitized_language) {
            return update_option(AFFILILABS_AI_LANGUAGE_OPTION, $sanitized_language, 'no');
        }
        return true;
    }

    /**
     * Retrieves the AI language setting.
     *
     * @return string The saved AI language, defaults to 'english'.
     */
    public static function get_ai_language() {

        return get_option(AFFILILABS_AI_LANGUAGE_OPTION, 'english');
    }

    public static function save_list_separator($separator) {

        $sanitized_separator = sanitize_text_field(trim($separator));
        $current_separator = get_option(AFFILILABS_LIST_SEPARATOR_OPTION, 'newline');

        if ($current_separator !== $sanitized_separator) {
            return update_option(AFFILILABS_LIST_SEPARATOR_OPTION, $sanitized_separator, 'no');
        }
        return true;
    }

    public static function get_list_separator() {

        return get_option(AFFILILABS_LIST_SEPARATOR_OPTION, 'newline');
    }
    
    public static function save_enable_all_categories_page($value) {

        $sanitized_value = intval($value);
        $current_value = (int) get_option(AFFILILABS_ENABLE_ALL_CATEGORIES_PAGE_OPTION, 0);

        if ($current_value !== $sanitized_value) {
            return update_option(AFFILILABS_ENABLE_ALL_CATEGORIES_PAGE_OPTION, $sanitized_value, 'no');
        }
        return true;
    }
    
    public static function get_enable_all_categories_page() {

      return get_option(AFFILILABS_ENABLE_ALL_CATEGORIES_PAGE_OPTION, 1);
    }

    /**
     * Saves the setting to enable/disable Optimizepress global elements.
     *
     * @param int $value 1 to enable, 0 to disable.
     * @return bool True on success, false on failure.
     */
    public static function save_enable_op_global_elements($value): bool {

        $sanitized_value = intval($value);
        $current_value = (int) get_option(AFFILILABS_ENABLE_OP_GLOBAL_ELEMENTS_OPTION, 0);

        if ($current_value !== $sanitized_value) {
            $updated = update_option(AFFILILABS_ENABLE_OP_GLOBAL_ELEMENTS_OPTION, $sanitized_value, 'no');
            return $updated;
        }
        return true;
    }

    /**
     * Retrieves the setting for enabling/disabling Optimizepress global elements.
     *
     * @return int 1 if enabled, 0 if disabled.
     */
    public static function get_enable_op_global_elements(): int {

        return (int) get_option(AFFILILABS_ENABLE_OP_GLOBAL_ELEMENTS_OPTION, 0);
    }

    /**
     * Saves the flag indicating whether Optimizepress global elements have been created.
     *
     * @param int $value 1 if elements have been created, 0 otherwise.
     * @return bool True on success, false on failure.
     */
    public static function save_op_global_elements_created_flag(int $value): bool {

        $sanitized_value = intval($value);
        $current_value = (int) get_option(AFFILILABS_OP_GLOBAL_ELEMENTS_CREATED_FLAG, 0);

        if ($current_value !== $sanitized_value) {
            $updated = update_option(AFFILILABS_OP_GLOBAL_ELEMENTS_CREATED_FLAG, $sanitized_value, 'no');
            return $updated;
        }
        return true;
    }

    /**
     * Retrieves the flag indicating whether Optimizepress global elements have been created.
     *
     * @return int 1 if elements have been created, 0 otherwise.
     */
    public static function get_op_global_elements_created_flag(): int {

        return (int) get_option(AFFILILABS_OP_GLOBAL_ELEMENTS_CREATED_FLAG, 0);
    }

    public static function save_enable_custom_template($value) {

        $sanitized_value = intval($value);
        $current_value = (int) get_option(AFFILILABS_ENABLE_CUSTOM_TEMPLATE_OPTION, 0);

        if ($current_value !== $sanitized_value) {
            return update_option(AFFILILABS_ENABLE_CUSTOM_TEMPLATE_OPTION, $sanitized_value, 'no');
        }
        return true;
    }

    public static function get_enable_custom_template() {

      return get_option(AFFILILABS_ENABLE_CUSTOM_TEMPLATE_OPTION, 0);
    }

    public static function save_cleanup_on_uninstall($value) {

        $sanitized_value = intval($value);
        $current_value = (int) get_option('affililabs_cleanup_on_uninstall', 0);

        if ($current_value !== $sanitized_value) {
            return update_option('affililabs_cleanup_on_uninstall', $sanitized_value, 'no');
        }
        return true;
    }
}
