<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

affl_import(AFFILILABS_MODEL_PRODUCT_FINDER_ABSTRACT_FILE);
class AFFLDigistoreProductFinder extends AFFLProductFinder {

    private const API_BASE_URL = 'https://www.digistore24-app.com/v2/api/marketplace/results';
    private const IMAGE_URL_PREFIX = 'https://www.digistore24-app.com';

    /**
     * Finds products on Digistore24 based on a keyword.
     *
     * @param string $keyword The search term.
     * @param array $filters Optional filters for the search.
     * @return array An array of formatted product data. Returns empty array on failure or no results.
     * @throws Exception If an error occurs during the API call or data processing.
     */
    public function find_products(string $keyword, array $filters = []): array {
        $formatted_products = [];
        $encoded_keyword = urlencode($keyword);

        $currency = $filters['currency'] ?? 'USD';
        $language = $filters['language'] ?? [];
        $sort = $filters['sort'] ?? 'stars';

        $api_url_args = array(
            'query' => $encoded_keyword,
            'page' => 1,
            'itemsPerPage' => 100,
            'sort' => $sort,
        );

        if (!empty($currency)) {
            $api_url_args['currency'] = $currency;
        }

        // If empty, Digistore API defaults to all languages
        if (!empty($language) && is_array($language)) {
            $api_url_args['language'] = $language; // Pass the array directly (no need to transform data)
        }

        $marketplace_category_ids = $filters['marketplace_category_id'] ?? [];
        if (!empty($marketplace_category_ids) && is_array($marketplace_category_ids)) {
            $api_url_args['marketplaceCategoryId'] = $marketplace_category_ids;
        }

        $api_url = add_query_arg($api_url_args, self::API_BASE_URL);

        AFFL_LoggerService::debug('Digistore24 API Request URL: ' . esc_url($api_url), 'DEBUG');

        try {
            $response = wp_remote_get($api_url, array(
                'timeout' => 30,
            ));
            if (is_wp_error($response)) {
                AFFL_LoggerService::log('Digistore24 API request failed: ' . esc_html($response->get_error_message()), 'ERROR');
                throw new Exception('API request failed: ' . $response->get_error_message());
            }

            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);

            if ($response_code === 401) {
                AFFL_LoggerService::debug('AI API request failed due to invalid credentials.', 'ERROR');
                throw new Exception('Unauthorized access to AI API. Consider upgrading.');
            }

            if ($response_code !== 200) {
                AFFL_LoggerService::log("Digistore24 API request returned status code " . esc_html($response_code) . ". Body: " . esc_html($response_body) . ", Response: " . esc_html(print_r($response, true)), 'ERROR');
                throw new Exception("There was an error while processing the request. API request returned status code {$response_code}");
            }

            $data = json_decode($response_body, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                AFFL_LoggerService::log('Failed to decode Digistore24 API response JSON. Error: ' . esc_html(json_last_error_msg()) . ". Body: " . esc_html($response_body), 'ERROR');
                throw new Exception('Failed to decode API response JSON: ' . json_last_error_msg());
            }

            if (!isset($data['result']['items']) || !is_array($data['result']['items'])) {
                AFFL_LoggerService::log('Unexpected Digistore24 API response structure. Body: ' . esc_html($response_body), 'WARNING');
                return [];
            }

            foreach ($data['result']['items'] as $item) {
                if (!is_array($item) || !isset($item['label'])) {
                    continue;
                }

                $image_url = (!empty($item['imageUrl'])) ? self::IMAGE_URL_PREFIX . $item['imageUrl'] : AFFILILABS_DEFAULT_PRODUCT_IMAGE;

                $formatted_products[] = array(
                    'title' => sanitize_text_field($item['label'] ?? ''),
                    'category' => sanitize_text_field($item['type'] ?? 'N/A'),
                    'image_url' => esc_url($image_url),
                    'cancel_rate' => isset($item['cancelRate']) ? affl_sanitize_float_input($item['cancelRate']) : null,
                    'vendor_name' => sanitize_text_field($item['vendorName'] ?? 'N/A'),
                    'price' => isset($item['price']) ? affl_sanitize_float_input($item['price']) : null,
                    'currency' => sanitize_text_field($data['result']['currency'] ?? 'USD'),
                    'earnings_per_sale' => isset($item['earningsPerSale']) ? affl_sanitize_float_input($item['earningsPerSale']) : null,
                    'commission_percent' => isset($item['commission']) ? affl_sanitize_float_input($item['commission']) : null,
                    'promo_link_url' => esc_url($item['promoLinkUrl'] ?? ''),
                    'created_at' => sanitize_text_field($item['createdAt'] ?? ''),
                    'accepts_affiliations_automatically' => isset($item['acceptsAffiliationsAutomatically']) ? boolval($item['acceptsAffiliationsAutomatically']) : false,
                    'description' => wp_kses_post($item['description'] ?? ''),                    'product_id_external' => sanitize_text_field($item['productId'] ?? ''),
                    'marketplace' => 'digistore24',
                );
            }

        } catch (Exception $e) {
            AFFL_LoggerService::log('Error in AFFLDigistoreProductFinder::find_products: ' . esc_html($e->getMessage()), 'ERROR');
            throw $e;
        }

        return $formatted_products;
    }
}
