<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

affl_import(AFFILILABS_SERVICE_SETTINGS_FILE);

/**
 * Optimizepress integration logic for AffiliLabs.
 */
class Affililabs_Optimizepress_Integration
{
    const PRODUCT_FIELDS_CATEGORY_ID = 'affililabs_product_fields';
    const PRODUCT_FIELDS_CATEGORY_TITLE = 'AffiliLabs Product Fields';

    public function __construct()
    {
        add_action('admin_init', [$this, 'register_product_field_global_elements'], 99); // Postpone registration to ensure that the global elements are registered after Optimizepress Global Element CPT itself is ready
    }

    /**
     * Registers global elements for AffiliLabs product fields.
     */
    public function register_product_field_global_elements()
    {
        if (!post_type_exists('op_global_element')) {
            return;
        }

        if (!function_exists('op3_element_config') || !class_exists('OPBuilder\\Repositories\\GlobalElementRepository')) {
            return;
        }

        if (!AFFLSettingsService::get_enable_op_global_elements()) {
            return;
        }

        if (AFFLSettingsService::get_op_global_elements_created_flag()) {
            return;
        }
        $elements_to_register = [
            0 => [
                'title' => 'Product Name',
                'tag' => '[affl_name]',
            ],
            1 => [
                'title' => 'Product Price',
                'tag' => '[affl_price]',
            ],
            2 => [
                'title' => 'Product Rating',
                'tag' => '[affl_rating]',
            ],
            3 => [
                'title' => 'Product Description',
                'tag' => '[affl_description]',
            ],
            4 => [
                'title' => 'Product Features',
                'tag' => '[affl_features]',
            ],
            5 => [
                'title' => 'Product Pros',
                'tag' => '[affl_pros]',
            ],
            6 => [
                'title' => 'Product Cons',
                'tag' => '[affl_cons]',
            ],
            7 => [
                'title' => 'Product Category',
                'tag' => '[affl_category]',
            ],
            8 => [
                'title' => 'Product Marketplace',
                'tag' => '[affl_marketplace]',
            ],
            9 => [
                'title' => 'Product Image',
                'tag' => '[affl_image]',
            ],
            10 => [
                'title' => 'Product Button',
                'tag' => '[affl_button]',
            ],
        ];

        try {
            foreach ($elements_to_register as $index => $element) {
                $this->_create_product_field_global_element($element['title'], $element['tag']);
            }
        } catch (Throwable $e) {
            AFFLSettingsService::save_enable_op_global_elements(0);
            AFFL_LoggerService::log(
                'Error: Failed to register Optimizepress Global Elements. Integration was disabled for safety reasons. Error: ' . esc_html($e->getMessage()) . ' Stack: ' . esc_html($e->getTraceAsString()),
                'ERROR'
            );
        }
    }

    /**
     * Creates or updates a global element for a given shortcode.
     *
     * @param string $title The title of the global element.
     * @param string $shortcode_tag The shortcode tag (e.g., "[affl_name]").
     */
    private function _create_product_field_global_element($title, $shortcode_tag)
    {
        $unique_id = md5($shortcode_tag);

        $existing_elements = get_posts([
            'post_type'      => 'op_global_element',
            'meta_key'       => AFFILILABS_OP_GLOBAL_ELEMENT_UNIQUE_ID,
            'meta_value'     => $unique_id,
            'fields'         => 'ids',
            'numberposts'    => 1,
            'post_status'    => 'publish',
        ]);

        $element_id = 0;
        if (!empty($existing_elements)) {
            $element_id = reset($existing_elements);
        }

        $element_meta = [
            'type'        => 'customhtml',
            'title'       => $title,
            /* translators: %s: element title */
            'description' => sprintf(__('Displays the %s.', 'affililabs'), strtolower($title)),
            'category_id' => self::PRODUCT_FIELDS_CATEGORY_ID,
            'thumb'       => AFFILILABS_ASSETS_LOGO_DARK_50PX,
        ];

        $element_data = [
            'type'        => 'customhtml',
            'title'       => $title,
            /* translators: %s: element title */
            'description' => sprintf(__('Displays the %s.', 'affililabs'), strtolower($title)),
            'category_id' => self::PRODUCT_FIELDS_CATEGORY_ID,
            'thumb'       => AFFILILABS_ASSETS_LOGO_DARK_50PX,
            'data'        => [
                'type' => 'customhtml',
                'options' => [
                    'all' => [
                        'codeHtml' => [ '[data-op3-code-html]' => $shortcode_tag ],
                        'marginLeft' => 'auto',
                        'marginRight' => 'auto',
                    ],
                ],
                'children' => [],
            ],
        ];

        /** @disregard P1009 - It is considered an undefined function because it comes from a third-party plugin */
        $global_element_repository = new \OPBuilder\Repositories\GlobalElementRepository();

        if (!$element_id) {
            $new_element_object = $global_element_repository->create($element_data);
            
            if ($new_element_object && !is_wp_error($new_element_object) && isset($new_element_object->post->ID)) {
                $new_element_id = (int) $new_element_object->post->ID;
                update_post_meta($new_element_id, '_op3_ge_meta', json_encode($element_meta));
                update_post_meta($new_element_id, AFFILILABS_OP_GLOBAL_ELEMENT_UNIQUE_ID, $unique_id);
                AFFLSettingsService::save_op_global_elements_created_flag(1);
            }
        } else {
            // If element already exists, ensure the flag is set
            AFFLSettingsService::save_op_global_elements_created_flag(1);
        }
    }

    public function is_op3_present() {
        if (is_plugin_active('op-dashboard/op-dashboard.php') || is_plugin_active('op-builder/op-builder.php')) {
            return true;
        }

        if (class_exists('OptimizePress') || class_exists('OPBuilder')) {
            return true;
        }

        $current_theme_slug = get_template();
        $op_theme_slugs = array( 'optimizepress', 'op-smarttheme', 'optimizepress-smarttheme', 'op-minimal-theme' );

        if (in_array($current_theme_slug, $op_theme_slugs)) {
            return true;
        }

        return false;
    }
}

new Affililabs_Optimizepress_Integration();
