<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

affl_import(AFFILILABS_SERVICE_CRYPT_SERVICE_FILE);
affl_import(AFFILILABS_SERVICES_LOGGER_FILE);

class AFFLMarketplaceService {

    /**
     * Updates the product count meta for a given marketplace.
     *
     * @param int $marketplace_id The ID of the marketplace post.
     * @return void
     */
    public static function update_product_count(int $marketplace_id): void {
        $args = array(
            'post_type' => AFFILILABS_CPT_PRODUCT,
            'posts_per_page' => -1,
            'post_status' => 'any',
            'meta_query' => array(
                array(
                    'key' => AFFILILABS_META_KEY_MARKETPLACE,
                    'value' => $marketplace_id,
                ),
            ),
            'fields' => 'ids',
        );

        $products_query = new WP_Query($args);
        $product_count = $products_query->found_posts;

        update_post_meta($marketplace_id, AFFILILABS_META_KEY_MARKETPLACE_TOTAL_PRODUCTS, $product_count);
    }

    /**
     * Checks if a specific supported marketplace has been connected (created).
     *
     * @param string $marketplace_type The type of marketplace (e.g., 'digistore').
     * @return bool True if connected, false otherwise.
     */
    public static function is_marketplace_connected(string $marketplace_type): bool {
        if ($marketplace_type === 'digistore') {
            $digistore_id = get_option(AFFILILABS_DIGISTORE_MARKETPLACE_ID);

            if ($digistore_id !== false && !empty($digistore_id)) {

                $marketplace_post = get_post($digistore_id);
                if ($marketplace_post && $marketplace_post->post_type === AFFILILABS_CPT_MARKETPLACE) {
                    return true;
                } else {
                    AFFL_LoggerService::log('Digistore marketplace ID found in options (' . esc_html($digistore_id) . ') but post does not exist or is not a marketplace. Deleting option.', 'WARNING');
                    delete_option(AFFILILABS_DIGISTORE_MARKETPLACE_ID);
                }
            }
        }
        return false;
    }

    /**
     * Saves or updates a marketplace post.
     *
     * @param array $marketplace_data Associative array of marketplace data. Expected keys:
     *        'id' (int, optional), 'name' (string), 'url' (string), 'category' (string|int, optional),
     *        'affiliate_id' (string, optional), 'image_id' (int, optional),
     *        'api_key' (string, optional), 'custom_fields' (array, optional).
     * @return int|WP_Error The post ID on success, or WP_Error on failure.
     * @throws Exception If saving fails and results in WP_Error.
     */
    public static function save(array $marketplace_data) {
        $marketplace_id = isset($marketplace_data['id']) ? intval($marketplace_data['id']) : 0;
        $category_input = $marketplace_data['category'] ?? null;
        $category_term = $category_input ? term_exists($category_input, AFFILILABS_TAXONOMY_MARKETPLACE_CATEGORY) : null;

        $post_data = array(
            'post_title'    => sanitize_text_field($marketplace_data['name'] ?? ''),
            'post_content'  => '',
            'post_status'   => 'publish',
            'post_type'     => AFFILILABS_CPT_MARKETPLACE,
            'post_author'   => get_current_user_id(),
            'tax_input'     => array(),
            'meta_input'    => array(
                AFFILILABS_META_KEY_MARKETPLACE_URL => esc_url_raw($marketplace_data['url'] ?? ''),
                AFFILILABS_META_KEY_MARKETPLACE_AFFILIATE_ID => sanitize_text_field($marketplace_data['affiliate_id'] ?? ''),
                AFFILILABS_META_KEY_MARKETPLACE_LAST_UPDATE => current_time('mysql'),
            )
        );

        $digistore_id = get_option(AFFILILABS_DIGISTORE_MARKETPLACE_ID);
        $status_mode = 'manual';
        if ($marketplace_id > 0 && $marketplace_id == $digistore_id) {
            $status_mode = 'active';
        }
        $post_data['meta_input'][AFFILILABS_META_KEY_MARKETPLACE_STATUS_MODE] = $status_mode;

        if (is_array($category_term)) {
            $post_data['tax_input'][AFFILILABS_TAXONOMY_MARKETPLACE_CATEGORY] = array($category_term['term_id']);
        } elseif (is_numeric($category_term)) {
             $post_data['tax_input'][AFFILILABS_TAXONOMY_MARKETPLACE_CATEGORY] = array(intval($category_term));
        }

        if (isset($marketplace_data['custom_fields']) && is_array($marketplace_data['custom_fields'])) {
            foreach ($marketplace_data['custom_fields'] as $key => $value) {
                $sanitized_key = sanitize_key($key);
                if (!empty($sanitized_key)) {
                    $post_data['meta_input']["_affl_marketplace_custom_" . $sanitized_key] = sanitize_text_field($value);
                }
            }
        }

        $result_id = 0;
        if ($marketplace_id > 0) {
            $post_data['ID'] = $marketplace_id;
            $result_id = wp_update_post($post_data, true); // Pass true for WP_Error object on failure
        } else {
            $result_id = wp_insert_post($post_data, true); // Idem
        }

        if (is_wp_error($result_id)) {
            AFFL_LoggerService::log('Error saving marketplace: ' . esc_html($result_id->get_error_message()), 'ERROR');
            throw new Exception('Error saving marketplace: ' . esc_html($result_id->get_error_message()));
        }

        $saved_marketplace_id = $result_id;

        $thumbnail_id = isset($marketplace_data['image_id']) ? intval($marketplace_data['image_id']) : 0;
        if ($thumbnail_id > 0) {
            set_post_thumbnail($saved_marketplace_id, $thumbnail_id);
        } elseif (isset($marketplace_data['image_id']) && $marketplace_data['image_id'] === null) {
             delete_post_thumbnail($saved_marketplace_id);
        }


        if (isset($marketplace_data['api_key'])) {
            $saved_post = get_post($saved_marketplace_id);
            if ($saved_post) {
                 $marketplace_slug = $saved_post->post_name;
                 self::save_api_key($marketplace_data['api_key'], $marketplace_slug);
            }
        }

        self::update_product_count($saved_marketplace_id);

        return $saved_marketplace_id;
    }

    /**
     * Creates a default supported marketplace (e.g., Digistore24) if it doesn't already exist.
     * This logic was refactored from AffiliLabs::activate_plugin().
     *
     * @param string $marketplace_type The type of marketplace to create (e.g., 'digistore').
     * @return int|WP_Error The post ID on success, or WP_Error on failure.
     */
    public static function create_default_marketplace(string $marketplace_type) {
        if ($marketplace_type === 'digistore') {
            if (self::is_marketplace_connected('digistore')) {
                $existing_id = get_option(AFFILILABS_DIGISTORE_MARKETPLACE_ID);
                AFFL_LoggerService::debug('Digistore marketplace already connected and valid (ID: ' . esc_html($existing_id) . '). Skipping creation.', 'DEBUG');
                return $existing_id;
            }

            $image_url_to_import = AFFILILABS_DIGISTORE24_LOGO;
            
            
            $image_id = media_sideload_image($image_url_to_import, 0, null, 'id');
            if (is_wp_error($image_id)) {
              AFFL_LoggerService::log('Failed to sideload Digistore24 logo: ' . esc_html($image_id->get_error_message()), 'WARNING');
              $image_id = null;
            }
            
            $marketplace_data = array(
                'name' => 'Digistore24',
                'url' => 'https://digistore24.com',
                'category' => null,
                'affiliate_id' => '',
                'image_id' => $image_id,
                'api_key' => null,
            );

            $marketplace_id = self::save($marketplace_data);

            if (is_numeric($marketplace_id) && $marketplace_id > 0) {
                update_option(AFFILILABS_DIGISTORE_MARKETPLACE_ID, $marketplace_id, true);
                update_post_meta($marketplace_id, AFFILILABS_META_KEY_MARKETPLACE_STATUS_MODE, 'active');
                AFFL_LoggerService::log('Digistore24 marketplace created successfully with ID: ' . esc_html($marketplace_id) . ' and status set to active.', 'INFO');
                AFFL_LoggerService::debug('create_default_marketplace returning (int): ' . esc_html($marketplace_id), 'DEBUG');
                return $marketplace_id;
            } else if (is_wp_error($marketplace_id)) {
                 AFFL_LoggerService::log('WP_Error creating initial Digistore24 marketplace: ' . esc_html($marketplace_id->get_error_message()), 'ERROR');
                 AFFL_LoggerService::debug('create_default_marketplace returning (WP_Error): ' . esc_html($marketplace_id->get_error_message()), 'DEBUG');
                 return $marketplace_id;
            } else {
                 AFFL_LoggerService::log('Failed to create initial Digistore24 marketplace (unknown error or ID 0). Result: ' . esc_html(print_r($marketplace_id, true)), 'ERROR');
                 AFFL_LoggerService::debug('create_default_marketplace returning (new WP_Error): Failed to create Digistore24 marketplace.', 'DEBUG');
                 return new WP_Error('marketplace_creation_failed', __('Failed to create Digistore24 marketplace.', 'affililabs'));
            }
        }
        return new WP_Error('unsupported_marketplace_type', __('Unsupported marketplace type for default creation.', 'affililabs'));
    }

    /**
     * Deletes a marketplace post and its associated API key.
     *
     * @param int $marketplace_id The ID of the marketplace post to delete.
     * @return bool True on success, false on failure.
     * @throws Exception If the ID is invalid or deletion fails.
     */
    public static function delete(int $marketplace_id): bool {
        if ($marketplace_id <= 0) {
            throw new Exception('Invalid Marketplace ID provided for deletion.');
        }

        $marketplace_post = get_post($marketplace_id);
        if (!$marketplace_post || $marketplace_post->post_type !== AFFILILABS_CPT_MARKETPLACE) {
             throw new Exception('Marketplace not found or invalid post type for ID: ' . esc_html($marketplace_id));
        }
        $marketplace_slug = $marketplace_post->post_name;

        $deleted_post = wp_delete_post($marketplace_id, true);

        if ($deleted_post === false || $deleted_post === null) {
            AFFL_LoggerService::log("Failed to delete marketplace post with ID: " . esc_html($marketplace_id) . ".", 'ERROR');
            throw new Exception('Error deleting the marketplace post. Please try again later.');
        }

        if ($marketplace_id == AFFILILABS_DIGISTORE_MARKETPLACE_ID) {
            delete_option(AFFILILABS_DIGISTORE_MARKETPLACE_ID);
        }

        self::delete_api_key($marketplace_slug);

        return true;
    }

    /**
     * Saves an encrypted API key for a marketplace using its slug.
     *
     * @param string $api_key The API key to save.
     * @param string $marketplace_slug The sanitized slug of the marketplace.
     * @return bool True on success, false on failure.
     */
    public static function save_api_key(string $api_key, string $marketplace_slug): bool {
        if (empty($marketplace_slug)) {
             AFFL_LoggerService::log('Marketplace slug cannot be empty when saving API key.', 'WARNING');
             return false;
        }


        $option_prefix = 'affl_marketplace_api_key_' . $marketplace_slug;
        $option_name = $option_prefix;

        $salt = AFFLCryptographyService::get_or_generate_salt($option_prefix);
        if (empty($salt)) {
            AFFL_LoggerService::log('Failed to get or generate salt for marketplace API key: ' . esc_html($marketplace_slug), 'ERROR');
            return false;
        }

        $encrypted_api_key = AFFLCryptographyService::encrypt_api_key($api_key, $salt);
        if (empty($encrypted_api_key)) {
            AFFL_LoggerService::log('Failed to encrypt API key for marketplace: ' . esc_html($marketplace_slug), 'ERROR');
            return false;
        }

        return update_option($option_name, $encrypted_api_key, false);
    }

    /**
     * Retrieves and decrypts the API key for a marketplace using its name (converted to slug).
     *
     * @param string $marketplace_name The name of the marketplace.
     * @return string The decrypted API key, or an empty string if not found or decryption fails.
     */
    public static function get_api_key(string $marketplace_name): string {
         if (empty($marketplace_name)) return '';



         $marketplace_slug = sanitize_title($marketplace_name);
         $option_prefix = 'affl_marketplace_api_key_' . $marketplace_slug;
         $option_name = $option_prefix;

         $salt = AFFLCryptographyService::get_or_generate_salt($option_prefix);
         if (empty($salt)) {
             AFFL_LoggerService::debug('Salt missing for marketplace API key decryption: ' . esc_html($marketplace_slug), 'DEBUG');
             return ''; // Cannot decrypt without salt
         }

         $encrypted_api_key = trim(get_option($option_name, ''));
         if (empty($encrypted_api_key)) {
             return ''; // No key saved or effectively empty
         }

         $decrypted_api_key = AFFLCryptographyService::decrypt_api_key($encrypted_api_key, $salt);
         if ($decrypted_api_key === false) {
             AFFL_LoggerService::debug('Failed to decrypt API key for marketplace: ' . esc_html($marketplace_slug));
             update_option($option_name, '', false); // @TODO: Check for viability of short keys
             return '';
         }

         return $decrypted_api_key;
    }

     /**
      * Deletes the API key and salt for a specific marketplace slug.
      *
      * @param string $marketplace_slug The sanitized slug of the marketplace.
      * @return bool Result from delete_option().
      */
     public static function delete_api_key(string $marketplace_slug): bool {
         if (empty($marketplace_slug)) return false;



         $option_prefix = 'affl_marketplace_api_key_' . $marketplace_slug;

         AFFLCryptographyService::delete_salt($option_prefix);

         return delete_option($option_prefix);
     }

    /**
     * Hook callback to update marketplace product count when a product is saved/updated via editor.
     * @deprecated Consider using direct hooks on save_post_{post_type} or transition_post_status.
     *
     * @param int $post_id The ID of the post being saved.
     * @return void
     */
    public static function update_counter_on_editor(int $post_id): void {
        if (get_post_type($post_id) === AFFILILABS_CPT_PRODUCT) {
            $marketplace_id = get_post_meta($post_id, AFFILILABS_META_KEY_MARKETPLACE, true);
            if (!empty($marketplace_id) && is_numeric($marketplace_id)) {
                self::update_product_count(intval($marketplace_id));
            }
        }
    }

    /**
     * Formats a WP_Post object for a marketplace into a structured array.
     *
     * @param WP_Post $marketplace_post The marketplace post object.
     * @return array A structured array with marketplace data (snake_case keys).
     */
    private static function format_marketplace_data(WP_Post $marketplace_post): array {
        $marketplace_id = $marketplace_post->ID;
        $all_meta = get_post_meta($marketplace_id);
        $custom_fields = [];

        foreach ($all_meta as $key => $value) {
            if (strpos($key, '_affl_marketplace_custom_') === 0) {
                $custom_key = str_replace('_affl_marketplace_custom_', '', $key);
                // get_post_meta returns array, take the (potential) first value
                $custom_fields[$custom_key] = isset($value[0]) ? maybe_unserialize($value[0]) : null;
            }
        }

        $category_terms = wp_get_post_terms($marketplace_id, AFFILILABS_TAXONOMY_MARKETPLACE_CATEGORY, ['fields' => 'names']);
        $category_names = !is_wp_error($category_terms) ? $category_terms : [];

        return [
            'id' => $marketplace_id,
            'name' => $marketplace_post->post_title,
            'slug' => $marketplace_post->post_name,
            'url' => $all_meta[AFFILILABS_META_KEY_MARKETPLACE_URL][0] ?? '',
            'image_url' => get_the_post_thumbnail_url($marketplace_id, 'full') ?? '',
            'description' => $marketplace_post->post_content,
            'affiliate_id' => $all_meta[AFFILILABS_META_KEY_MARKETPLACE_AFFILIATE_ID][0] ?? '',
            'last_update' => $all_meta[AFFILILABS_META_KEY_MARKETPLACE_LAST_UPDATE][0] ?? '',
            'total_products' => isset($all_meta[AFFILILABS_META_KEY_MARKETPLACE_TOTAL_PRODUCTS][0]) ? intval($all_meta[AFFILILABS_META_KEY_MARKETPLACE_TOTAL_PRODUCTS][0]) : 0,
            'api_key' => self::get_api_key($marketplace_post->post_name),
            'category_names' => $category_names,
            'custom_fields' => $custom_fields,
            'status_mode' => $all_meta[AFFILILABS_META_KEY_MARKETPLACE_STATUS_MODE][0] ?? 'manual'
        ];
    }

    /**
     * Retrieves a single marketplace by its ID and formats it.
     *
     * @param int $marketplace_id The ID of the marketplace post.
     * @return array|null A formatted array of the marketplace data, or null if not found.
     */
    public static function get_marketplace_by_id(int $marketplace_id): ?array {
        if ($marketplace_id <= 0) {
            return null;
        }


        $marketplace_post = get_post($marketplace_id);

        if (!$marketplace_post || $marketplace_post->post_type !== AFFILILABS_CPT_MARKETPLACE) {
            return null;
        }

        return self::format_marketplace_data($marketplace_post);
    }

    /**
     * Retrieves all marketplace posts, formatted into structured arrays.
     *
     * @param array $query_args Optional WP_Query arguments to override defaults.
     * @return array An array of formatted marketplace data arrays.
     */
    public static function get_all_marketplaces(array $query_args = []): array {
        $default_args = [
            'post_type' => AFFILILABS_CPT_MARKETPLACE,
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ];

        $args = wp_parse_args($query_args, $default_args);

        $marketplaces_query = new WP_Query($args);
        $formatted_marketplaces = [];

        if ($marketplaces_query->have_posts()) {
            while ($marketplaces_query->have_posts()) {
                $marketplaces_query->the_post();
                $post_object = get_post(get_the_ID());
                if ($post_object) {
                     $formatted_marketplaces[] = self::format_marketplace_data($post_object);
                }
            }
            wp_reset_postdata();
        }

        return $formatted_marketplaces;
    }
}
