<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Service for handling encryption and decryption tasks.
 */
class AFFLCryptographyService {

    private const ENCRYPTION_METHOD = 'aes-256-ctr';

    /**
     * Encrypts an API key using OpenSSL.
     *
     * @param string $api_key The API key to encrypt.
     * @param string $salt The salt to use for encryption.
     * @return string The encrypted key prefixed with the IV.
     */
    public static function encrypt_api_key($api_key, $salt) {
        $method = self::ENCRYPTION_METHOD;
        $ivlen = openssl_cipher_iv_length($method);
        $iv = wp_generate_password($ivlen, true, true); 
        if ($iv === false) {
            AFFL_LoggerService::log('Failed to generate IV for encryption.');
            return ''; // Or throw an exception
        }

        $encrypted = openssl_encrypt($api_key, $method, $salt, 0, $iv);
        if ($encrypted === false) {
             AFFL_LoggerService::log('Failed to encrypt data: ' . esc_html(openssl_error_string()));
            return ''; // Or throw an exception
        }
        return $iv . $encrypted;
    }

    /**
     * Decrypts an API key encrypted with encrypt_api_key.
     *
     * @param string $encrypted_api_key The encrypted key (IV prefixed).
     * @param string $salt The salt used during encryption.
     * @return string|false The decrypted API key, or false on failure.
     */
    public static function decrypt_api_key($encrypted_api_key, $salt) {
        $method = self::ENCRYPTION_METHOD;
        $ivlen = openssl_cipher_iv_length($method);
        if (strlen($encrypted_api_key) <= $ivlen) {
             AFFL_LoggerService::debug('Encrypted data is too short to contain IV.');
            return false;
        }
        $iv = substr($encrypted_api_key, 0, $ivlen);
        $encrypted_part = substr($encrypted_api_key, $ivlen);

        $decrypted = openssl_decrypt($encrypted_part, $method, $salt, 0, $iv);
         if ($decrypted === false) {
             AFFL_LoggerService::debug('Failed to decrypt data: ' . esc_html(openssl_error_string()));
            return false;
        }
        return $decrypted;
    }

    /**
     * Retrieves or generates a salt for a given option prefix.
     *
     * @param string $option_prefix The prefix for the salt option name (e.g., 'affl_internal_api_key').
     * @return string The salt.
     */
    public static function get_or_generate_salt($option_prefix) {


        $salt = get_option($option_prefix . '_salt');

        if (!$salt) {
            $salt = wp_generate_password(32, true, true); 
            if ($salt) {
                add_option($option_prefix . '_salt', $salt, '', 'no'); // Autoload 'no' for security options
            } else {
                $salt = strval(time()) ?? '';
            }
        }
        return $salt;
    }

     /**
     * Deletes the salt associated with a given option prefix.
     *
     * @param string $option_prefix The prefix for the salt option name.
     * @return bool True on success, false on failure.
     */
    public static function delete_salt($option_prefix) {

        return delete_option($option_prefix . '_salt');
    }
}
