<?php

if (!defined('ABSPATH')) {
    exit;
}

class AFFLCleanupService {

    /**
     * Handles the cleanup process during uninstallation.
     */
    public static function cleanup() {
        // Check if the user opted for cleanup via settings
        $cleanup_enabled = get_option('affililabs_cleanup_on_uninstall');

        if (!$cleanup_enabled) {
            return;
        }

        self::delete_logs();
        self::delete_custom_post_types();
        self::delete_taxonomies();
        self::delete_transients();
        self::delete_options();
    }

    /**
     * Clears the cleanup flag upon activation.
     */
    public static function clear_cleanup_flag() {
        delete_option('affililabs_cleanup_pending');
    }

    /**
     * Deletes plugin-related taxonomies.
     */
    private static function delete_taxonomies() {
        $taxonomies = [
            'affl_tax_product'
        ];

        foreach ($taxonomies as $taxonomy) {
            $terms = get_terms([
                'taxonomy'   => $taxonomy,
                'hide_empty' => false,
            ]);

            if (is_wp_error($terms)) {
                continue;
            }

            foreach ($terms as $term) {
                wp_delete_term($term->term_id, $taxonomy);
            }
        }
    }

    /**
     * Deletes all plugin-related options.
     */
    private static function delete_options() {
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE 'affililabs_%' OR option_name LIKE 'affl_%'");
    }

    /**
     * Deletes all plugin-related custom post types.
     */
    private static function delete_custom_post_types() {
        $post_types = [
            'affl_marketplace',
            'affl_product',
            'affl_criteria',
            'affl_comp_table',
            'affl_template'
        ];

        foreach ($post_types as $post_type) {
            $posts = get_posts([
                'post_type'   => $post_type,
                'numberposts' => -1,
                'post_status' => 'any'
            ]);

            foreach ($posts as $post) {
                wp_delete_post($post->ID, true);
            }
        }
    }

    /**
     * Deletes plugin logs.
     */
    private static function delete_logs() {
        // Delete internal log directory
        $log_dir = AFFILILABS_PLUGIN_PATH . 'log/';
        self::recursive_rmdir($log_dir);

        // Delete uploads log directory
        if (defined('AFFILILABS_UPLOADS_LOG_PATH')) {
            self::recursive_rmdir(AFFILILABS_UPLOADS_LOG_PATH);
        }
    }

    /**
     * Deletes plugin-related transients.
     */
    private static function delete_transients() {
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_affililabs_%' OR option_name LIKE '_transient_timeout_affililabs_%'");
    }

    /**
     * Recursively removes a directory and its contents.
     *
     * @param string $dir The directory path.
     */
    private static function recursive_rmdir($dir) {
        global $wp_filesystem;

        if (empty($wp_filesystem)) {
            WP_Filesystem();
        }

        if (!$wp_filesystem->is_dir($dir)) {
            return;
        }

        $wp_filesystem->delete($dir, true);
    }
}
