<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

affl_import(AFFILILABS_CPT_PATH);
affl_import(AFFILILABS_SERVICE_CRITERIA_FILE);

class AFFLCategoryService {

    /**
     * Save category meta data, including creating or updating criteria.
     *
     * @param int   $term_id      The ID of the category term.
     * @param array $criteria_data An array of criteria data to save.
     * @return void
     */
    public static function save_category_meta($term_id, $criteria_data) {
        if (empty($term_id) || !is_array($criteria_data)) {
            return;
        }

        foreach ($criteria_data as $criterion_id => $criterion) {
            if (!is_array($criterion)) {
                continue;
            }
            $sanitized_criterion = array(
                'name' => sanitize_text_field($criterion['name']),
                'description' => sanitize_textarea_field($criterion['description']),
                'fields' => array(),
            );

            if (isset($criterion['fields']) && is_array($criterion['fields'])) {
                foreach ($criterion['fields'] as $field_id => $field) {
                    $sanitized_criterion['fields'][$field_id] = array(
                        'name' => sanitize_text_field($field['name']),
                        'type' => sanitize_text_field($field['type']),
                        'unit' => sanitize_text_field($field['unit']),
                    );
                }
            }

            if (strpos($criterion_id, 'new-') === 0 || $criterion_id === 'default') {
                $term = get_term($term_id);
                $criterion_title = ($term && !is_wp_error($term)) ? $term->name : 'Default Criterion';

                if (!empty($sanitized_criterion['name']) && $sanitized_criterion['name'] !== 'Default') {
                    $criterion_title = $sanitized_criterion['name'];
                }

                $new_criterion_data = array(
                    'title' => $criterion_title,
                    'description' => $sanitized_criterion['description'],
                    'fields' => $sanitized_criterion['fields'],
                    'linked_categories' => array($term_id),
                );
                AFFLCriteriaService::save($new_criterion_data);
            } else {
                $existing_criterion = AFFLCriteriaService::get_criterion($criterion_id);
                if ($existing_criterion) {
                    $updated_criterion_data = array(
                        'id' => $criterion_id,
                        'title' => $sanitized_criterion['name'],
                        'description' => $sanitized_criterion['description'],
                        'fields' => $sanitized_criterion['fields'],
                        'linked_categories' => array_unique(array_merge($existing_criterion['linked_categories'], array($term_id))),
                    );
                    AFFLCriteriaService::save($updated_criterion_data);
                }
            }
        }
    }
    /**
     * Delete a category.
     *
     * @param int $category_id The ID of the category to delete.
     * @return bool True on success, false on failure.
     */
    public static function delete(int $category_id): bool {
        try {
            if (empty($category_id)) {
                throw new Exception(__('Category ID is required.', 'affililabs'));
            }

            $category_exists = term_exists($category_id, AFFILILABS_TAXONOMY_PRODUCT_CATEGORY);
            if ($category_exists === 0 || $category_exists === null) {
                throw new Exception(__('Category ID does not exist.', 'affililabs'));
            }

            $res = wp_delete_term($category_id, AFFILILABS_TAXONOMY_PRODUCT_CATEGORY);
            
            if (is_wp_error($res)) {
                 throw new Exception($res->get_error_message());
            }

            if (!$res) {
                 throw new Exception(__('There was an error deleting the category. Please try again.', 'affililabs'));
            }

            return true;
        } catch (Exception $e) {
            AFFL_LoggerService::log(esc_html($e->getMessage()));
            return false;
        }
    }
}
