jQuery(document).ready(function($) {
  const {
      template_id,
      template_type,
      modules,
      assets_url,
      sample_image_url,
      nonce,
      ajax_render_action,
      initial_preview_table_id,
      initial_preview_product_id
  } = affililabs_template_editor;
  const {
      loading_preview,
      preview_error,
      no_tables_found,
      no_products_found,
      select_table_preview,
      select_product_preview,
      load_success_title,
      load_success_message,
      save_success_title,
      save_success_message,
      save_error_title,
      save_error_message,
      delete_row_confirm,
      delete_module_confirm,
      drag_here,
      preview_updated,
      template_name_required,
      add_row,
      add_column,
      delete_row,
      delete_column,
  } = affililabs_template_editor.i18n;

  let template_data = {
      id: $('#template-id').val(),
      name: $('#template-name').val(),
      type: $('#template-type').val(),
      modules: []
  };
  init_drag_and_drop();
  
  $('#save-template-button').off('click').on('click', function() {
    save_template();
  });
  
  init_preview();


  if (template_data.id) {
      load_template_data();
  }
  
  /**
   * Initialize drag and drop functionality
   */
  function init_drag_and_drop() {
      $('.template-editor__module').draggable({
          helper: function() {
              const module_id = $(this).data('module-id');
              const module = affililabs_template_editor.modules.find(m => m.id === module_id);
              if (!module) return '<div>Error</div>';

              const module_template = $('#module-instance-template').html();
              let module_html = module_template.replace(/{module-type}/g, module.id);
              module_html = module_html.replace(/{module-name}/g, module.name); // Title and h4
              module_html = module_html.replace(/{module-icon}/g, module.icon); 
              const $helper = $(module_html).addClass('template-editor__module-instance--helper');
              $helper.find('.delete-module').remove(); 
              return $helper;
          },
          cursor: 'grabbing',
          containment: 'document',
          zIndex: 1000,
          revert: 'invalid'
      });
      
      $('.template-editor__module').on('dblclick', function() {
          const module_id = $(this).data('module-id');
          add_module_to_canvas(module_id);
          let preview_id_for_event;
          if (template_type === 'comparison_table') {
              preview_id_for_event = $('#preview-table-select').val();
          } else {
              preview_id_for_event = $('#preview-product-select').val();
          }
          if (preview_id_for_event) {
              fetch_and_render_preview(preview_id_for_event);
          }

          const $canvas = $('#template-canvas');
          $('html, body').animate({
               scrollTop: $canvas.prop("scrollHeight")
          }, 500);
      });
      
      $('#template-canvas').sortable({
          items: '.template-editor__module-instance',
          handle: '.template-editor__module-handle',
          placeholder: 'template-editor__module-placeholder',
          forcePlaceholderSize: true,
          axis: 'y',
          start: function(e, ui) {
              ui.placeholder.height(ui.item.outerHeight());
          },
          update: function() {
              let preview_id_for_event;
              if (template_type === 'comparison_table') {
                  preview_id_for_event = $('#preview-table-select').val();
              } else {
                  preview_id_for_event = $('#preview-product-select').val();
              }
              if (preview_id_for_event) {
                  fetch_and_render_preview(preview_id_for_event);
              }
          }
      }).droppable({
          accept: '.template-editor__module',
          hoverClass: 'template-editor__canvas--hover',
          drop: function(event, ui) {
              const module_id = ui.draggable.data('module-id');
              add_module_to_canvas(module_id);
          }
      });
  }

    /**
     * Initialize preview functionality (dropdown, refresh button, initial load)
     */
    function init_preview() {
        const $preview_table_select = $('#preview-table-select');
        const $preview_product_select = $('#preview-product-select');
        const $refresh_button = $('#refresh-preview-button');
    
        function handle_preview_change() {
            let selected_id;
            if (template_type === 'comparison_table') {
                selected_id = $preview_table_select.val();
                if (!selected_id) {
                    $('#preview-container').html(`<p>${no_tables_found}</p>`);
                    return;
                }
            } else {
                selected_id = $preview_product_select.val();
                if (!selected_id) {
                    $('#preview-container').html(`<p>${no_products_found}</p>`);
                    return;
                }
            }
            fetch_and_render_preview(selected_id);
        }
    
        $preview_table_select.on('change', handle_preview_change);
        $preview_product_select.on('change', handle_preview_change);
        $refresh_button.on('click', handle_preview_change);
    
        // Do not render preview on initial load if no specific ID is set
        // The dropdowns now start with empty placeholders.
        if (template_type === 'comparison_table') {
             $('#preview-container').html(`<p>${no_tables_found}</p>`);
        } else {
             $('#preview-container').html(`<p>${no_products_found}</p>`);
        }
    }

    function load_template_data() {        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'affililabs_get_template',
                template_id: template_data.id,
                nonce: affililabs_template_editor.nonce
            },
            beforeSend: function() {
              if (window.affililabs_spinner) {
                window.affililabs_spinner.show('Loading template data...');
              } else {
                console.log('Could not find spinner. Showing loading message instead.');
                  $('#template-canvas').html('<p>' + loading_preview + '</p>');
              }
            },
            success: function(response) {
                if (response.data.status === 'ok') {
                    $('#template-canvas').empty();
                    $('#template-name').val(response.data.title);

                    template_data.modules = response.data.rows || [];

                    if (template_data.modules.length > 0) {
                        $('#empty-canvas-message').hide();
                        template_data.modules.forEach(function(module_data) {
                            const module_type = typeof module_data === 'string' ? module_data : (module_data.type || null);
                            if(module_type) {
                                add_module_to_canvas(module_type);
                            }
                        });
                        
                        let preview_id;
                        if (template_type === 'comparison_table') {
                            preview_id = $('#preview-table-select').val() || initial_preview_table_id;
                        } else {
                            preview_id = $('#preview-product-select').val() || initial_preview_product_id;
                        }

                        if (preview_id) {
                            fetch_and_render_preview(preview_id);
                        }
                    }
                } else {
                    console.error('[Response.status.error] Error loading template data:', response.data ? response.data.message : 'Unknown error');
                    if (window.affililabs_modal) {
                        window.affililabs_modal.error(
                            affililabs_template_editor.i18n.error_title,
                            response.data.message || affililabs_template_editor.i18n.error_loading || 'Error loading template'
                        );
                    }
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX Error loading template:', error);
                if (window.affililabs_modal) {
                    window.affililabs_modal.error(
                        save_error_title,
                        save_error_message
                    );
                } else {
                  $('#template-canvas').html('');
                }
            },
            complete: function() {
                if (window.affililabs_spinner) {
                    window.affililabs_spinner.hide();
                }
            }
        });
    }

    function add_module_to_canvas(module_id) {
        const module = affililabs_template_editor.modules.find(m => m.id === module_id);
        if (!module) {
            console.error('Module not found:', module_id);
            return;
        }

        $('#empty-canvas-message').hide();

        const module_template = $('#module-instance-template').html();
        let module_html = module_template.replace(/{module-type}/g, module.id);
        module_html = module_html.replace(/{module-name}/g, module.name);
        module_html = module_html.replace(/{module-icon}/g, module.icon);
        const $module_instance = $(module_html);

        $('#template-canvas').append($module_instance);

        $module_instance.find('.delete-module').on('click', function() {
              $module_instance.remove();
              if ($('#template-canvas .template-editor__module-instance').length === 0) {
                  $('#empty-canvas-message').show();
              }
              let preview_id_for_event;
              if (template_type === 'comparison_table') {
                  preview_id_for_event = $('#preview-table-select').val();
              } else {
                  preview_id_for_event = $('#preview-product-select').val();
              }
              if (preview_id_for_event) {
                  fetch_and_render_preview(preview_id_for_event);
              }
        });

        return $module_instance;
    }

    function collect_template_data() {
        template_data.name = $('#template-name').val().trim();
        template_data.modules = [];
        template_data.applied_table_ids = [];
        console.log('Collecting template data:', template_data);
    
        $('#template-canvas .template-editor__module-instance').each(function() {
            const module_type = $(this).data('module-type');
            if (module_type) {
                template_data.modules.push(module_type);
            }
        });
    
        if (template_type === 'comparison_table') {
            $('.template-editor__table-checkbox:checked').each(function() {
                template_data.applied_table_ids.push($(this).val());
            });
        }
    }

    function fetch_and_render_preview(preview_id) {
        const $preview_container = $('#preview-container');
        
        const current_modules = [];
        $('#template-canvas .template-editor__module-instance').each(function() {
            const module_type = $(this).data('module-type');
            if (module_type) {
                current_modules.push(module_type);
            }
        });
    
        $preview_container.html(`<div class="template-editor__preview-loading"><p>${loading_preview}</p></div>`);
    
        let ajax_data = {
            action: ajax_render_action,
            template_modules: current_modules,
            nonce: nonce,
            template_id: template_id
        };
    
        if (template_type === 'comparison_table') {
            ajax_data.table_id = preview_id;
        } else {
            ajax_data.product_id = preview_id;
        }
    
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: ajax_data,
            success: function(response) {
              if (response.success && response.data && response.data.html && response.data.html.trim().length > 0) {
                  const $preview_container = $('#preview-container');
                  $preview_container.html(response.data.html);


                  adjust_preview_table_width();
              } else {
                  let error_message = preview_error;
                  if (response.data && response.data.message) {
                      error_message += ' ' + response.data.message;
                  } else if (!response.success) {
                       error_message += ' Request failed.';
                  } else {
                       error_message += ' No content returned.';
                  }
                  console.error('Error fetching preview:', response.data ? response.data.message : 'Unknown error or empty content');
                  $('#preview-container').html(`<p class="template-editor__preview-error">${error_message}</p>`);
              }
          },
          error: function(xhr, status, error) {
              console.error('AJAX Error fetching preview:', error);
               $preview_container.html(`<p class="template-editor__preview-error">${preview_error}</p>`);
          },
          complete: function() {
               // @TODO: Hide spinner if used
          }
      });
  }

    const original_fetch_and_render_preview = fetch_and_render_preview;
    fetch_and_render_preview = function(preview_id) {
        const promise = original_fetch_and_render_preview(preview_id);
        adjust_preview_table_width();
        return promise;
    };

    function save_template() {
        const $button = $('#save-template-button');
        const button_text = $button.text();
        if (!$('#template-name').val().trim()) {
            if (window.affililabs_modal) {
                window.affililabs_modal.notice(
                    affililabs_template_editor.i18n.notice_title,
                    affililabs_template_editor.i18n.template_name_required
                );
            } else {
                alert(affililabs_template_editor.i18n.template_name_required);
            }
            return;
        }

        collect_template_data();

        $button.prop('disabled', true).html('<span class="dashicons dashicons-update"></span> Saving...'); //@TODO: i18n

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'affililabs_save_template',
                template_data: template_data,
                nonce: nonce
            },
            beforeSend: function() {
              if (window.affililabs_spinner) {
                window.affililabs_spinner.show('Saving template...');
              }      
            },
            success: function(response) {
                // @TODO: check for status.ok
                if (response.data.template_id) {
                    if (window.affililabs_modal) {
                      window.affililabs_modal.success(save_success_title, save_success_message);
                    }

                    template_data.id = response.data.template_id;
                    $('#template-id').val(response.data.template_id);

                    const new_url = new URL(window.location.href);
                    new_url.searchParams.set('template_id', response.data.template_id);
                    window.history.pushState({}, '', new_url);

                    $('#template-canvas').empty();
                    $('#empty-canvas-message').show();
                    load_template_data(); 

                    $button.prop('disabled', false);
                    setTimeout(() => $button.html(button_text), 1000);


                } else {
                    console.error('Error saving template:', response);
                    if (window.affililabs_modal) {
                        window.affililabs_modal.error(
                            affililabs_template_editor.i18n.save_error_title,
                            response.data.message || affililabs_template_editor.i18n.save_error_message
                        );
                    } else {
                        alert(affililabs_template_editor.i18n.save_error_message);
                    }
                    $button.prop('disabled', false).html('Save Template'); // //@TODO: i18n
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX Error:', error);
                if (window.affililabs_modal) {
                    window.affililabs_modal.error(
                        affililabs_template_editor.i18n.save_error_title,
                        affililabs_template_editor.i18n.save_error_message
                    );
                } else {
                    alert(affililabs_template_editor.i18n.save_error_message);
                }
                $button.prop('disabled', false).html('Save Template');
            },
            complete: function() {
                if (window.affililabs_spinner) {
                    window.affililabs_spinner.hide();
                }
            }
        });
    }

    function adjust_preview_table_width() {
        const $preview_container = $('#preview-container');
        if (!$preview_container.length) {
            return;
        }

        const top_panels_element = $('.template-editor__top-panels');
        let calculated_width;

        if (top_panels_element.length) {
            calculated_width = top_panels_element.outerWidth() - 32;
        } else {
            const wp_admin_menu_element = $('#adminmenu');
            const wp_admin_menu_width = wp_admin_menu_element.length ? wp_admin_menu_element.outerWidth() : 0;

            const affililabs_sidebar_element = $('.affililabs-plugin-dashboard__sidebar');
            const affililabs_sidebar_width = affililabs_sidebar_element.length ? affililabs_sidebar_element.outerWidth() : 0;

            calculated_width = window.innerWidth - wp_admin_menu_width - affililabs_sidebar_width;
        }

        $preview_container.css('max-width', `${calculated_width}px`);
        
        const $comparison_table = $preview_container.find('.affl-comparison-table');
        if ($comparison_table.length) {
            $comparison_table.css('margin', '0 auto');
        }
    }

    adjust_preview_table_width();

    $(window).on('resize', adjust_preview_table_width);
});
