jQuery(document).ready(function ($) {
  const { 
    delete_confirm,
    delete_success,
    delete_error,
    delete_error_unknown,
    no_products_found,
    finding_products,
    find_products_error,
    no_results_found,
    add_to_my_products,
    button_edit_text,
    button_view_text,
    import_success_title,
    import_success_message,
    copied,
    failed_to_copy,
  } = affililabs_products.i18n;

  const {
    find_products_action,
    find_products_nonce,
    save_imported_product_action,
    digistore_affiliate_id,
    digistore_marketplace_id
  } = affililabs_products;

  const $product_search = $("#product-search");
  const $product_sort = $("#product-sort");
  const $product_filter = $("#product-filter");
  let $product_cards = $(".affililabs-plugin__card");
  const $no_results = $(".affililabs-plugin__no-results");

  const $finder_marketplace = $("#product-finder-marketplace");
  const $finder_keyword = $("#product-finder-keyword");
  const $finder_button = $("#product-finder-button");
  const $finder_modal = $("#product-finder-modal");
  const $finder_modal_overlay = $finder_modal.find('.affililabs-plugin__products__modal__overlay');
  const $finder_modal_close_button = $finder_modal.find('.affililabs-plugin__products__modal__close');
  const $finder_results_container = $finder_modal.find("#product-finder-results");

  const $digistore_filters_container = $("#digistore-filters");
  const $finder_currency = $("#product-finder-currency");
  const $finder_language_checkboxes = $("#product-finder-language-checkboxes");
  const $finder_sort = $("#product-finder-sort");

  const $import_modal = $("#product-import-modal");
  const $import_modal_overlay = $import_modal.find('.affililabs-plugin__products__modal__overlay');
  const $import_modal_close_buttons = $import_modal.find('.affililabs-plugin__products__modal__close, [data-close-modal]');
  const $import_save_button = $import_modal.find('#save-imported-product-button');
  const $import_image_upload_button = $import_modal.find('#import-upload-product-image-button');
  const $import_image_preview = $import_modal.find('#import-product-image-preview');

  const original_order = [];
  $product_cards.each(function(index) {
    original_order.push({
      element: this,
      index: index
    });
  });

  $product_sort.val('default');
  
  filter_products();

  $product_search.on("input", filter_products);
  $product_sort.on("change", sort_products);
  $product_filter.on("change", filter_products);

  function filter_products() {
    const search_term = $product_search.val().toLowerCase();
    const filter_value = $product_filter.val();
    let visible_count = 0;

    $product_cards.each(function() {
      const $card = $(this);
      const product_name = $card.data("product-name")?.toLowerCase() || "";
      const price = parseFloat($card.data("product-price")) || 0;
      const rating = parseFloat($card.data("rating")) || 0;
      const marketplace_id = $card.data("marketplace-id") || "";
      const category_ids_attr = $card.data("category-ids") || "";
      
      const category_ids = category_ids_attr ? 
        String(category_ids_attr).split(",").filter(id => id.trim() !== "") : [];
      
      const category = $card.find(".affililabs-plugin__card__category").text().toLowerCase();
      const marketplace = $card.find(".affililabs-plugin__card__marketplace").text().toLowerCase();
      
      let matches_search = true;
      let matches_filter = true;
      
      if (search_term) {
        matches_search = product_name.includes(search_term) || 
                       category.includes(search_term) || 
                       marketplace.includes(search_term) || 
                       price.toString().includes(search_term);
      }
      
      if (filter_value) {
        if (filter_value.startsWith("category-")) {
          const category_id = filter_value.replace("category-", "");
          
          matches_filter = false;
          if (category_ids && category_ids.length) {
            matches_filter = category_ids.some(id => {
              return String(id).trim() === String(category_id).trim();
            });
          }
        } else if (filter_value.startsWith("marketplace-")) {
          const filter_marketplace_id = filter_value.replace("marketplace-", "");
          matches_filter = String(marketplace_id) === String(filter_marketplace_id);
        } else if (filter_value.startsWith("rating-")) {
          const min_rating = parseFloat(filter_value.replace("rating-", ""));
          matches_filter = rating >= min_rating;
        }
      }
      
      const is_visible = matches_search && matches_filter;
      $card.toggle(is_visible);
      
      if (is_visible) {
        visible_count++;
      }
    });
    
    $no_results.toggle(visible_count === 0);
    
    sort_products();
  }

  function sort_products() {
    const sort_value = $product_sort.val();
    const $visible_cards = $product_cards.filter(":visible");
    
    const $card_container = $(".affililabs-plugin__card-container");
    const cards = $visible_cards.get();
    
    if (sort_value === "default") {
      cards.sort(function(a, b) {
        const index_a = original_order.findIndex(item => item.element === a);
        const index_b = original_order.findIndex(item => item.element === b);
        return index_a - index_b;
      });
    } else if (sort_value === "name-asc") {
      cards.sort(function(a, b) {
        const $a = $(a);
        const $b = $(b);
        return $a.data("product-name").localeCompare($b.data("product-name"));
      });
    } else if (sort_value === "name-desc") {
      cards.sort(function(a, b) {
        const $a = $(a);
        const $b = $(b);
        return $b.data("product-name").localeCompare($a.data("product-name"));
      });
    } else if (sort_value === "price-asc") {
      cards.sort(function(a, b) {
        const $a = $(a);
        const $b = $(b);
        return parseFloat($a.data("product-price")) - parseFloat($b.data("product-price"));
      });
    } else if (sort_value === "price-desc") {
      cards.sort(function(a, b) {
        const $a = $(a);
        const $b = $(b);
        return parseFloat($b.data("product-price")) - parseFloat($a.data("product-price"));
      });
    } else if (sort_value === "rating-desc") {
      cards.sort(function(a, b) {
        const $a = $(a);
        const $b = $(b);
        const rating_a = parseFloat($a.data("rating")) || 0;
        const rating_b = parseFloat($b.data("rating")) || 0;
        return rating_b - rating_a;
      });
    } else if (sort_value === "rating-asc") {
      cards.sort(function(a, b) {
        const $a = $(a);
        const $b = $(b);
        const rating_a = parseFloat($a.data("rating")) || 0;
        const rating_b = parseFloat($b.data("rating")) || 0;
        return rating_a - rating_b;
      });
    }
    
    $.each(cards, function(index, card) {
      $card_container.append(card);
    });
  }

  $('.affililabs-plugin__list .affililabs-plugin__card-container').on("click", ".affililabs-plugin__card__copy-id-btn", function (e) {
    e.preventDefault();
    e.stopPropagation();
    
    const product_id = $(this).data("product-id");
    const $btn = $(this);
    
    navigator.clipboard.writeText(product_id).then(function() {
        const original_title = $btn.attr('title');
        $btn.attr('title', copied);
        $btn.css('background-color', '#e0ffe0');
        setTimeout(() => {
            $btn.css('background-color', '');
            $btn.attr('title', original_title);
        }, 1000);
    }, function(err) {
        console.error('Async: Could not copy text: ', err);
        const original_title = $btn.attr('title');
        $btn.attr('title', failed_to_copy);
        setTimeout(() => {
            $btn.attr('title', original_title);
        }, 1000);
    });
  });
  
  $('.affililabs-plugin__list .affililabs-plugin__card-container').on("click", ".affililabs-plugin__delete-btn", function (e) {
    e.preventDefault();
    
    const $button = $(this);
    const product_id = $button.data("product-id");
    const product_name = $(this).data("product-name");

    window.affililabs_modal.delete_confirm(
      delete_confirm + ' "' + product_name + '"',
      function () {
        delete_product(product_id);
      }
    );
  });

  function delete_product(product_id) {
    $.ajax({
      url: ajaxurl,
      type: "POST",
      data: {
        action: "affililabs_delete_product",
        product_id: product_id,
      },
      beforeSend: function() {
        if (window.affililabs_spinner) {
          window.affililabs_spinner.show('Deleting product...');
        }
      },
      success: function (response) {
        console.log('The product has been deleted', response);

        if (response.data.status === 'ok') {          
          window.affililabs_modal.success(
            null,
            delete_success
          );
          
          $(`.affililabs-plugin__delete-btn[data-product-id="${product_id}"]`).closest('.affililabs-plugin__card').fadeOut(300, function() {
            $(this).remove();
            if ($(".affililabs-plugin__card:visible").length === 0) {
              $no_results.show();
            }
          });
        } else {
          window.affililabs_modal.error(
            null,
            delete_error + ": " + (response.data.message || delete_error_unknown)
          );
        }
      },
      error: function (xhr, status, error) {
        console.error(xhr, status, error);
        
        window.affililabs_modal.error(
          null,
          delete_error_unknown
        );
      },
      complete: function() {
        if (window.affililabs_spinner) {
          window.affililabs_spinner.hide();
        }
      }
    });
  }


  $finder_button.on("click", function(e) {
    e.preventDefault();
    handle_find_products();
  });

  $finder_keyword.on("keypress", function(e) {
    const ENTER_KEY_ID = 13;
    if (e.which === ENTER_KEY_ID) {
      e.preventDefault();
      handle_find_products();
    }
  });

  function close_finder_modal() {
    $finder_modal.removeClass('affililabs-plugin__products__modal--show');
  }

  function toggle_digistore_filters() {
      if ($finder_marketplace.val() == digistore_marketplace_id) {
          $digistore_filters_container.slideDown();
      } else {
          $digistore_filters_container.slideUp();
      }
  }

  $finder_marketplace.on('change', toggle_digistore_filters);

  toggle_digistore_filters();


  function handle_find_products() {
    const keyword = $finder_keyword.val().trim();
    const marketplace_id = $finder_marketplace.val();

    $finder_results_container.html('');
    if (window.affililabs_spinner) {
      window.affililabs_spinner.show(finding_products);
    }

    const ajax_data = {
        action: find_products_action,
        nonce: find_products_nonce,
        keyword: keyword,
        marketplace_id: marketplace_id
    };

    const $finder_category_checkboxes = $("#product-finder-category-checkboxes");
  
    if (marketplace_id == digistore_marketplace_id && $digistore_filters_container.is(':visible')) {
        ajax_data.currency = $finder_currency?.val()?.trim();
        ajax_data.language = $finder_language_checkboxes.find('input[type="checkbox"]:checked').map(function() {
            return $(this).val();
        }).get(); // .get() converts jQuery object to array
        ajax_data.marketplace_category_id = $finder_category_checkboxes.find('input[type="checkbox"]:checked').map(function() {
            return $(this).val();
        }).get();
        ajax_data.sort = $finder_sort.val()?.trim();
    }

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: ajax_data,
      success: function(response) {
        if (response.data && response.data.products && response.data.products.length > 0) {
          $finder_modal.addClass('affililabs-plugin__products__modal--show');
          render_product_cards(response.data.products);
        } else if (response?.data?.status === 'ok') {
          $finder_results_container.html('<p class="affililabs-plugin__no-results">' + no_results_found + '</p>');
        } else {
          console.error('[Response.status.error] Error fetching products:', response.data.message);
          const error_message = response.data && response.data.message ? response.data.message : find_products_error;
          $finder_results_container.html('<p class="affililabs-plugin__no-results affililabs-text--error">' + error_message + '</p>');
        }
      },
      error: function() {
        console.error('[Response.error] Error fetching products');
        $finder_results_container.html('<p class="affililabs-plugin__no-results affililabs-text--error">' + find_products_error + '</p>');
      },
      complete: function() {
        if (window.affililabs_spinner) {
          window.affililabs_spinner.hide();
        }
      }
    });
  }

  function render_product_cards(products) {
    let cards_html = '';
    $.each(products, function(index, product) {
      const product_data_for_attr = { ...product };
      let product_json_string_encoded = '';
      try {
        product_json_string_encoded = btoa(encodeURIComponent(JSON.stringify(product_data_for_attr)));
      } catch (e) {
        console.error("Error Base64 encoding product data:", e, product_data_for_attr);
      }

      const price_display = product.price !== null ? `${product.currency || '$'} ${parseFloat(product.price).toFixed(2)}` : 'N/A';
      const earnings_display = product.earnings_per_sale !== null ? `${product.currency || '$'} ${parseFloat(product.earnings_per_sale).toFixed(2)}` : 'N/A';
      const commission_display = product.commission_percent !== null ? `${parseFloat(product.commission_percent).toFixed(1)}%` : 'N/A';
      const cancel_rate_display = product.cancel_rate !== null ? `${parseFloat(product.cancel_rate).toFixed(1)}%` : 'N/A';
      const auto_approve_display = product.accepts_affiliations_automatically ? 'Yes' : 'No';
      const created_at_display = product.created_at ? new Date(product.created_at).toLocaleDateString() : 'N/A';

      cards_html += `
        <div class="affililabs-plugin__card affililabs-plugin__card--finder"
             data-external-id="${product.product_id_external || ''}"
             data-marketplace="${product.marketplace || ''}"
             data-product='${product_json_string_encoded}'>
          <div class="affililabs-plugin__card__image-container">
              <img src="${product.image_url || ''}" alt="${product.title || 'Product Image'}" class="affililabs-plugin__card__image">
          </div>
          <div class="affililabs-plugin__card__content">
              <div class="affililabs-plugin__card__title-container">
                  <strong class="affililabs-plugin__card__title">${product.title || 'N/A'}</strong>
                  <div class="affililabs-plugin__card__marketplace-container">
                    <span class="affililabs-plugin__card__marketplace">${product.vendor_name || 'N/A'}</span>
                    <span class="affililabs-plugin__card__marketplace">${product.category || 'N/A'}</span>
                  </div>
              </div>
              <div class="affililabs-plugin__card__details-grid">
                  <p><strong>Price:</strong> ${price_display}</p>
                  <p><strong>Earnings:</strong> ${earnings_display}</p>
                  <p><strong>Commission:</strong> ${commission_display}</p>
                  <p><strong>Cancel Rate:</strong> ${cancel_rate_display}</p>
                  <p><strong>Created:</strong> ${created_at_display}</p>
                  <p><strong>Auto Approve:</strong> ${auto_approve_display}</p>
              </div>
          </div>
          <div class="affililabs-plugin__card__footer">
              ${product.promo_link_url ? `<a href="${product.promo_link_url}" target="_blank" class="affililabs-plugin__button affililabs-plugin__button--secondary">View Offer</a>` : ''}
              <button class="affililabs-plugin__button add-found-product-button">${add_to_my_products}</button>
          </div>
        </div>
      `;
    });
    $finder_results_container.html(cards_html);

    $finder_results_container.find('.affililabs-plugin__card--finder').each(function(index) {
        const product = products[index];
        if (product && product.description) {
            const $content_area = $(this).find('.affililabs-plugin__card__content');
            const description_html = product.description.replace(/\n/g, '<br>');
            $('<div>')
                .addClass('affililabs-plugin__card__description')
                .html(description_html)
                .appendTo($content_area);
        }
    });
  }

  $finder_results_container.on('click', '.add-found-product-button', function() {
    const $card = $(this).closest('.affililabs-plugin__card--finder');
    const encoded_product_data_string = $card.attr('data-product');
    
    if (encoded_product_data_string) {
      let product_data;
      try {
        const decoded_json_string = decodeURIComponent(atob(encoded_product_data_string));
        product_data = JSON.parse(decoded_json_string);

        open_import_modal(product_data);
      } catch (e) {
        console.error("Error parsing product data for import:", e);
         if (window.affililabs_modal && window.affililabs_modal.error) {
            window.affililabs_modal.error('Import Error', 'Could not read product data.');
         }
      }
    } else {
       console.error("Could not find product data on card.");
        if (window.affililabs_modal && window.affililabs_modal.error) {
            window.affililabs_modal.error('Import Error', 'Could not find product data.');
         }
    }
  });

  function open_import_modal(product_data) {
    $import_modal.find('input[type="text"], input[type="hidden"], textarea').val('');
    $import_modal.find('select').prop('selectedIndex', 0);
    $import_modal.find('#import-product-image-id').val('');
    $import_modal.find('#import-product-price-custom-currency').hide();

    $import_modal.find('#import-product-name').val(product_data.title || '');
    $import_modal.find('#import-product-price').val(product_data.price !== null ? parseFloat(product_data.price).toFixed(2) : '');
    const affiliate_id = digistore_affiliate_id || 'CAMPAIGNKEY';
    let affiliate_link = product_data.promo_link_url || '';

    if (affiliate_link.includes('/AFFILIATE/')) {
        affiliate_link = affiliate_link.replace('/AFFILIATE/', `/${affiliate_id}/`);
    } else if (affiliate_link.includes('/AFFILIATE')) {
        affiliate_link = affiliate_link.replace('/AFFILIATE', `/${affiliate_id}`);
    } else if (affiliate_link.includes('AFFILIATE')) {
        affiliate_link = affiliate_link.replace('AFFILIATE', affiliate_id);
    } else if (affiliate_link.includes('CAMPAIGNKEY')) {
        affiliate_link = affiliate_link.replace('CAMPAIGNKEY', affiliate_id);
    }

    $import_modal.find('#import-product-affiliate-link').val(affiliate_link);
    $import_modal.find('#import-product-description').val(product_data.description || '');

    $import_modal.find('#import-product-external-id').val(product_data.product_id_external || '');
    $import_modal.find('#import-product-marketplace-source').val(product_data.marketplace || '');
    $import_modal.find('#import-product-image-url-source').val(product_data.image_url || '');

    $import_image_preview.attr('src', product_data.image_url || '');

    const currency_code_to_symbol = {
        'USD': '$',
        'EUR': '€',
        'GBP': '£',
        'CNY': '¥',
        'INR': '₹',
        'PLN': 'PLN',
        'CHF': 'CHF'
    };
    const api_currency_code = product_data.currency || 'USD';
    const dropdown_currency_symbol = currency_code_to_symbol[api_currency_code] || api_currency_code;

    const $currency_select = $import_modal.find('#import-product-price-currency');
    const $custom_currency_input = $import_modal.find('#import-product-price-custom-currency');
    
    if ($currency_select.find(`option[value="${dropdown_currency_symbol}"]`).length > 0) {
        $currency_select.val(dropdown_currency_symbol);
        $custom_currency_input.hide();
    } else {
        $currency_select.val('custom');
        $custom_currency_input.val(api_currency_code).show();
    }

    const source_marketplace_identifier = product_data.marketplace || '';
    const $marketplace_select = $import_modal.find('#import-product-marketplace');
    if (source_marketplace_identifier === 'digistore24' && digistore_marketplace_id) {
        $marketplace_select.val(digistore_marketplace_id);
    } else {
        $marketplace_select.val('');
    }

    $import_modal.addClass('affililabs-plugin__products__modal--show');
  }
  
  function close_import_modal() {
      $import_modal.removeClass('affililabs-plugin__products__modal--show');
  }

  $import_save_button.on('click', function() {
      const product_data = gather_import_modal_data();
      
      save_imported_product(product_data);
  });

  $import_image_upload_button.on('click', function(e) {
      e.preventDefault();
      open_media_uploader_for_import();
  });
  
  function gather_import_modal_data() {
      const product_data = {};
      product_data.id = null;
      product_data.name = $import_modal.find('#import-product-name').val().trim();
      product_data.price = $import_modal.find('#import-product-price').val().trim();
      
      const $currency_select = $import_modal.find('#import-product-price-currency');
      if ($currency_select.val() === 'custom') {
          product_data.price_currency = $import_modal.find('#import-product-price-custom-currency').val().trim() || '$';
      } else {
          product_data.price_currency = $currency_select.val().trim();
      }
      
      product_data.affiliate_link = $import_modal.find('#import-product-affiliate-link').val().trim();
      product_data.category = $import_modal.find('#import-product-category').val().trim();
      product_data.description = $import_modal.find('#import-product-description').val().trim();
      
      product_data.image_id = $import_modal.find('#import-product-image-id').val() || null;
      product_data.image_url = product_data.image_id ? null : $import_modal.find('#import-product-image-url-source').val();

      product_data.status = 'active';

      product_data.marketplace_id = $import_modal.find('#import-product-marketplace').val().trim();

      return product_data;
  }

  function save_imported_product(product_data) {

      $.ajax({
          url: ajaxurl,
          type: 'POST',
          data: {
              'action': save_imported_product_action,
              'product_data': JSON.stringify(product_data),
              'nonce': find_products_nonce
          },
          beforeSend: function() {
              $import_save_button.prop('disabled', true);
              if (window.affililabs_spinner) {
                  window.affililabs_spinner.show('Saving product...'); //@TODO: i18n
              }
          },
          success: function(response) {
              if (response.success && response.data.status === 'ok' && response.data.new_product_data) {
                  close_import_modal();
                  if (window.affililabs_modal && window.affililabs_modal.success) {
                      window.affililabs_modal.success(import_success_title, import_success_message);
                  }
                  
                  const new_product_data = response.data.new_product_data;
                  const new_card_html = render_single_product_card(new_product_data);
                  
                  if (new_card_html) {
                      $no_results.hide();
                      $('.affililabs-plugin__list .affililabs-plugin__card-container').prepend(new_card_html);
                      
                      $product_cards = $('.affililabs-plugin__list .affililabs-plugin__card');
                      
                      const $new_card = $('.affililabs-plugin__list .affililabs-plugin__card-container').find(`[data-product-id="${new_product_data.id}"]`);
                      if ($new_card.length) {
                           original_order.unshift({ element: $new_card[0], index: -1 });
                      }
                      
                      filter_products();
                  }
                  
              } else if (response.success && response?.data?.status === 'ok') {
                    close_import_modal();
                    if (window.affililabs_modal && window.affililabs_modal.success) {
                      window.affililabs_modal.success(import_success_title, import_success_message);
                    }
              } else {
                  console.error('[Response.data.status.error] Error saving imported product:', response.data);
                  const error_message = response.data && response.data.message ? response.data.message : 'Unknown error saving product.';
                   if (window.affililabs_modal && window.affililabs_modal.error) {
                      window.affililabs_modal.error('Save Error', error_message);
                   }
              }
          },
          error: function(xhr, status, error) {
              console.error('[Response.error] Error saving imported product:', xhr, status, error);
               if (window.affililabs_modal && window.affililabs_modal.error) {
                  window.affililabs_modal.error('Save Error', 'Could not connect to server.');
               }
          },
          complete: function() {
              $import_save_button.prop('disabled', false);
              if (window.affililabs_spinner) {
                  window.affililabs_spinner.hide();
              }
          }
      });
  }

  let import_file_frame;
  function open_media_uploader_for_import() {
       if (import_file_frame) {
          import_file_frame.open();
          return;
      }

      import_file_frame = wp.media({
          title: 'Select or Upload Product Image', //@TODO: i18n
          button: {
              text: 'Use this image' //@TODO: i18n
          },
          multiple: false
      });

      import_file_frame.on('select', function() {
          const attachment = import_file_frame.state().get('selection').first().toJSON();
          $import_image_preview.attr('src', attachment.url);
          $import_modal.find('#import-product-image-id').val(attachment.id);
          console.log('Selected import image obj', attachment);
      });

      import_file_frame.open();
  }
  
  $import_modal.find('#import-product-price-currency').on('change', function() {
      const $custom_input = $import_modal.find('#import-product-price-custom-currency');
      if ($(this).val() === 'custom') {
          $custom_input.show();
      } else {
          $custom_input.hide().val('');
      }
  });

  /**
   * Renders the HTML for a single product card using product data object.
   * @param {object} product_data - The product data object from AJAX response.
   * @returns {string} HTML string for the product card.
   */
  function render_single_product_card(product_data) {
      if (!product_data || !product_data.id) {
          console.error("Invalid product data received for rendering card:", product_data);
          return '';
      }
      
      const {
        trash_icon_url,
        star_full_url,
        star_half_url,
        star_empty_url,
        default_image_url,
        copy_icon_url
      } = affililabs_products.assets;

      const product_id = product_data.id;
      const product_name = product_data.name || 'N/A';
      const product_price_num = product_data.price !== null && !isNaN(parseFloat(product_data.price)) ? parseFloat(product_data.price) : null;
      const product_price_display = product_price_num !== null ? product_price_num.toFixed(2) : 'N/A';
      const price_currency = product_data.price_currency || '$';
      const rating_num = product_data.rating !== null && !isNaN(parseFloat(product_data.rating)) ? parseFloat(product_data.rating) : 0;
      const marketplace_id = product_data.marketplace_id || '';
      const marketplace_name = product_data.marketplace_name || '';
      const category_ids = product_data.category_ids || [];
      const category_ids_string = category_ids.join(',');
      const category_name = product_data.category_names && product_data.category_names.length > 0 ? product_data.category_names[0] : '';
      const image_url = product_data.image_url || '';
      const edit_url = `admin.php?page=affililabs-product-edit&product_id=${product_id}&nonce=${find_products_nonce}`;
      const view_url = product_data.permalink || '#';

      const display_image_url = image_url || default_image_url;

      let rating_html = '';
      if (rating_num > 0) {
          let r = rating_num;
          if (r > 5) r = 5;
          if (r < 0) r = 0;
          const full_stars = Math.floor(r);
          const half_stars = (r - full_stars) >= 0.1 ? 1 : 0;
          const empty_stars = 5 - full_stars - half_stars;
          
          rating_html += '<div class="affililabs-plugin__card__rating">';
          rating_html += '<span class="affililabs-plugin__card__rating-stars">';
          for (let i = 0; i < full_stars; i++) { rating_html += `<img style="width: 18px; height: auto;" src="${star_full_url}" alt="star">`; }
          if (half_stars) { rating_html += `<img style="width: 18px; height: auto;" src="${star_half_url}" alt="star">`; }
          for (let i = 0; i < empty_stars; i++) { rating_html += `<img style="width: 18px; height: auto;" src="${star_empty_url}" alt="star">`; }
          rating_html += '</span>';
          rating_html += `<span>(${rating_num.toFixed(1)})</span>`;
          rating_html += '</div>';
      }

      //@TODO: encode to avoid escaping issues
      const escaped_product_name = product_name.replace(/'/g, '\'').replace(/"/g, '"');
      const card_html = `
          <div class="affililabs-plugin__card"
               data-product-id="${product_id}"
               data-product-name="${escaped_product_name}"
               data-product-price="${product_price_num !== null ? product_price_num : ''}"
               data-rating="${rating_num}"
               data-marketplace-id="${marketplace_id}"
               data-category-ids="${category_ids_string}">
              <img src="${copy_icon_url}" class="affililabs-plugin__card__copy-id-btn" data-product-id="${product_id}" title="Copy Product ID" />
              <span class="affililabs-plugin__card__id-badge">ID: ${product_id}</span>
              <div class="affililabs-plugin__card__image-container">
                  <img src="${display_image_url}" alt="${escaped_product_name}" class="affililabs-plugin__card__image">
              </div>
              <div class="affililabs-plugin__card__content">
                  <div class="affililabs-plugin__card__title-container">
                      <strong class="affililabs-plugin__card__title">${product_name}</strong>
                      ${marketplace_name ? `<span class="affililabs-plugin__card__marketplace">${marketplace_name}</span>` : ''}
                  </div>
                  <p class="affililabs-plugin__card__price">
                      ${price_currency}${product_price_display}
                  </p>
                  ${category_name ? `
                  <div class="affililabs-plugin__card__category-container">
                      <strong>Category:</strong>
                      <span class="affililabs-plugin__card__category">${category_name}</span>
                  </div>` : ''}
                  ${rating_html}
              </div>
              <div class="affililabs-plugin__card__footer">
                  <a href="${edit_url}" class="affililabs-plugin__edit-btn">${button_edit_text}</a>
                  <a href="${view_url}" class="affililabs-plugin__edit-btn affililabs-plugin__view-btn" target="_blank">${button_view_text}</a>
                  <img src="${trash_icon_url}" alt="trash-icon" class="affililabs-plugin__trash-icon affililabs-plugin__delete-btn" data-product-id="${product_id}" data-product-name="${escaped_product_name}" />
              </div>
          </div>
      `;

      return card_html;
  }

  $finder_modal_close_button.on('click', close_finder_modal);
  $finder_modal_overlay.on('click', close_finder_modal);
  $import_modal_close_buttons.on('click', close_import_modal);
  $import_modal_overlay.on('click', close_import_modal);

  $(document).on('keydown', function(e) {
      if (e.key === 'Escape') {
          if ($import_modal.hasClass('affililabs-plugin__products__modal--show')) {
              close_import_modal();
          } else if ($finder_modal.hasClass('affililabs-plugin__products__modal--show')) {
              close_finder_modal();
          }
      }
  });


});
