jQuery(document).ready(function($) {
  
  /**
   * Escapes HTML characters in a string to prevent rendering.
   * @param {string} text The string to escape.
   * @returns {string} The escaped string.
   */
  function escape_html(text) {
    const escape_map = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&apos;',
      '¢': '&cent;',
      '£': '&pound;',
      '¥': '&yen;',
      '€': '&euro;',
      '©': '&copy;',
      '®': '&reg;',
      '™': '&trade;'
    };
    if (typeof text !== 'string') {
      return '';
    }
    const unescaped_text = text.replace(/\\'/g, "'");
    return unescaped_text.replace(/[&<>"'¢£¥€©®™]/g, (m) => escape_map[m] || m);
  }
  
  let file_frame;
  const {
      file_upload_button_title,
      file_upload_button_text,
      save_success_title,
      save_success,
      save_error_title,
      save_error,
      save_error_unknown,
      price_format_error,
      price_negative_error,
      save_error_unknown_message,
      apply_suggestion_button_message,
      deny_suggestion_button_message,
      rating_error,
      no_criteria_for_category,
      loading_criteria_message,
      error_message,
      error_unknown_message,
      add_new_category_title,
      add_new_marketplace_title,
      name_label,
      slug_label,
      description_label,
      save_button,
      saving_category_message,
      category_created_message,
      saving_marketplace_message,
      marketplace_created_message,
      name_slug_required_error,
      name_required_error,
      rating_product_with_ai,
      ai_rating_error_title,
      deep_review_product_with_ai,
      deep_review_error_title,
      edit_template,
      create_new_template,
      no_criteria_found_message,
      filling_criteria_with_ai_message,
      fill_criteria_with_ai,
      unknown_error,
      no_content_received_error,
      deep_review_title,
      deep_review_initializing,
      deep_review_writing,
      deep_review_completed,
      close,
      generate_review
  } = affililabs_products_edit.i18n;
  const { nonce, list_separator, ai_config } = affililabs_products_edit;
  
  // Socket Connection variables
  let socket = null;
  let currentRequestId = null;
  let connectionRetryCount = 0;
  let maxSocketTimeout = null;

  const connect_socket_if_needed = function() {
      if (socket) {
          if (!socket.connected) {
              console.log('Socket exists but disconnected, reconnecting...');
              socket.connect();
          }
          return;
      }

      if (ai_config && ai_config.socket_url) {
            console.log('Connecting to socket...', ai_config.socket_url);
            socket = io(ai_config.socket_url, {
                transports: ['websocket', 'polling'],
                reconnection: true,
                reconnectionAttempts: 3
            });

            socket.on('connect', function() {
                console.log('Socket connected:', socket.id);
                maxSocketTimeout = setTimeout(function() {
                    console.log('Socket disconnected after timeout:');
                    socket.disconnect();
                }, 30 * 60 * 1000);
                
                // Room-based reconnection logic
                if (currentRequestId) {
                    console.log('Reconnected, joining room for request:', currentRequestId);
                    socket.emit('ai:join', { requestId: currentRequestId });
                }
            });
            
            socket.on('connect_error', function(error) {
                console.error('Socket connection error:', error);
                
                connectionRetryCount++;
                const $modal = $('#affililabs-modal');
                
                if ($modal.hasClass('affililabs-modal--show') && currentRequestId) {
                     if (connectionRetryCount > 3) {
                         connectionRetryCount = 0;
                         clearTimeout(maxSocketTimeout);
                         socket.disconnect();
                         $modal.find('#ai-deep-review-status-label')
                             .text('Connection failed after 3 attempts. Please check your internet or try again later.')
                             .css('color', 'red');
                     } else {
                         $modal.find('#ai-deep-review-status-label')
                             .text(`Connection error. Retrying (${connectionRetryCount}/3)...`)
                             .css('color', 'orange');
                     }
                }
            });

            socket.on('reconnect_failed', function() {
                console.error('Socket reconnection failed (event).');
                connectionRetryCount = 0;
                clearTimeout(maxSocketTimeout);
                socket.disconnect();
                const $modal = $('#affililabs-modal');
                if ($modal.hasClass('affililabs-modal--show') && currentRequestId) {
                     $modal.find('#ai-deep-review-status-label').text('Connection failed after 3 attempts. Please check your internet or try again later.').css('color', 'red');
                }
            });

            socket.on('ai:stream', function(payload) {
                if (payload.requestId !== currentRequestId) return;

                const $modal = $('#affililabs-modal');
                const $statusLabel = $modal.find('#ai-deep-review-status-label');
                const $streamLog = $modal.find('#ai-deep-review-stream-log');

                if (payload.type === 'thinking') {
                    $statusLabel.text('Thinking: ' + payload.content);
                } else if (payload.type === 'fixing') {
                    $statusLabel.text('Refining output...');
                    $streamLog.append(document.createTextNode(payload.content));
                } else {
                    $statusLabel.text(deep_review_writing); 
                    // Append as text node to show raw HTML code
                    $streamLog.append(document.createTextNode(payload.content));
                }
                
                // Auto-scroll
                if ($streamLog.length) {
                    $streamLog.scrollTop($streamLog[0].scrollHeight);
                }
            });

            socket.on('ai:update', function(payload) {
                if (payload.requestId !== currentRequestId) return;
                const $modal = $('#affililabs-modal');
                const $statusLabel = $modal.find('#ai-deep-review-status-label');
                const $streamLog = $modal.find('#ai-deep-review-stream-log');

                // Special handling for resume
                if (payload.message === 'Resumed session') {
                    $streamLog.append(document.createTextNode('\n[System: Connection established...]\n'));
                    if ($streamLog.length) {
                        $streamLog.scrollTop($streamLog[0].scrollHeight);
                    }
                    $statusLabel.text('Connection established').css('color', '#1e3b8a');
                } else {
                    $statusLabel.text(payload.message).css('color', '#1e3b8a');
                }
            });

            socket.on('ai:result', function(payload) {
                if (payload.requestId !== currentRequestId) return;

                const $statusLabel = $('#ai-deep-review-status-label');
                const $streamLog = $('#ai-deep-review-stream-log');

                if (payload.status === 'success') {
                    $statusLabel.text(deep_review_completed).css('color', 'green');
                    
                    // Prefer the structured data from the server if available
                    let reviewContent = '';
                    
                    if (payload.data) {
                        if (typeof payload.data === 'string') {
                            reviewContent = payload.data;
                        } else if (payload.data.review) {
                            reviewContent = payload.data.review;
                        } else {
                             // Fallback to what we captured in the stream (using .text() gets raw text content)
                             reviewContent = $streamLog.text();
                        }
                    } else {
                         reviewContent = $streamLog.text();
                    }

                    // Attempt cleanup of potential Markdown wrappers (even if raw HTML requested, models sometimes talk)
                    if (reviewContent && typeof reviewContent === 'string') {
                         reviewContent = reviewContent
                             .replace(/^```html\s*/i, '')
                             .replace(/^```\s*/, '')
                             .replace(/\s*```$/, '');
                    }

                    if (reviewContent && typeof reviewContent === 'string' && reviewContent.trim() !== '') {
                         // Update main description
                         $('#product-description').val(reviewContent);
                         
                         // Visual feedback in modal (replace content with final clean version)
                         $streamLog.text(reviewContent);

                         // Close modal after delay
                         setTimeout(function() {
                             window.affililabs_modal.close();
                         }, 2000);
                    } else {
                        $statusLabel.text(no_content_received_error || 'No content received from AI.').css('color', 'red');
                        console.warn('AI returned success but content was empty.');
                    }
                } else {
                    $statusLabel.text('Error: ' + payload.message).css('color', 'red');
                }

                clearTimeout(maxSocketTimeout);
            });
      } else {
            console.error('AI config or socket URL missing.', ai_config);
      }
  };

  const toggle_custom_currency = function() {
    const select_value = $('#product-price-currency').val();
    if (select_value === 'custom') {
      $('#product-price-custom-currency').show();
    } else {
      $('#product-price-custom-currency').hide();
    }
  };

  const render_criteria = function(criteria) {
    const container = $('#affililabs-criteria-container');
    container.empty();

    if (criteria.length === 0) {
        container.append(`<p>${no_criteria_for_category}</p>`);
        return;
    }

    const ai_button_html = `
        <div class="affililabs-plugin-product-edit__ai-criteria-button-container" style="display: flex; align-items: center; margin-bottom: 20px;">
            <button class="affililabs-plugin__button affililabs-plugin__product-edit-ai-criteria-button">${fill_criteria_with_ai}</button>
            <span class="affililabs-tooltip"
                    aria-label="The &quot;Fill Criteria with AI&quot; feature provides a more in-depth analysis and higher quality rating compared to the general &quot;Fill data with AI&quot; feature."> 
                <img
                    src="${affililabs_products_edit.ai_config.question_mark_svg || ''}"
                    alt="Help"
                    width="16"
                    style="margin-left: 10px;"
                >
            </span>
        </div>
    `;
    container.append(ai_button_html);

    criteria.forEach(function(criterion) {
        let fields_html = '';
        if (criterion.fields && criterion.fields.length > 0) {
            criterion.fields.forEach(function(field) {
                const input_id = `criterion-${criterion.id}-field-def-${field.definition_id}`;
                let field_html = `
                    <div class="affililabs-plugin-product-edit__config__field">
                        <label for="${input_id}">${field.name}</label>`;
                
                if (field.type === 'text') {
                    field_html += `
                        <input
                            type="text"
                            id="${input_id}"
                            name="criterion_value_${criterion.id}_${field.name}"
                            value=""
                            data-field-name="${field.name}"
                            data-definition-id="${field.definition_id}"
                            data-field-type="${field.type}"
                            data-criterion-id="${criterion.id}"
                        >`;
                } else if (field.type === 'number') {
                    field_html += `
                        <div class="number-field-container">
                            <input
                                type="text"
                                id="${input_id}"
                                name="criterion_value_${criterion.id}_${field.name}"
                                value=""
                                data-field-name="${field.name}"
                                data-definition-id="${field.definition_id}"
                                data-field-type="${field.type}"
                                data-field-unit="${field.unit || ''}"
                                data-criterion-id="${criterion.id}"
                            >
                            ${field.unit ? `<span class="field-unit">${field.unit}</span>` : ''}
                        </div>`;
                } else if (field.type === 'boolean') {
                    field_html += `
                        <select
                            id="${input_id}"
                            name="criterion_value_${criterion.id}_${field.name}"
                            data-field-name="${field.name}"
                            data-definition-id="${field.definition_id}"
                            data-field-type="${field.type}"
                            data-criterion-id="${criterion.id}"
                        >
                            <option value="">None</option>
                            <option value="yes">Yes</option>
                            <option value="no">No</option>
                        </select>`;
                }

                field_html += `</div>`;
                fields_html += field_html;
            });
        }

        const criterion_html = `
            <div class="affililabs-plugin-product-edit__config__criterion" data-criterion-id="${criterion.id}">
                <h3>${criterion.title}</h3>
                <p class="description">${criterion.description}</p>
                ${fields_html}
            </div>`;
        container.append(criterion_html);
    });

    container.append('<button id="affililabs-plugin__product-edit-save-button__criteria" class="affililabs-plugin__button affililabs-plugin__product-edit-save-button" data-save-tab="criteria">Save Settings</button>');
  }


  const fetch_criteria = function(category_slug) {
      $.ajax({
          url: ajaxurl,
          type: 'POST',
          data: {
              action: 'affililabs_get_criteria_by_category',
              category_slug: category_slug,
              nonce: nonce
          },
          beforeSend: function() {
              if (window.affililabs_spinner) {
                  window.affililabs_spinner.show(loading_criteria_message);
              }
          },
          success: function(response) {
              if (response.success) {
                  render_criteria(response.data.criteria);
              } else {
                  window.affililabs_modal.error(error_message, response.data.message);
              }
          },
          error: function(xhr) {
              window.affililabs_modal.error(error_message, xhr.responseJSON.data.message || error_unknown_message);
          },
          complete: function() {
              if (window.affililabs_spinner) {
                  window.affililabs_spinner.hide();
              }
          }
      });
  };

  const save_product = function() {
    if ($(this).prop('disabled')) {
      return;
    }

    const $save_button = $(this);
    $save_button.prop('disabled', true);

    const price_val = $('#product-price').val().trim();
    const rating_val = $('#product-rating').val().trim();
    let is_valid = true;
    const error_messages = [];

    const price_pattern = /^(?:\d{1,3}(?:[.,]\d{3})*|\d+)(?:[.,]\d+)?$/;
    const rating_pattern = /^(?:[0-4](?:[.,]\d)?|5(?:[.,]0)?)$/;

    if (price_val !== '') {
        if (!price_pattern.test(price_val)) {
            is_valid = false;
            error_messages.push(price_format_error);
            $('#product-price').addClass('affl-input-error');
        } else {
            const price = parseFloat(price_val.replace(',', '.'));
            if (isNaN(price) || price < 0) {
                is_valid = false;
                error_messages.push(price_negative_error);
                $('#product-price').addClass('affl-input-error');
            } else {
                $('#product-price').removeClass('affl-input-error');
            }
        }
    } else {
        $('#product-price').removeClass('affl-input-error');
    }

    if (rating_val !== '') {
        const rating = parseFloat(rating_val.replace(',', '.'));
        if (!rating_pattern.test(rating_val) || isNaN(rating) || rating < 0 || rating > 5) {
            is_valid = false;
            error_messages.push(rating_error);
            $('#product-rating').addClass('affl-input-error');
        } else {
            $('#product-rating').removeClass('affl-input-error');
        }
    } else {
        $('#product-rating').removeClass('affl-input-error');
    }

    if (!is_valid) {
        window.affililabs_modal.error(
            save_error_title,
            error_messages.join('\n')
        );
        $save_button.prop('disabled', false);
        return;
    }
    
    const product_data = {};

    product_data.id = $('#product-id').val().trim();
    product_data.name = $('#product-name').val().trim();
    product_data.price = $('#product-price').val().trim();
    
    if ($('#product-price-currency').val() === 'custom') {
        product_data.price_currency = $('#product-price-custom-currency').val().trim() || '$';
    } else {
        product_data.price_currency = $('#product-price-currency').val().trim();
    }
    
    product_data.affiliate_link = $('#product-affiliate-link')?.val()?.trim() ?? null;
    product_data.rating = $('#product-rating')?.val()?.trim() ?? null;
    product_data.image_url = $('#product-image-preview').attr('src');
    product_data.image_id = $('#product-image-preview').data('attachment_id');
    product_data.category = $('#product-category')?.val()?.trim() ?? null;
    product_data.marketplace_id = $('#product-marketplace')?.val()?.trim() ?? null;
    product_data.description = $('#product-description')?.val()?.trim() ?? null;
    product_data.short_features = $('#product-short-features')?.val()?.trim() ?? null;
    product_data.pros = $('#product-pros')?.val()?.trim() ?? null;
    product_data.cons = $('#product-cons')?.val()?.trim() ?? null;
    product_data.rating_rationale = $('#product-rating-rationale')?.val()?.trim() ?? null;
    product_data.rating_sources = $('#product-rating-sources')?.val()?.trim() ?? null;
    product_data.criteria_sources = $('#product-criteria-sources')?.val()?.trim() ?? null;
    product_data.applied_template_id = $('#product-template').val()?.trim() ?? null;
    
    const criterion_values = {};
    
    $('input[name^="criterion_value_"], select[name^="criterion_value_"]').each(function() {
        const name = $(this).attr('name');
        const value = $(this).val()?.trim() ?? '';
        const criterion_id = $(this).data('criterion-id');
        const field_name = $(this).data('field-name');
        const definition_id = $(this).data('definition-id');
        const field_type = $(this).data('field-type') || '';
        const field_unit = $(this).data('field-unit') || '';
        
        console.log('Processing form field:', {
            name: name,
            criterion_id: criterion_id,
            field_name: field_name,
            definition_id: definition_id,
            value: value
        });
        
        if (!criterion_id || !definition_id) {
            console.warn('Field missing required data attributes (criterion_id or definition_id):', name);
            return;
        }
        
        if (!criterion_values[criterion_id]) {
            criterion_values[criterion_id] = [];
        }
        
        criterion_values[criterion_id].push({
            name: field_name,
            ['definition_id']: definition_id,
            type: field_type,
            unit: field_unit,
            value: value
        });
    });
    
    console.log('Final criterion_values structure:', criterion_values);
    product_data.criterion_values = criterion_values;
    
    product_data.custom_fields = {};
    $('.affililabs-plugin-product-edit__config__field').each(function() {
        const $field_name = $(this).find('input[name$="-name"]');
        if ($field_name.length) {
            const field_name = $field_name.val().trim();
            const custom_field_key = $field_name.attr('name').replace('-name', '');
            const $field_value = $(this).find('input[name="' + custom_field_key + '-value"]');
            const field_value = $field_value.val();
            
            if (field_name && field_value) {
                product_data.custom_fields[custom_field_key] = field_value;
            }
        }
    });
    
    console.log('product data', product_data);
      
    $.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
            'action': 'affililabs_save_product',
            'product_data': JSON.stringify(product_data),
            'nonce': nonce
        },
        beforeSend: function() {
            if (window.affililabs_spinner) {
                window.affililabs_spinner.show('Saving product...');
            }
        },
        success: function(response) {
            if (response.data.status === 'ok') {
                if (!product_data.id) {
                    $('#product-id').val(response.data.product_id);

                    const new_url = new URL(window.location.href);
                    new_url.searchParams.set('product_id', response.data.product_id);
                    window.history.pushState({}, '', new_url);
                }
                
                window.affililabs_modal.success(
                    save_success_title,
                    save_success
                );
            } else {
                console.error('[Response.data.status.error] There was an error saving product:', response.data);
                window.affililabs_modal.error(
                    save_error_title,
                    save_error + ": " + (response.data ? response.data.message : save_error_unknown)
                );
            }
        },
        error: function(xhr, status, error) {
            console.error('[Response.error] There was an error saving product:', xhr.responseJSON, status, error);
            window.affililabs_modal.error(
            save_error_title,
            xhr.responseJSON.data.message || save_error_unknown
            );
        },
        complete: function() {
            $('.affililabs-plugin__product-edit-save-button').prop('disabled', false);
            if (window.affililabs_spinner) {
                window.affililabs_spinner.hide();
            }
        }
    });
      
  }

  $('#product-category').on('change', function() {
      const category_slug = $(this).val();
      fetch_criteria(category_slug);
  });

  $('#product-price-currency').on('change', toggle_custom_currency);

  $('#product-template').on('change', function() {
    const template_id = $(this).val();
    const edit_button = $('#edit-template-button');
    const base_url = new URL(window.location.origin + window.location.pathname);
    base_url.searchParams.set('page', 'affililabs-template-editor');
    base_url.searchParams.set('nonce', nonce);
    
    if (template_id) {
        base_url.searchParams.set('template_id', template_id);
        base_url.searchParams.set('type', 'single_product');
        edit_button.text(edit_template);
    } else {
        base_url.searchParams.set('type', 'single_product');
        edit_button.text(create_new_template);
    }
    
    edit_button.attr('href', base_url.toString());
  }).trigger('change');

  $('.affililabs-plugin-product-edit__config__tab').on('click', function(tab) {
      $('.affililabs-plugin-product-edit__config__tab')
          .removeClass('affililabs-plugin-product-edit__config__tab--active');
      $(tab.target).addClass('affililabs-plugin-product-edit__config__tab--active');

      $('.affililabs-plugin-product-edit__config__tab-content')
          .removeClass('affililabs-plugin-product-edit__config__tab-content--active');
      $(`.affililabs-plugin-product-edit__config__tab-content[data-tab-content="${
          $(tab.target).data('tab')}"]`)
          .addClass('affililabs-plugin-product-edit__config__tab-content--active');
  });

  $('#upload-product-image-button').on('click', function(e) {
      e.preventDefault();

      if (file_frame) {
          file_frame.open();
          return;
      }

      file_frame = wp.media({
          title: file_upload_button_title,
          button: {
              text: file_upload_button_text
          },
          multiple: false
      });

      file_frame.on('select', function() {
          const attachment = file_frame.state().get('selection').first().toJSON();
          $('#product-image-preview').attr('src', attachment.url);
          $('#product-image-preview').data('attachment_id', attachment.id);
          console.log('image obj', attachment);
      });

      file_frame.open();
  });

  let index_field = 1;
  jQuery('#add-new-field').on('click', function() {
      let newFieldId = ++index_field;
      let newFieldHtml = `
          <div class="affililabs-plugin-product-edit__config__field">
          <label for="custom-field-${newFieldId}-name">Name</label>
          <input type="text" id="custom-field-${newFieldId}-name" name="custom-field-${newFieldId}-name" value="">
          <label for="custom-field-${newFieldId}-value">Value</label>
          <input type="text" id="custom-field-${newFieldId}-value" name="custom-field-${newFieldId}-value" value="">
          </div>
      `;
      jQuery('.affililabs-plugin-product-edit__config__tab-content--active .affililabs-plugin-product-edit__config__field:last-of-type').after(newFieldHtml);
  });

  $(document).on('click', '.affililabs-plugin__product-edit-save-button', save_product);

  $(document).on('click', '.affililabs-plugin__product-edit-rate-ai-button', function() {
    const $button = $(this);
    $button.prop('disabled', true);

    const product_data = {
        name: $('#product-name').val().trim(),
        price: $('#product-price').val().trim(),
        affiliate_link: '',
        category: $('#product-category').val().trim(),
        description: $('#product-description').val().trim(),
        short_features: $('#product-short-features').val().trim(),
        pros: $('#product-pros').val().trim(),
        cons: $('#product-cons').val().trim(),
    };

    $.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
            action: 'affililabs_rate_product_with_ai',
            product_data: JSON.stringify(product_data),
            nonce: nonce
        },
        beforeSend: function() {
            if (window.affililabs_spinner) {
                window.affililabs_spinner.show(rating_product_with_ai);
            }
        },
        success: function(response) {
            if (response.success && response.data.product_rating_response.status === 'ok') {
                let raw_data = response.data.product_rating_response.data;
                let ai_data = null;

                try {
                    if (typeof raw_data === 'string') {
                        // Clean markdown code blocks if present
                        raw_data = raw_data.replace(/^```json\s*/, '').replace(/\s*```$/, '');
                        
                        try {
                            ai_data = JSON.parse(raw_data);
                        } catch (e) {
                            console.warn('Full JSON parse failed for rating data, attempting partial extraction', e);
                            ai_data = {};
                            
                            // Helper to extract field by key
                            const extract_field = (key) => {
                                const match = raw_data.match(new RegExp(`"${key}"\\s*:\\s*(".*?"|\\[.*?\\]|{.*?}|[0-9.]+|true|false)`, 's'));
                                if (match) {
                                    try {
                                        return JSON.parse(match[1]);
                                    } catch (e2) {
                                        // Fallback for simple strings
                                        if (match[1].startsWith('"')) {
                                            return match[1].slice(1, -1);
                                        }
                                        return match[1]; // numbers/booleans
                                    }
                                }
                                return null;
                            };

                            ai_data.rating = extract_field('rating');
                            ai_data.rationale = extract_field('rationale');
                            
                            // Try to extract sources array specifically
                            const sourcesMatch = raw_data.match(/"sources"\s*:\s*(\[[\s\S]*?\])/);
                            if (sourcesMatch) {
                                try {
                                    let sourcesJson = sourcesMatch[1];
                                    if (sourcesJson.trim().endsWith(']')) {
                                        ai_data.sources = JSON.parse(sourcesJson);
                                    }
                                } catch (e3) {
                                    console.error('Sources extraction failed', e3);
                                }
                            }
                        }
                    } else {
                        ai_data = raw_data;
                    }
                } catch (e) {
                    console.error('JSON Processing Error:', e);
                    ai_data = {};
                }

                if (ai_data) {
                    if (ai_data.rating) $('#product-rating').val(ai_data.rating);
                    if (ai_data.rationale) $('#product-rating-rationale').val(ai_data.rationale);
                    if (Array.isArray(ai_data.sources)) {
                        if (list_separator === 'newline') {
                          $('#product-rating-sources').val(ai_data.sources.join('\n'));
                        } else {
                          $('#product-rating-sources').val(ai_data.sources.join(', '));
                        }
                    }
                }
            } else {
                window.affililabs_modal.error(
                    ai_rating_error_title,
                    response.data.message || 'An unknown error occurred.'
                );
            }
        },
        error: function(xhr) {
            window.affililabs_modal.error(
                ai_rating_error_title,
                xhr.responseJSON?.data?.message || 'A server error occurred.'
            );
        },
        complete: function() {
            $button.prop('disabled', false);
            if (window.affililabs_spinner) {
                window.affililabs_spinner.hide();
            }
        }
    });
  });

  $(document).on('click', '.affililabs-plugin__product-edit-deep-review-ai-button', function() {
    connect_socket_if_needed();

    if (!socket) {
        window.affililabs_modal.error(
            deep_review_error_title,
             'Socket connection initialization failed. Please check console for details.'
        );
        return;
    }

    const $button = $(this);
    // Gather Data
    const product_data = {
        name: $('#product-name').val().trim(),
        price: $('#product-price').val().trim(),
        affiliate_link: '',
        category: $('#product-category').val().trim(),
        description: $('#product-description').val().trim(),
        short_features: $('#product-short-features').val().trim(),
        pros: $('#product-pros').val().trim(),
        cons: $('#product-cons').val().trim(),
    };

    const criteria = [];
    $('.affililabs-plugin-product-edit__config__criterion').each(function() {
      const criterion_id = $(this).data('criterion-id');
      const criterion_name = $(this).find('h3').text();

      const fields = [];
      $(this).find('input[name^="criterion_value_"], select[name^="criterion_value_"]').each(function() {
        const field_name = $(this).data('field-name');
        const field_value = $(this).val();
        
        if (field_value) {
            fields.push({
              name: field_name,
              value: field_value
            });
        }
      });

      if (fields.length > 0) {
        criteria.push({
          id: criterion_id,
          name: criterion_name,
          fields: fields
        });
      }
    });

    if (criteria.length > 0) {
      product_data.criteria = criteria;
    }

    // Generate Request ID
    currentRequestId = 'req_' + Date.now();

    // Show Modal
    window.affililabs_modal.show({
        type: 'notice',
        title: deep_review_title,
        message: $('#affililabs-deep-review-content-template').html(),
        buttons: { ok: close }
    });
    
    // Manually trigger "Initializing" state in the fresh modal DOM
    $('#affililabs-modal #ai-deep-review-status-label').text(deep_review_initializing);

    // Emit Request
    socket.emit('ai:request', {
        taskType: 'deep_review_product', 
        language: ai_config.language || 'english',
        api_key: ai_config.api_key,
        freemius_user_id: ai_config.freemius_user_id,
        freemius_secret_key: ai_config.freemius_secret_key,
        data: product_data,
        requestId: currentRequestId
    });
  });

  $(document).on('click', '.affililabs-plugin__product-edit-ai-button', function() {
    const product_data = {};
    product_data.name = $('#product-name').val();
    product_data.price = $('#product-price').val();
    product_data.category = $('#product-category').val();
    product_data.description = $('#product-description').val();
    product_data.short_features = $('#product-short-features').val();
    product_data.pros = $('#product-pros').val();
    product_data.cons = $('#product-cons').val();

    product_data.current_rating = $('#product-rating').val();
    product_data.current_rating_rationale = $('#product-rating-rationale').val();
    product_data.current_rating_sources = $('#product-rating-sources').val();

    const criteria = [];
    $('.affililabs-plugin-product-edit__config__criterion').each(function() {
      const criterion_id = $(this).data('criterion-id');
      const criterion_name = $(this).find('h3').text();
      const criterion_description = $(this).find('p.description').text();

      const fields = [];
      $(this).find('input[name^="criterion_value_"], select[name^="criterion_value_"]').each(function() {
        const field_name = $(this).data('field-name');
        const field_type = $(this).data('field-type');
        const field_unit = $(this).data('field-unit') || '';
        const field_value = $(this).val();
        const definition_id = $(this).data('definition-id');

        fields.push({
          definition_id: definition_id,
          name: field_name,
          type: field_type,
          unit: field_unit,
          value: field_value
        });
      });

      if (fields.length > 0) {
        criteria.push({
          id: criterion_id,
          name: criterion_name,
          description: criterion_description,
          fields: fields
        });
      }
    });

    if (criteria.length > 0) {
      product_data.criteria = criteria;
    }

    console.log('product data', product_data);
    $(this).prop('disabled', true);

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        'action': 'affililabs_fill_product_data',
        'product_data': JSON.stringify(product_data),
        'nonce': nonce
      },
      beforeSend: function() {
        if (window.affililabs_spinner) {
            window.affililabs_spinner.show('Generating product data with AI...');
        }
      },
      success: function(response) {
        if (response.data.status === 'ok') {
          const ai_response = response.data;

          if (ai_response.product_response.status !== 'ok') {
            console.error('[AIResponse.data.status.error] There was an error generating product data with AI:', ai_response.product_response.message);
            window.affililabs_modal.error(
              save_error_title,
              'Error: ' + (ai_response.product_response.message || save_error_unknown)
            );
            return;
          }

          let product_data_raw = ai_response.product_response.data;
          let ai_generated_data = null;

          try {
              if (typeof product_data_raw === 'string') {
                  // Clean markdown code blocks if present
                  product_data_raw = product_data_raw.replace(/^```json\s*/, '').replace(/\s*```$/, '');
                  
                  try {
                      ai_generated_data = JSON.parse(product_data_raw);
                  } catch (e) {
                      console.warn('Full JSON parse failed for product data, attempting partial extraction', e);
                      ai_generated_data = {};
                      
                      // Helper to extract field by key
                      const extractField = (key) => {
                          const match = product_data_raw.match(new RegExp(`"${key}"\\s*:\\s*(".*?"|\\[.*?\\]|{.*?})`, 's')); // Basic regex, might need refinement for nested structures
                          if (match) {
                              try {
                                  return JSON.parse(match[1]);
                              } catch (e2) {
                                  // Fallback for simple strings that might be truncated or malformed
                                  if (match[1].startsWith('"')) {
                                      return match[1].slice(1, -1); // Strip quotes
                                  }
                              }
                          }
                          return null;
                      };

                      ai_generated_data.description = extractField('description');
                      ai_generated_data.short_features = extractField('short_features');
                      ai_generated_data.pros = extractField('pros');
                      ai_generated_data.cons = extractField('cons');
                      
                      // Try to extract criteria array
                      const criteriaMatch = product_data_raw.match(/"criteria"\s*:\s*(\[[\s\S]*?\])/);
                      if (criteriaMatch) {
                          try {
                              let criteriaJson = criteriaMatch[1];
                              if (criteriaJson.trim().endsWith(']')) {
                                  ai_generated_data.criteria = JSON.parse(criteriaJson);
                              }
                          } catch (e3) {
                              console.error('Criteria extraction failed', e3);
                          }
                      }
                  }
              } else {
                  ai_generated_data = product_data_raw;
              }
          } catch (e) {
              console.error('JSON Processing Error:', e);
              ai_generated_data = {};
          }

          const fields_to_update = {
            'product-description': ai_generated_data.description,
            'product-short-features': ai_generated_data.short_features,
            'product-pros': ai_generated_data.pros,
            'product-cons': ai_generated_data.cons
          };

          $('.affililabs-plugin-product-edit__ai-suggestion-card').remove();

          for (const [field_id, new_value] of Object.entries(fields_to_update)) {
            if (new_value && new_value !== '' && new_value.length > 0) {
              let value_to_escape = new_value;
              if (Array.isArray(new_value)) {
                value_to_escape = new_value.join(', ');
              }
              
              if (!value_to_escape || value_to_escape.trim() === '') {
                continue;
              }

              const $target_field = $(`#${field_id}`);
              if ($target_field.length) {
                const current_label = $target_field.closest('.affililabs-plugin-product-edit__config__field').find('label').text();
                const suggestion_html = `
                  <div class="affililabs-plugin-product-edit__ai-suggestion-card" data-field-id="${field_id}">
                    <label>${current_label}</label>
                    <p>${escape_html(value_to_escape)}</p>
                    <div class="affililabs-plugin-product-edit__ai-suggestion-actions">
                      <button class="affililabs-plugin__button affililabs-plugin__button--apply-ai-suggestion">${apply_suggestion_button_message}</button>
                      <button class="affililabs-plugin__button affililabs-plugin__button--deny-ai-suggestion">${deny_suggestion_button_message}</button>
                    </div>
                  </div>
                `;
                $target_field.closest('.affililabs-plugin-product-edit__config__field').append(suggestion_html);
              }
            }
          }

          for (const [key, value] of Object.entries(ai_generated_data.custom_fields || {})) {
            if (value && value !== '' && value.length > 0) {
              let value_to_escape = value;
              if (Array.isArray(value)) {
                value_to_escape = value.join(', ');
              }
              
              if (!value_to_escape || value_to_escape.trim() === '') {
                continue;
              }

              const field_id = `custom-field-${key}`;
              const $target_field = $(`#${field_id}`);
              if ($target_field.length) {
                const current_label = $target_field.closest('.affililabs-plugin-product-edit__config__field').find('label').text();
                const suggestion_html = `
                  <div class="affililabs-plugin-product-edit__ai-suggestion-card" data-field-id="${field_id}">
                    <label>${current_label}</label>
                    <p>${escape_html(value_to_escape)}</p>
                    <div class="affililabs-plugin-product-edit__ai-suggestion-actions">
                      <button class="affililabs-plugin__button affililabs-plugin__button--apply-ai-suggestion">${apply_suggestion_button_message}</button>
                      <button class="affililabs-plugin__button affililabs-plugin__button--deny-ai-suggestion">${deny_suggestion_button_message}</button>
                    </div>
                  </div>
                `;
                $target_field.closest('.affililabs-plugin-product-edit__config__field').append(suggestion_html);
              }
            }
          }

          if (ai_generated_data.criteria) {
            for (const suggestion of ai_generated_data.criteria) {
              render_criteria_suggestion(suggestion);
            }
          }

          const rating_suggestion = {
              rating: ai_generated_data.current_rating,
              rationale: ai_generated_data.current_rating_rationale,
              sources: ai_generated_data.current_rating_sources
          };
          if (rating_suggestion.rating || rating_suggestion.rationale || rating_suggestion.sources) {
            render_rating_suggestion(rating_suggestion);
          }

          console.log('AI generated product data', ai_generated_data);
        } else {
          console.error('[Response.data.status.error] There was an error generating product data with AI:', response);
          window.affililabs_modal.error(
            save_error_title,
            response.data.message || save_error_unknown
          );
        }
      },
      error: function(xhr, status, error) {
          console.error('[Response.error] There was an error generating product data with AI:', xhr, status, error);
          window.affililabs_modal.error(
            save_error_title,
            xhr.responseJSON.data.message || save_error_unknown
          );
      },
      complete: function() {
        $('.affililabs-plugin__product-edit-ai-button').prop('disabled', false);
        if (window.affililabs_spinner) {
            window.affililabs_spinner.hide();
        }
      }
    });
  });

  $(document).on('click', '.affililabs-plugin__button--apply-ai-suggestion', function() {
    const $field_container = $(this).closest('.affililabs-plugin-product-edit__ai-suggestion-card');
    const field_id = $field_container.data('field-id');
    const criterion_id = $field_container.data('criterion-id');
    let new_value = $field_container.find('p').get(0).innerText.trim();

    if (criterion_id) {
        const $target_field = $(`#${field_id}`);
        if ($target_field.length) {
            $target_field.val(new_value);
        }
    } else {
        if (field_id === 'product-rating-sources' || field_id === 'product-criteria-sources') {
            new_value = new_value.split(',').map(item => item.trim()).join('\n'); // separator is always newline because there is no guarantee that the incoming URLs will be comma-free
        } else if (field_id === 'product-short-features' || field_id === 'product-pros' || field_id === 'product-cons') {
            if (typeof new_value === 'string') {
                if (list_separator === 'newline') {
                    new_value = new_value.split(',').map(item => item.trim()).join('\n');
                } else {
                    new_value = new_value.split('\n').map(item => item.trim()).join(', ');
                }
            }
        }
        $(`#${field_id}`).val(new_value);
    }
    $field_container.remove();
  });

  $(document).on('click', '.affililabs-plugin__button--deny-ai-suggestion', function() {
    const $field_container = $(this).closest('.affililabs-plugin-product-edit__ai-suggestion-card');
    $field_container.remove();
  });

  function sanitize_slug(text) {
    return text.toString().toLocaleLowerCase()
      .replace(/\s+/g, '-')           // Replace spaces with -
      .replace(/\-\-+/g, '-')         // Replace multiple - with single -
      .replace(/^-+/, '')             // Trim - from start of text
      .replace(/-+$/, '');            // Trim - from end of text
  }

  $('#add-product-category-button').on('click', function() {
    const modal_content = `
      <div class="form-field" style="margin-bottom: 15px;">
        <label for="new-category-name" style="display: block; margin-bottom: 5px;">${name_label}</label>
        <input type="text" id="new-category-name" name="new-category-name" required style="width: 100%;">
      </div>
      <div class="form-field" style="margin-bottom: 15px;">
        <label for="new-category-slug" style="display: block; margin-bottom: 5px;">${slug_label}</label>
        <input type="text" id="new-category-slug" name="new-category-slug" required style="width: 100%;">
      </div>
      <div class="form-field">
        <label for="new-category-description" style="display: block; margin-bottom: 5px;">${description_label}</label>
        <textarea id="new-category-description" name="new-category-description" style="width: 100%; min-height: 80px;"></textarea>
      </div>
    `;

    window.affililabs_modal.confirm(
      add_new_category_title,
      modal_content,
      function() {
        const name = $('#new-category-name').val().trim();
        const slug = $('#new-category-slug').val().trim();
        const description = $('#new-category-description').val().trim();

        if (!name || !slug) {
          window.affililabs_modal.error(save_error_title, name_slug_required_error);
          return;
        }

        $.ajax({
          url: ajaxurl,
          type: 'POST',
          data: {
            action: 'affililabs_add_product_category_with_criteria',
            name: name,
            slug: slug,
            description: description,
            nonce: nonce
          },
          beforeSend: function() {
            if (window.affililabs_spinner) {
              window.affililabs_spinner.show(saving_category_message);
            }
          },
          success: function(response) {
            if (response.success) {
              const new_option = new Option(response.data.name, response.data.slug, true, true);
              $('#product-category').append(new_option).trigger('change');
              window.affililabs_modal.success(save_success_title, category_created_message);
            } else {
              window.affililabs_modal.error(save_error_title, response.data.message);
            }
          },
          error: function(xhr) {
            window.affililabs_modal.error(save_error_title, xhr.responseJSON.data.message || error_unknown_message);
          },
          complete: function() {
            if (window.affililabs_spinner) {
              window.affililabs_spinner.hide();
            }
          }
        });
      },
      {
        buttons: { confirm: save_button }
      }
    );

    $('#new-category-name').on('input', function() {
      $('#new-category-slug').val(sanitize_slug($(this).val()));
    });
  });

  $('#add-marketplace-button').on('click', function() {
    const modal_content = `
      <div class="form-field" style="margin-bottom: 15px;">
        <label for="new-marketplace-name" style="display: block; margin-bottom: 5px;">${name_label}</label>
        <input type="text" id="new-marketplace-name" name="new-marketplace-name" required style="width: 100%;">
      </div>
      <div class="form-field">
        <label for="new-marketplace-description" style="display: block; margin-bottom: 5px;">${description_label}</label>
        <textarea id="new-marketplace-description" name="new-marketplace-description" style="width: 100%; min-height: 80px;"></textarea>
      </div>
    `;

    window.affililabs_modal.confirm(
      add_new_marketplace_title,
      modal_content,
      function() {
        const name = $('#new-marketplace-name').val().trim();
        const description = $('#new-marketplace-description').val().trim();

        if (!name) {
          window.affililabs_modal.error(save_error_title, name_required_error);
          return;
        }

        $.ajax({
          url: ajaxurl,
          type: 'POST',
          data: {
            action: 'affililabs_add_marketplace',
            name: name,
            description: description,
            nonce: nonce
          },
          beforeSend: function() {
            if (window.affililabs_spinner) {
              window.affililabs_spinner.show(saving_marketplace_message);
            }
          },
          success: function(response) {
            if (response.success) {
              const new_option = new Option(response.data.name, response.data.marketplace_id, true, true);
              $('#product-marketplace').append(new_option).trigger('change');
              window.affililabs_modal.success(save_success_title, marketplace_created_message);
            } else {
              window.affililabs_modal.error(save_error_title, response.data.message);
            }
          },
          error: function(xhr) {
            window.affililabs_modal.error(save_error_title, xhr.responseJSON.data.message || error_unknown_message);
          },
          complete: function() {
            if (window.affililabs_spinner) {
              window.affililabs_spinner.hide();
            }
          }
        });
      },
      {
        buttons: { confirm: save_button }
      }
    );
  });

  /**
   * Render criteria field suggestions in the criteria tab
   * @param {Object} suggestion - The criteria suggestion object
   */
  function render_criteria_suggestion(suggestion) {
    const criterion_id = suggestion.id;
    const $criterion_container = $(`.affililabs-plugin-product-edit__config__criterion[data-criterion-id="${criterion_id}"]`);

    if ($criterion_container.length) {
      for (const field_suggestion of suggestion.fields) {
        const field_id = `criterion-${criterion_id}-field-def-${field_suggestion.definition_id}`;
        const $target_field = $(`#${field_id}`);

        if ($target_field.length) {
          let value_to_set = field_suggestion.value;
          
          if (field_suggestion.type === 'boolean') {
              if (value_to_set === true || value_to_set === 'true' || value_to_set === '1' || value_to_set === 1) {
                  value_to_set = 'yes';
              } else if (value_to_set === false || value_to_set === 'false' || value_to_set === '0' || value_to_set === 0) {
                  value_to_set = 'no';
              }
          }
          
          $target_field.val(value_to_set);
          $target_field.trigger('change');
        }
      }
    }
  }

  /**
   * Render rating suggestions in the basics tab
   * @param {Object} rating_suggestion - The rating suggestion object
   */
  function render_rating_suggestion(rating_suggestion) {
    // Rating field suggestion
    if (rating_suggestion.rating) {
      $('#product-rating').val(rating_suggestion.rating);
    }

    // Rating rationale suggestion
    if (rating_suggestion.rationale && rating_suggestion.rationale !== '' && rating_suggestion.rationale.trim() !== '') {
      const rationale_field_html = `
        <div class="affililabs-plugin-product-edit__ai-suggestion-card" data-field-id="product-rating-rationale">
          <label>Rating Rationale</label>
          <p>${escape_html(rating_suggestion.rationale)}</p>
          <div class="affililabs-plugin-product-edit__ai-suggestion-actions">
            <button class="affililabs-plugin__button affililabs-plugin__button--apply-ai-suggestion">${apply_suggestion_button_message}</button>
            <button class="affililabs-plugin__button affililabs-plugin__button--deny-ai-suggestion">${deny_suggestion_button_message}</button>
          </div>
        </div>
      `;
      $('#product-rating-rationale').closest('.affililabs-plugin-product-edit__config__field').append(rationale_field_html);
    }

    // Rating sources suggestion
    if (rating_suggestion.sources) {
      let sources_value = rating_suggestion.sources;
      if (Array.isArray(sources_value)) {
          sources_value = sources_value.join(', '); //@TODO: use according to list_separator
        }
      
      if (sources_value && sources_value !== '' && sources_value.trim() !== '') {
        $('#product-rating-sources').val(sources_value);
      }
    }
  }

  $(document).on('click', '.affililabs-plugin__product-edit-ai-criteria-button', function() {
    const product_data = {};
    product_data.name = $('#product-name').val();
    product_data.description = $('#product-description').val();
    product_data.short_features = $('#product-short-features').val();
    product_data.pros = $('#product-pros').val();
    product_data.cons = $('#product-cons').val();

    const criteria = [];
    $('.affililabs-plugin-product-edit__config__criterion').each(function() {
      const criterion_id = $(this).data('criterion-id');
      const criterion_name = $(this).find('h3').text();
      const criterion_description = $(this).find('p.description').text();

      const fields = [];
      $(this).find('input[name^="criterion_value_"], select[name^="criterion_value_"]').each(function() {
        const field_name = $(this).data('field-name');
        const field_type = $(this).data('field-type');
        const field_unit = $(this).data('field-unit') || '';
        const field_value = $(this).val();
        const definition_id = $(this).data('definition-id');

        fields.push({
          definition_id: definition_id,
          name: field_name,
          type: field_type,
          unit: field_unit,
          value: field_value
        });
      });

      if (fields.length > 0) {
        criteria.push({
          id: criterion_id,
          name: criterion_name,
          description: criterion_description,
          fields: fields
        });
      }
    });

    if (criteria.length > 0) {
      product_data.criteria = criteria;
    } else {
        window.affililabs_modal.error(save_error_title, no_criteria_found_message);
        return;
    }

    $(this).prop('disabled', true);

    $.ajax({
      url: ajaxurl,
      type: 'POST',
      data: {
        'action': 'affililabs_fill_product_criteria',
        'product_data': JSON.stringify(product_data),
        'nonce': nonce
      },
      beforeSend: function() {
        if (window.affililabs_spinner) {
            window.affililabs_spinner.show(filling_criteria_with_ai_message);
        }
      },
      success: function(response) {
        if (response.data.status === 'ok') {
          const ai_response = response.data;
          console.log('AI Response:', ai_response);

          if (ai_response.product_response.status !== 'ok') {
             window.affililabs_modal.error(save_error_title, ai_response.product_response.message || 'Unknown error from AI');
             return;
          }
          
          let ai_generated_data = null;
          try {
              let raw_data = ai_response.product_response.data;
              console.log('Product Response Data (Raw):', raw_data);
              
              if (typeof raw_data === 'string') {
                  // Clean markdown code blocks if present
                  raw_data = raw_data.replace(/^```json\s*/, '').replace(/\s*```$/, '');
                  
                  try {
                      ai_generated_data = JSON.parse(raw_data);
                  } catch (e) {
                      console.warn('Full JSON parse failed, attempting partial extraction', e);
                      ai_generated_data = {};
                      
                      // Try to extract criteria array
                      const criteria_match = raw_data.match(/"criteria"\s*:\s*(\[[\s\S]*?\])/);
                      if (criteria_match) {
                          try {
                              // Ensure the extracted array is valid JSON (e.g. check for closing bracket)
                              let criteria_json = criteria_match[1];
                              // Simple check if it looks complete
                              if (criteria_json.trim().endsWith(']')) {
                                  ai_generated_data.criteria = JSON.parse(criteria_json);
                              }
                          } catch (e2) {
                              console.error('Criteria extraction failed', e2);
                          }
                      }
                      
                      // Try to extract sources array
                      const sources_match = raw_data.match(/"sources"\s*:\s*(\[[\s\S]*?\])/);
                      if (sources_match) {
                          try {
                              let sources_json = sources_match[1];
                              if (sources_json.trim().endsWith(']')) {
                                  ai_generated_data.sources = JSON.parse(sources_json);
                              }
                          } catch (e3) {
                              console.error('Sources extraction failed', e3);
                          }
                      }
                      
                      if (!ai_generated_data.criteria && !ai_generated_data.sources) {
                          ai_generated_data = null; // Failed completely
                      }
                  }
              } else {
                  ai_generated_data = raw_data;
              }
          } catch (e) {
              console.error('JSON Processing Error:', e);
              ai_generated_data = null;
          }
          
          console.log('AI Generated Data (Parsed):', ai_generated_data);

          if (ai_generated_data && ai_generated_data.criteria) {
            let all_sources = [];
            for (const suggestion of ai_generated_data.criteria) {
              render_criteria_suggestion(suggestion);
              
              if (suggestion.fields && Array.isArray(suggestion.fields)) {
                  suggestion.fields.forEach(field => {
                      if (field.sources && Array.isArray(field.sources)) {
                          all_sources = all_sources.concat(field.sources);
                      }
                  });
              }
            }
            
            if (ai_generated_data.sources && Array.isArray(ai_generated_data.sources)) {
                all_sources = all_sources.concat(ai_generated_data.sources);
            }

            if (all_sources.length > 0) {
                // Unique sources only
                all_sources = [...new Set(all_sources)];
                
                let sources_text = '';
                if (list_separator === 'newline') {
                    sources_text = all_sources.join('\n');
                } else {
                    sources_text = all_sources.join(', ');
                }
                
                const sources_field_id = 'product-criteria-sources';
                const $target_field = $(`#${sources_field_id}`);
                if ($target_field.length) {
                    $target_field.val(sources_text);
                }
            }
            
            window.affililabs_modal.success(save_success_title, 'Criteria filled successfully!');
          } else {
             console.error('Missing criteria in AI data', ai_generated_data);
             window.affililabs_modal.error(save_error_title, 'No criteria data received from AI.');
          }

        } else {
          window.affililabs_modal.error(save_error_title, response.data.message || save_error_unknown);
        }
      },
      error: function(xhr) {
          window.affililabs_modal.error(save_error_title, xhr.responseJSON?.data?.message || save_error_unknown);
      },
      complete: function() {
        $('.affililabs-plugin__product-edit-ai-criteria-button').prop('disabled', false);
        if (window.affililabs_spinner) {
            window.affililabs_spinner.hide();
        }
      }
    });
  });
});
