jQuery(document).ready(function($) {
    const { nonce, i18n } = affililabs_category_edit;
    let { criteria_groups } = affililabs_category_edit;
    const {
        saving,
        save_success_title,
        save_success_message,
        save_error_title,
        save_error_message,
        field_name,
        field_type,
        unit,
        remove,
        add_field,
        remove_criterion,
        criterion_object_name,
        criterion_object_description,
    } = i18n;

    const {
        ai_suggest_criteria_title,
        ai_suggest_criteria_loading,
        ai_suggest_criteria_error,
        ai_suggest_criteria_no_suggestions,
        ai_suggest_criteria_accept,
        ai_suggest_criteria_saving,
        ai_suggest_criteria_save_success,
        ai_suggest_criteria_save_error,
        no_fields_defined,
        ai_suggest_criteria_save_list_first,
        ai_suggest_criteria_category_label,
        ai_suggest_criteria_fields_label,
        ai_suggest_criteria_replacing_label
    } = affililabs_category_edit.i18n;

    let is_ai_suggest_in_progress = false;

    const init = () => {
        const show_groups = criteria_groups && criteria_groups.length > 1;
        $('#criteria-list').empty();

        if (criteria_groups && criteria_groups.length > 0) {
            criteria_groups.forEach(group => {
                if (group.fields && Array.isArray(group.fields)) {
                    group.fields.sort((a, b) => (a.definition_id || 0) - (b.definition_id || 0));
                }
                render_criterion_group(group, show_groups);
            });
        } else {
            render_criterion_group({ id: 'default', name: 'Default', description: '', fields: [] }, false);
        }

        if (show_groups) {
            $('#add-criterion-button').show();
        } else {
            $('#add-criterion-button').hide();
        }
        update_default_card_state();
    };

    const render_criterion_group = (group, show_header = true) => {
        console.log(group);
        const is_empty = !(group.fields && group.fields.length > 0);
        const group_name = group.name || group.title;
        const criterion_html = `
            <div class="affililabs-plugin-product-category-edit__criterion-card ${is_empty ? 'is-empty' : ''}" data-id="${group.id}">
                <div class="affililabs-plugin-product-category-edit__criterion-card-header" style="display: ${show_header ? 'block' : 'none'};">
                    <div class="affililabs-plugin-product-category-edit__config__field">
                        <label>${criterion_object_name}</label>
                        <input type="text" name="criteria[${group.id}][name]" class="affililabs-plugin__input" value="${group_name || ''}" />
                    </div>
                    <div class="affililabs-plugin-product-category-edit__config__field">
                        <label>${criterion_object_description}</label>
                        <textarea name="criteria[${group.id}][description]" class="affililabs-plugin__textarea">${group.description || ''}</textarea>
                    </div>
                </div>
                <div class="affililabs-plugin-product-category-edit__criterion-fields-list">
                    ${(group.fields || []).map(field => get_field_html(group.id, field)).join('')}
                </div>
                <div class="affililabs-plugin-product-category-edit__criterion-card-footer">
                    <button type="button" class="add-field-button affililabs-plugin__button affililabs-plugin__button--secondary">${add_field}</button>
                    ${show_header ? `<button type="button" class="remove-criterion-button affililabs-plugin__button affililabs-plugin__button--danger">${remove_criterion}</button>` : ''}
                </div>
            </div>
        `;
        $('#criteria-list').append(criterion_html);
    };

    const get_field_html = (criterion_id, field) => {
        const field_id = field.id || 'new-' + new Date().getTime();
        const is_number = field.type === 'number';
        return `
            <div class="affililabs-plugin-product-category-edit__config__field" data-id="${field_id}">
                <label>${field_name}</label>
                <input type="text" name="criteria[${criterion_id}][fields][${field_id}][name]" class="affililabs-plugin__input" placeholder="Field Name" value="${field.name || ''}" />

                <label>${field_type}</label>
                <select name="criteria[${criterion_id}][fields][${field_id}][type]" class="affl-w-100px field-type-select">
                    <option value="text" ${field.type === 'text' ? 'selected' : ''}>Text</option>
                    <option value="number" ${is_number ? 'selected' : ''}>Number</option>
                    <option value="boolean" ${field.type === 'boolean' ? 'selected' : ''}>Boolean</option>
                </select>

                <div class="field-unit-container" style="display: ${is_number ? 'flex' : 'none'};">
                  <label>${unit}</label>
                  <input type="text" name="criteria[${criterion_id}][fields][${field_id}][unit]" class="affililabs-plugin__input field-unit" placeholder="e.g. GB, MHz, etc." value="${field.unit || ''}" />
                </div>

                <button type="button" class="remove-field-button affililabs-plugin__button affililabs-plugin__button--danger">${remove}</button>
            </div>
        `;
    };

    const update_default_card_state = () => {
        const default_card = $('#criteria-list .affililabs-plugin-product-category-edit__criterion-card[data-id="default"]');
        if (default_card.length) {
            const field_count = default_card.find('.affililabs-plugin-product-category-edit__config__field[data-id]').length;
            if (field_count === 0) {
                default_card.addClass('is-empty');
            } else {
                default_card.removeClass('is-empty');
            }
        }
    };

    $('.affililabs-plugin-product-category-edit__config__tab').on('click', function() {
        const tab = $(this).data('tab');
        $('.affililabs-plugin-product-category-edit__config__tab').removeClass('affililabs-plugin-product-category-edit__config__tab--active');
        $(this).addClass('affililabs-plugin-product-category-edit__config__tab--active');
        $('.affililabs-plugin-product-category-edit__config__tab-content').removeClass('affililabs-plugin-product-category-edit__config__tab-content--active');
        $(`.affililabs-plugin-product-category-edit__config__tab-content[data-tab-content="${tab}"]`).addClass('affililabs-plugin-product-category-edit__config__tab-content--active');
    });

    $('#add-criterion-button').on('click', function() {
        render_criterion_group({ id: 'new-' + new Date().getTime(), name: '', description: '', fields: [] }, true);
    });

    $(document).on('click', '.add-field-button', function() {
        const card = $(this).closest('.affililabs-plugin-product-category-edit__criterion-card');
        const target_list = card.find('.affililabs-plugin-product-category-edit__criterion-fields-list');
        const criterion_id = card.data('id');
        target_list.append(get_field_html(criterion_id, {}));
        update_default_card_state();
    });

    $(document).on('change', '.field-type-select', function() {
        const unit_container = $(this).closest('.affililabs-plugin-product-category-edit__config__field').find('.field-unit-container');
        if ($(this).val() === 'number') {
            unit_container.css('display', 'flex');
        } else {
            unit_container.hide();
        }
    });

    const criteria_to_remove = [];
    $(document).on('click', '.remove-criterion-button', function() {
        const card = $(this).closest('.affililabs-plugin-product-category-edit__criterion-card');
        const criterion_id = card.data('id');
        
        if (criterion_id && !String(criterion_id).startsWith('new-')) {
            criteria_to_remove.push(criterion_id);
        }

        card.remove();
    });

    $(document).on('click', '.remove-field-button', function() {
        $(this).closest('.affililabs-plugin-product-category-edit__config__field').remove();
        update_default_card_state();
    });

    $('.affililabs-plugin-product-category-edit__save-button').on('click', function() {
        const category_data = {
            id: $('#category-id').val().trim(),
            name: $('#category-name').val().trim(),
            slug: $('#category-slug').val().trim(),
            description: $('#category-description').val().trim(),
            criteria: {}
        };

        $('#criteria-list .affililabs-plugin-product-category-edit__criterion-card').each(function() {
            const criterion_id = $(this).data('id');
            const criterion = {
                name: $(this).find('input[name*="[name]"]').first().val().trim(),
                description: $(this).find('textarea[name*="[description]"]').val().trim(),
                fields: {}
            };

            const fields = [];
            $(this).find('.affililabs-plugin-product-category-edit__config__field[data-id]').each(function() {
                const field_id = $(this).data('id');
                const original_group = criteria_groups.find(g => g.id == criterion_id || (g.name || g.title) == criterion.name);
                const original_field = original_group && original_group.fields ? original_group.fields.find(f => f.id == field_id) : null;

                const field = {
                    name: $(this).find('input[name*="[name]"]').val().trim(),
                    type: $(this).find('select[name*="[type]"]').val().trim(),
                    unit: $(this).find('input[name*="[unit]"]').val().trim(),
                    definition_id: original_field ? original_field.definition_id : null,
                };
                fields.push(field);
            });
            criterion.fields = fields;
            category_data.criteria[criterion_id] = criterion;
        });

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'affililabs_save_product_category',
                nonce: nonce,
                category_id: category_data.id,
                name: category_data.name,
                slug: category_data.slug,
                description: category_data.description,
                criteria: category_data.criteria,
                criteria_to_remove: criteria_to_remove
            },
            beforeSend: function() {
                if (window.affililabs_spinner) {
                    window.affililabs_spinner.show(saving);
                }
            },
            success: function(response) {
                if (response.success) {
                    window.affililabs_modal.success(save_success_title, save_success_message);
                    
                    if(response.data.criteria_groups) {
                        criteria_groups = response.data.criteria_groups;
                    }
                    
                    if (!category_data.id && response.data.term_id) {
                        $('#category-id').val(response.data.term_id);
                        const new_url = new URL(window.location.href);
                        new_url.searchParams.set('category_id', response.data.term_id);
                        window.history.pushState({ path: new_url.href }, '', new_url.href);
                    }
                } else {
                    window.affililabs_modal.error(save_error_title, response.data.message);
                }
            },
            error: function() {
                window.affililabs_modal.error(save_error_title, save_error_message);
            },
            complete: function() {
                if (window.affililabs_spinner) {
                    window.affililabs_spinner.hide();
                }
            }
        });
    });

    init();
    const sanitize_slug = (text) =>
        text.toString().toLocaleLowerCase()
          .replace(/\s+/g, '-')           // Replace spaces with -
          .replace(/\-\-+/g, '-')         // Replace multiple - with single -
          .replace(/^-+/, '')             // Trim - from start of text
          .replace(/-+$/, '');            // Trim - from end of text

    $('#category-name').on('input', function() {
        const name = $(this).val();
        const slug = sanitize_slug(name);
        $('#category-slug').val(slug);
    });


    const $ai_criteria_suggest_button = $('#ai-suggest-criteria-button');
    const $ai_criteria_suggestions_container = $('#ai-criteria-suggestions-container');
    const $ai_criteria_suggestions_content = $('#ai-criteria-suggestions-content');
    const $close_ai_criteria_suggestions_button = $('#close-ai-criteria-suggestions');

    $ai_criteria_suggest_button.on('click', function() {
        if (is_ai_suggest_in_progress) {
            return;
        }

        const category_id = $('#category-id').val();

        is_ai_suggest_in_progress = true;
        $ai_criteria_suggest_button.prop('disabled', true);

        if (!category_id) {
            window.affililabs_modal.error(
                save_error_title,
                ai_suggest_criteria_save_list_first
            );
            is_ai_suggest_in_progress = false;
            $ai_criteria_suggest_button.prop('disabled', false);
            return;
        }

        $ai_criteria_suggestions_container.show();
        $ai_criteria_suggestions_content.html(`<p class="ai-loading-message">${ai_suggest_criteria_loading}</p>`);
        if (window.affililabs_spinner) {
            window.affililabs_spinner.show(ai_suggest_criteria_loading);
        }

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'affililabs_ai_suggest_criteria_for_category',
                category_id: category_id,
                nonce: nonce
            },
            success: function(response) {
                if (response.success && response.data && response.data.suggestions) {
                    render_ai_criteria_suggestions(response.data.suggestions);
                } else {
                    window.affililabs_modal.error(
                        ai_suggest_criteria_title,
                        response.data?.message || ai_suggest_criteria_error
                    );
                    $ai_criteria_suggestions_content.html('');
                }
            },
            error: function() {
                window.affililabs_modal.error(
                    ai_suggest_criteria_title,
                    ai_suggest_criteria_error
                );
                $ai_criteria_suggestions_content.html('');
            },
            complete: function() {
                if (window.affililabs_spinner) {
                    window.affililabs_spinner.hide();
                }
                is_ai_suggest_in_progress = false;
                $ai_criteria_suggest_button.prop('disabled', false);
            }
        });
    });

    $close_ai_criteria_suggestions_button.on('click', function() {
        $ai_criteria_suggestions_container.hide();
        $ai_criteria_suggestions_content.empty();
    });

    function render_ai_criteria_suggestions(suggestions) {
        $ai_criteria_suggestions_content.empty();

        if (suggestions.length === 0) {
            $ai_criteria_suggestions_content.html(`<p>${ai_suggest_criteria_no_suggestions}</p>`);
            return;
        }

        let suggestions_html = '<div class="affililabs-plugin__card-container">';
        $.each(suggestions, function(index, suggestion) {
            let fields_list_html = '';
            if (suggestion.fields && suggestion.fields.length > 0) {
                fields_list_html = '<ul>';
                $.each(suggestion.fields, function(field_index, field) {
                    let unit_text = field.unit ? ` (${field.unit})` : '';
                    let field_name = field.name || 'Unnamed Field';
                    let field_type = field.type || 'text';
                    let type_class = '';
                    switch (field_type) {
                        case 'number':
                            type_class = 'ai-field-type--number';
                            break;
                        case 'text':
                            type_class = 'ai-field-type--text';
                            break;
                        case 'boolean':
                            type_class = 'ai-field-type--boolean';
                            break;
                        default:
                            type_class = '';
                    }
                    fields_list_html += `<li class="${type_class}">${field_name} (${field_type}${unit_text})</li>`;
                });
                fields_list_html += '</ul>';
            } else {
                fields_list_html = `<p>${no_fields_defined}</p>`;
            }

            suggestions_html += `
                <div class="affililabs-plugin__card ai-suggestion-card${suggestion.is_existing_criterion ? ' ai-suggestion-card--existing' : ''}" data-suggestion-index="${index}">
                    <div class="affililabs-plugin__card__content">
                        <strong class="affililabs-plugin__card__title">${suggestion.criteria_name}</strong>
                        ${suggestion.is_existing_criterion && suggestion.existing_criterion_title ? `<p class="ai-replacing-text">${ai_suggest_criteria_replacing_label} <strong>${suggestion.existing_criterion_title}</strong></p>` : ''}
                        <p><strong>${ai_suggest_criteria_category_label}: ${suggestion.category_name}</strong></p>
                        <p><strong>${ai_suggest_criteria_fields_label}:</strong></p>
                        ${fields_list_html}
                    </div>
                    <div class="affililabs-plugin__card__footer">
                        <button class="affililabs-plugin__button affililabs-plugin__button--secondary ai-accept-criteria-suggestion-button" data-suggestion-index="${index}">
                            ${ai_suggest_criteria_accept}
                        </button>
                    </div>
                </div>
            `;
        });
        suggestions_html += '</div>';
        $ai_criteria_suggestions_content.html(suggestions_html);

        $('.ai-accept-criteria-suggestion-button').on('click', function() {
            const button = $(this);
            button.prop('disabled', true);
            const index = button.data('suggestion-index');
            const accepted_suggestion = suggestions[index];
            save_ai_suggested_criterion(accepted_suggestion, button);
        });
    }

    function save_ai_suggested_criterion(suggestion, button) {
        const category_id = $('#category-id').val();
        if (window.affililabs_spinner) {
            window.affililabs_spinner.show(ai_suggest_criteria_saving);
        }

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'affililabs_save_ai_suggested_criteria_for_category',
                nonce: nonce,
                category_id: category_id,
                suggested_criterion: JSON.stringify(suggestion)
            },
            success: function(response) {
                if (response.success) {
                    window.affililabs_modal.success(ai_suggest_criteria_title, ai_suggest_criteria_save_success);
                    criteria_groups = response.data.criteria_groups;
                    init(); // Re-render the criteria list
                    button.closest('.ai-suggestion-card').remove();
                    if ($ai_criteria_suggestions_content.find('.ai-suggestion-card').length === 0) {
                        $ai_criteria_suggestions_content.html(`<p>${ai_suggest_criteria_no_suggestions}</p>`);
                    }
                } else {
                    window.affililabs_modal.error(ai_suggest_criteria_title, response.data.message || ai_suggest_criteria_save_error);
                    button.prop('disabled', false);
                }
            },
            error: function() {
                window.affililabs_modal.error(ai_suggest_criteria_title, ai_suggest_criteria_save_error);
                button.prop('disabled', false);
            },
            complete: function() {
                if (window.affililabs_spinner) {
                    window.affililabs_spinner.hide();
                }
            }
        });
    }
});
