<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

affl_import(AFFILILABS_PLUGIN_CONSTANTS_PATH);
affl_import(AFFILILABS_SERVICE_MARKETPLACES);

if (! class_exists('AffiliLabs')) {
  /**
   * The main class for the AffiliLabs plugin.
   * It initializes the plugin and registers its settings.
   * It also renders the menu pages.
   * 
   * @package AffiliLabs
   */
  class AffiliLabs {

      public function __construct() {
          add_action('admin_menu', array($this, 'add_menus'));
          add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
          add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts'));
          add_action('save_post', 'AFFLMarketplaceService::update_counter_on_editor');
          add_action('delete_post', 'AFFLMarketplaceService::update_counter_on_editor');
          add_filter('the_content', array($this, 'inject_single_product_template_content'));
          add_action('template_redirect', array($this, 'redirect_to_custom_product_template'));
      }

      /**
       * Plugin activation hook callback.
       * Sets up the initial supported marketplaces if they don't exist.
       */
      public static function activate_plugin() {
          try {
              affl_import(AFFILILABS_PLUGIN_CONSTANTS_PATH);
              affl_import(AFFILILABS_SERVICES_LOGGER_FILE);
              affl_import(AFFILILABS_SERVICE_MARKETPLACES);

              $result = AFFLMarketplaceService::create_default_marketplace('digistore');

              if (is_wp_error($result)) {
                  AFFL_LoggerService::log('Error during plugin activation (Digistore marketplace creation): ' . esc_html($result->get_error_message()), 'ERROR');
              } else if ($result === false) {
                  AFFL_LoggerService::log('Digistore marketplace creation skipped or failed during activation.', 'WARNING');
              } else {
                  AFFL_LoggerService::debug('Digistore marketplace activation process completed.', 'INFO');
              }

          } catch (Throwable $e) {
              if (class_exists('AFFL_LoggerService')) {
                  AFFL_LoggerService::log('Exception during initial Digistore24 marketplace creation: ' . esc_html($e->getMessage()), 'CRITICAL');
              } else {
                  error_log('[AffiliLabs Activation Error] Exception creating Digistore24 marketplace: ' . $e->getMessage() . ' Trace: ' . $e->getTraceAsString());
              }
          }
      }



      function add_menus() {
          add_menu_page("Dashboard | " . AFFILILABS_PLUGIN_NAME, AFFILILABS_PLUGIN_NAME, "manage_options", AFFILILABS_SLUG, array($this, "render_dashboard"), AFFILILABS_ASSETS_LOGO_LIGHT_22PX, 25);

          // Classic menu - hidden from top-level but used as parent for CPTs
          add_menu_page("AffiliLabs Classic | " . AFFILILABS_PLUGIN_NAME, AFFILILABS_PLUGIN_NAME . " Classic", "manage_options", AFFILILABS_CLASSIC_SLUG, function() {}, "dashicons-superhero", 25);

          $product_categories_submenu = array(
            "All Product Categories",
            "manage_options",
            "edit-tags.php?taxonomy=" . AFFILILABS_TAXONOMY_PRODUCT_CATEGORY . "&post_type=" . AFFILILABS_CPT_PRODUCT
          );

          global $submenu;
          // classic menu_position 2
          array_splice( $submenu[AFFILILABS_CLASSIC_SLUG], 1, 0, array( $product_categories_submenu ) );

          add_submenu_page(AFFILILABS_SLUG, "Dashboard | " . AFFILILABS_PLUGIN_NAME, "Dashboard", "manage_options", AFFILILABS_SLUG, array($this, "render_dashboard"));
          
          add_submenu_page(AFFILILABS_SLUG, "Marketplaces | " . AFFILILABS_PLUGIN_NAME, "Marketplaces", "manage_options", AFFILILABS_SLUG . "-marketplaces", array($this, "render_marketplaces"));

          add_submenu_page(AFFILILABS_SLUG, "Marketplace Edit | " . AFFILILABS_PLUGIN_NAME, "Marketplace Edit", "manage_options", AFFILILABS_SLUG . "-marketplace-edit", array($this, "render_marketplace_edit"));

          add_submenu_page(AFFILILABS_SLUG, "Product Edit | " . AFFILILABS_PLUGIN_NAME, "Product Edit", "manage_options", AFFILILABS_SLUG . "-product-edit", array($this, "render_product_edit"));
          
          add_submenu_page(AFFILILABS_SLUG, "Categories | " . AFFILILABS_PLUGIN_NAME, "Categories", "manage_options", AFFILILABS_SLUG . "-product-categories", array($this, "render_product_categories"));
          
          add_submenu_page(AFFILILABS_SLUG, "Products | " . AFFILILABS_PLUGIN_NAME, "Products", "manage_options", AFFILILABS_SLUG . "-products", array($this, "render_products"));
          
          add_submenu_page(AFFILILABS_SLUG, "Category Edit | " . AFFILILABS_PLUGIN_NAME, "Category Edit", "manage_options", AFFILILABS_SLUG . "-product-category-edit", array($this, "render_product_category_edit"));


          add_submenu_page(AFFILILABS_SLUG, "Criteria | " . AFFILILABS_PLUGIN_NAME, "Criteria", "manage_options", AFFILILABS_SLUG . "-criteria", array($this, "render_criteria"));
 
          add_submenu_page(AFFILILABS_SLUG, "Criterion Edit | " . AFFILILABS_PLUGIN_NAME, "Criterion Edit", "manage_options", AFFILILABS_SLUG . "-criteria-edit", array($this, "render_criteria_edit"));
 
          add_submenu_page(AFFILILABS_SLUG, "Lists | " . AFFILILABS_PLUGIN_NAME, "Lists", "manage_options", AFFILILABS_SLUG . "-lists", array($this, "render_lists"));
          
          add_submenu_page(AFFILILABS_SLUG, "List Edit | " . AFFILILABS_PLUGIN_NAME, "List Edit", "manage_options", AFFILILABS_SLUG . "-list-edit", array($this, "render_list_edit"));

          add_submenu_page(AFFILILABS_SLUG, "Templates | " . AFFILILABS_PLUGIN_NAME, "Templates", "manage_options", AFFILILABS_SLUG . "-templates", array($this, "render_templates"));

          add_submenu_page(AFFILILABS_SLUG, "Template Editor | " . AFFILILABS_PLUGIN_NAME, "Template Editor", "manage_options", AFFILILABS_SLUG . "-template-editor", array($this, "render_template_editor"));

          add_submenu_page(AFFILILABS_SLUG, "Classic Links | " . AFFILILABS_PLUGIN_NAME, "Classic Links", "manage_options", AFFILILABS_SLUG . "-classic-links", array($this, "render_classic_links"));

          add_submenu_page(AFFILILABS_SLUG, "About Us | " . AFFILILABS_PLUGIN_NAME, "About Us", "manage_options", AFFILILABS_SLUG . "-about-us", array($this, "render_about_us"));

          add_submenu_page(AFFILILABS_SLUG, "Logs Visualizer | " . AFFILILABS_PLUGIN_NAME, "Logs", "manage_options", AFFILILABS_SLUG . "-logs", array($this, "render_logs_page"));

          add_submenu_page(AFFILILABS_SLUG, "Settings | " . AFFILILABS_PLUGIN_NAME, "Settings", "manage_options", AFFILILABS_SLUG . "-settings", array($this, "render_settings"));

          add_filter('plugin_action_links_' . AFFILILABS_PLUGIN_BASENAME, array($this, 'add_plugin_settings_link'));
          add_filter('submenu_file', array($this, 'hide_submenu_entries'));
          add_action( 'parent_file', array( $this, 'highlight_parent_menu' ));
          
          // Hide the Classic menu from the admin sidebar (but keep it for CPTs)
          remove_menu_page(AFFILILABS_CLASSIC_SLUG);
      }

      function add_plugin_settings_link($links) {
          $settings_link = '<a href="' . esc_url(admin_url('admin.php?page=' . AFFILILABS_SLUG)) . '">Dashboard</a>';
          $links[] = $settings_link;
          return $links;
      }

      function hide_submenu_entries($submenu_file) {
        global $plugin_page;

        $hidden_submenus = array(
            AFFILILABS_SLUG . "-product-edit" => true,
            AFFILILABS_SLUG . "-marketplace-edit" => true,
            AFFILILABS_SLUG . "-criteria-edit" => true,
            AFFILILABS_SLUG . "-list-edit" => true,
            AFFILILABS_SLUG . "-template-editor" => true,
            AFFILILABS_SLUG . "-product-category-edit" => true,
        );
    
        // highlight another submenu
        if ( $plugin_page && isset( $hidden_submenus[ $plugin_page ] ) ) {
          switch ($plugin_page) {
            case AFFILILABS_SLUG . "-product-edit":
                $submenu_file = AFFILILABS_SLUG . "-products";
                break;
            case AFFILILABS_SLUG . "-marketplace-edit":
                $submenu_file = AFFILILABS_SLUG . "-marketplaces";
                break;
            //     break;
            case AFFILILABS_SLUG . "-criteria-edit":
                $submenu_file = AFFILILABS_SLUG . "-criteria";
                break;
            case AFFILILABS_SLUG . "-list-edit":
                $submenu_file = AFFILILABS_SLUG . "-lists";
                break;
            case AFFILILABS_SLUG . "-template-editor":
                $submenu_file = AFFILILABS_SLUG . "-templates";
                break;
            case AFFILILABS_SLUG . "-product-category-edit":
                $submenu_file = AFFILILABS_SLUG . "-product-categories";
                break;
          }
        }
    
        foreach ( $hidden_submenus as $submenu => $_ ) {
            remove_submenu_page(AFFILILABS_SLUG , $submenu );
        }
    
        return $submenu_file;
      }

      function highlight_parent_menu($parent_file) {
        global $current_screen, $pagenow, $submenu_file;

        $is_current_page_product_categories = (
        $pagenow == 'edit-tags.php' 
        && $current_screen->post_type == AFFILILABS_CPT_PRODUCT
        && $current_screen->taxonomy == AFFILILABS_TAXONOMY_PRODUCT_CATEGORY
        );

        if ($is_current_page_product_categories) {
            $parent_file = AFFILILABS_CLASSIC_SLUG;
            $submenu_file = "edit-tags.php?taxonomy=" . AFFILILABS_TAXONOMY_PRODUCT_CATEGORY . "&post_type=" . AFFILILABS_CPT_PRODUCT;
        }
        
        return $parent_file;
      }

      function enqueue_admin_scripts() {
        wp_enqueue_style('affililabs-common', AFFILILABS_COMMON_BASE_CSS_URL, array(), AFFILILABS_VERSION);
      }

      function enqueue_frontend_scripts() {
        if ( is_singular( AFFILILABS_CPT_PRODUCT ) ) {
            wp_enqueue_style(
                'affililabs-single-product',
                AFFILILABS_SINGLE_PRODUCT_TEMPLATE_CSS_URL,
                array(),
                AFFILILABS_VERSION
            );
            
            // Logic to add inline styles/scripts from custom template
            affl_import(AFFILILABS_SERVICE_OPTIMIZEPRESS_FILE);
            if (class_exists('Affililabs_Optimizepress_Integration')) {
                $integration = new Affililabs_Optimizepress_Integration();
                $op3_present = $integration->is_op3_present();
                $template_enabled = AFFLSettingsService::get_enable_custom_template();

                if (!$op3_present && $template_enabled) {
                    $product_id = get_the_ID();
                    $template_id = get_post_meta($product_id, AFFILILABS_META_KEY_APPLIED_PRODUCT_TEMPLATE_ID, true);
                }
            }
        }
        wp_enqueue_style(
            'affl-product-fields',
            AFFILILABS_SHORTCODE_PRODUCT_FIELDS_CSS_URL,
            array(),
            AFFILILABS_VERSION
        );
      }

      function render_dashboard() {
          affl_import(AFFILILABS_DASHBOARD_FILE);
      }

      function render_marketplaces() {
          affl_import(AFFILILABS_MARKETPLACES_FILE);
      }
      
      function render_marketplace_edit() {
          affl_import(AFFILILABS_MARKETPLACE_EDIT_FILE);
      }

      
      function render_product_edit() {
          affl_import(AFFILILABS_PRODUCTS_EDIT_FILE);
      }

      
      function render_products() {
          affl_import(AFFILILABS_PRODUCTS_FILE);
      }

      
      function render_product_categories() {
          affl_import(AFFILILABS_PRODUCT_CATEGORIES_FILE);
      }
      
      function render_product_category_edit() {
          affl_import(AFFILILABS_PRODUCT_CATEGORY_EDIT_FILE);
      }


      function render_criteria() {
        affl_import(AFFILILABS_CRITERIA_FILE);
      }
 
      function render_criteria_edit() {
        affl_import(AFFILILABS_CRITERIA_EDIT_FILE);
      }
 
      function render_lists() {
        affl_import(AFFILILABS_LISTS_FILE);
      }
 
      function render_list_edit() {
        affl_import(AFFILILABS_LIST_EDIT_FILE);
      }

      function render_templates() {
          affl_import(AFFILILABS_TEMPLATES_FILE);
      }

      function render_template_editor() {
          affl_import(AFFILILABS_TEMPLATE_EDITOR_FILE);
      }

      function render_about_us() {
          affl_import(AFFILILABS_ABOUT_US_FILE);
      }

      function render_classic_links() {
          affl_import(AFFILILABS_CLASSIC_LINKS_FILE);
      }

      
      function render_settings() {
          affl_import(AFFILILABS_SETTINGS_FILE);
      }

      function render_logs_page() {
          affl_import(AFFILILABS_LOG_PAGE_FILE);
      }


      public function inject_single_product_template_content( $content ) {
          if ( is_singular( AFFILILABS_CPT_PRODUCT ) && in_the_loop() && is_main_query() ) {
              $integration = new Affililabs_Optimizepress_Integration();
              $op3_present = $integration->is_op3_present();
              $template_enabled = AFFLSettingsService::get_enable_custom_template();

              if (!$op3_present && $template_enabled) {
                  $product_id = get_the_ID();
                  $template_id = get_post_meta($product_id, AFFILILABS_META_KEY_APPLIED_PRODUCT_TEMPLATE_ID, true);

                  if ( $template_id ) {
                      $template = get_post($template_id);
                      if ($template && $template->post_type === AFFILILABS_CPT_TEMPLATE) {
                          $template_modules = get_post_meta($template_id, AFFILILABS_META_KEY_TEMPLATE_DATA, true);
                          $output_html = '';

                          if (!empty($template_modules) && is_array($template_modules)) {
                              foreach ($template_modules as $module) {
                                  $shortcode_output = do_shortcode('[affl_' . $module . ' id="' . $product_id . '"]');

                                  if (!empty(trim($shortcode_output))) {
                                      $class_name = 'affl_' . str_replace('_', '-', $module);
                                      // Injecting the shortcode output into a div with a class for styling
                                      // The actual HTML structure will depend on how each shortcode is designed to render
                                      $output_html .= '<div class="' . esc_attr($class_name) . '">' . $shortcode_output . '</div>';
                                  }
                              }
                          }
                          
                          $allowed_html = affl_get_allowed_html_tags();
                          return wp_kses($output_html, $allowed_html);
                      }
                  }
              }
          }
          return $content;
      }

      public function redirect_to_custom_product_template() {
          if (is_singular(AFFILILABS_CPT_PRODUCT)) {
              $integration = new Affililabs_Optimizepress_Integration();
              if ($integration->is_op3_present() || !AFFLSettingsService::get_enable_custom_template()) {
                  return;
              }
              
              $product_id = get_the_ID();
              $template_id = get_post_meta($product_id, AFFILILABS_META_KEY_APPLIED_PRODUCT_TEMPLATE_ID, true);
      
              if (!$template_id) {
                  $new_template = AFFILILABS_SINGLE_PRODUCT_TEMPLATE_FILE;
                  if (file_exists($new_template)) {
                      include($new_template);
                      exit();
                  }
              }
          }
      }
  }

}
