<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

affl_import(AFFILILABS_PLUGIN_CONSTANTS_PATH);

class AffiliLabs_VideoFrame {

    /**
     * Extracts the YouTube video ID from various URL formats.
     *
     * @param string $url The YouTube video URL or ID.
     * @return string|null The extracted video ID, or null if not found.
     */
    private function get_youtube_video_id($url) {
        if (empty($url)) {
            return null;
        }

        $video_id_regex = '/^[a-zA-Z0-9_-]{11}$/';
        if (preg_match($video_id_regex, $url)) {
            return $url;
        }

        $full_youtube_url_regex = '/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/i';
        if (preg_match($full_youtube_url_regex, $url, $matches)) {
            return $matches[1];
        }

        $embed_youtube_url_regex = '/youtube\.com\/embed\/([a-zA-Z0-9_-]{11})/i';
        if (preg_match($embed_youtube_url_regex, $url, $matches)) {
            return $matches[1];
        }

        return null;
    }

    /**
     * Renders the video frame component.
     *
     * @param string|null $url The YouTube video URL.
     * @return void
     */
    public function render($url = null) {
        $video_id = $this->get_youtube_video_id($url);
        ?>
        <div class="affililabs-video-frame">
            <?php if ($video_id) : ?>
                <iframe
                    class="affililabs-video-frame__iframe"
                    src="https://www.youtube.com/embed/<?php affl_print(esc_attr($video_id)); ?>"
                    frameborder="0"
                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share"
                    allowfullscreen
                    width="100%"
                    height="100%"
                ></iframe>
            <?php else : ?>
                <img
                    class="affililabs-video-frame__placeholder"
                    src="<?php affl_print(AFFILILABS_ASSETS_LOGO_DARK); ?>"
                    alt="<?php affl_print(esc_attr(__('AffiliLabs Logo Placeholder', 'affililabs'))); ?>"
                >
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Enqueues scripts and styles for the VideoFrame component.
     *
     * @return void
     */
    public function load_scripts() {
        wp_enqueue_style('affililabs-video-frame', AFFILILABS_VIDEO_FRAME_COMPONENT_CSS_URL, array(), AFFILILABS_VERSION, 'all');
    }

    /**
     * Initializes the VideoFrame component by adding necessary actions.
     *
     * @return void
     */
    public function init() {
        add_action('admin_enqueue_scripts', array($this, 'load_scripts'));
    }
}