<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

affl_import(AFFILILABS_PLUGIN_CONSTANTS_PATH);

class AffiliLabs_Spinner {

  const TYPE_BLOCKS = 'blocks';
  const TYPE_WIFI = 'wifi';
  const TYPE_PULSE = 'pulse';
  const TYPE_RANDOM = 'random';

  public function load_scripts() {
    wp_enqueue_style('affililabs-spinner-css', AFFILILABS_SPINNER_COMPONENT_CSS_URL, array(), AFFILILABS_VERSION, 'all');
    wp_enqueue_script('affililabs-spinner-js', AFFILILABS_SPINNER_COMPONENT_JS_URL, array('jquery'), AFFILILABS_VERSION, true);
  }

  /**
   * Get SVG content safely
   * 
   * @param string $svg_path Path to SVG file
   * @return string SVG content
   */
  private function get_svg_content($svg_path) {
    if (file_exists($svg_path)) {
      $svg_content = file_get_contents($svg_path);
      if (!empty($svg_content)) {
        return $svg_content;
      }
    }
    
    return '<svg viewBox="0 0 24 24"><circle cx="12" cy="12" r="10" stroke="#1e3b8a" stroke-width="2" fill="none"></circle></svg>';
  }

  /**
   * Renders the spinner component container.
   * This method outputs the HTML structure for the spinner overlay.
   * 
   * @param string $type The spinner type (blocks, wifi, pulse, random)
   * @return void
   */
  public function render($type = self::TYPE_RANDOM) { 
    $spinner_type = $type;
    
    if ($spinner_type === self::TYPE_RANDOM) {
      $spinner_types = array(self::TYPE_BLOCKS, self::TYPE_WIFI, self::TYPE_PULSE);
      $spinner_type = $spinner_types[array_rand($spinner_types)];
    }
    
    $blocks_svg_path = str_replace(AFFILILABS_PLUGIN_URL, AFFILILABS_PLUGIN_PATH, AFFILILABS_ASSETS_SPINNER_BLOCKS_SVG);
    $wifi_svg_path = str_replace(AFFILILABS_PLUGIN_URL, AFFILILABS_PLUGIN_PATH, AFFILILABS_ASSETS_SPINNER_WIFI_SVG);
    $pulse_svg_path = str_replace(AFFILILABS_PLUGIN_URL, AFFILILABS_PLUGIN_PATH, AFFILILABS_ASSETS_SPINNER_PULSE_SVG);
    
    $blocks_svg = $this->get_svg_content($blocks_svg_path);
    $wifi_svg = $this->get_svg_content($wifi_svg_path);
    $pulse_svg = $this->get_svg_content($pulse_svg_path);
    
    $current_svg = $blocks_svg;
    if ($spinner_type === self::TYPE_WIFI) {
      $current_svg = $wifi_svg;
    } elseif ($spinner_type === self::TYPE_PULSE) {
      $current_svg = $pulse_svg;
    }

    ?>
    <div id="affililabs-spinner" class="affililabs-spinner">
      <div class="affililabs-spinner__overlay"></div>
      <button class="affililabs-spinner__close-btn affililabs-tooltip" aria-label="<?php affl_print(esc_attr(__('Close spinner', 'affililabs'))); ?>">
        X
      </button>
      <div class="affililabs-spinner__container">
        <div class="affililabs-spinner__icon" data-spinner-type="<?php affl_print(esc_attr($spinner_type)); ?>">
          <?php $this->print_svg($current_svg); ?>
        </div>
        <div class="affililabs-spinner__text"></div>
        <button class="affililabs-spinner__close-button-bottom affililabs-plugin__button affililabs-plugin__button--secondary"><?php affl_print(esc_html__('Close', 'affililabs')); ?></button>
      </div>
      <div style="display: none;">
        <div data-svg-type="blocks"><?php $this->print_svg($blocks_svg); ?></div>
        <div data-svg-type="wifi"><?php $this->print_svg($wifi_svg); ?></div>
        <div data-svg-type="pulse"><?php $this->print_svg($pulse_svg); ?></div>
      </div>
    </div>
  <?php
  }

  /**
   * Prints sanitized SVG content.
   * 
   * @param string $svg The SVG content to print.
   */
  private function print_svg($svg) {
    if (empty($svg)) return;

    $allowed_tags = array(
      'svg' => array(
        'xmlns'   => true,
        'viewbox' => true,
        'width'   => true,
        'height'  => true,
        'style'   => true,
        'class'   => true,
        'id'      => true,
      ),
      'path' => array(
        'd'       => true,
        'fill'    => true,
        'stroke'  => true,
        'opacity' => true,
        'class'   => true,
      ),
      'rect' => array(
        'x'      => true,
        'y'      => true,
        'width'  => true,
        'height' => true,
        'fill'   => true,
        'class'  => true,
        'rx'     => true,
        'ry'     => true,
      ),
      'circle' => array(
        'cx'     => true,
        'cy'     => true,
        'r'      => true,
        'fill'   => true,
        'stroke' => true,
      ),
      'style' => array(),
    );

    print wp_kses($svg, $allowed_tags);
  }

  /**
   * Initialize the spinner component.
   * Attaches the spinner to the admin interface.
   */
  public function init() {
    add_action('admin_footer', array($this, 'render'));
  }
}
