/**
 * AffiliLabs Modal Component
 *
 * This script provides functionality for showing modal dialogs in the AffiliLabs plugin.
 * It handles different types of modals: notice, confirm, error, and success.
 */
jQuery(document).ready(function($) {
  const $modal = $('#affililabs-modal');
  const $overlay = $('.affililabs-modal__overlay');
  const $container = $('.affililabs-modal__container');
  const $close_button = $('.affililabs-modal__close');
  const $title = $('.affililabs-modal__title');
  const $content = $('.affililabs-modal__content');
  const $buttons_container = $('.affililabs-modal__buttons');
  
  let is_initialized = false;
  
  const i18n = affililabs_modal.i18n || {};
  const colors = affililabs_modal.colors || {};
  function init_modal() {
    if (is_initialized) return;
    
    const $main = $('.affililabs-plugin__main');
    if ($main.length && !$modal.closest('.affililabs-plugin__main').length) {
      $modal.appendTo($main);
    } else if (!$modal.closest('main').length) {
      const $outermost_main = $('main').first();
      if ($outermost_main.length) {
        $modal.appendTo($outermost_main);
      }
    }
    
    $close_button.on('click', close_modal);
    
    $overlay.on('click', close_modal);
    
    $(document).on('keydown', function(e) {
      if (e.key === 'Escape' && $modal.hasClass('affililabs-modal--show')) {
        close_modal();
      }
    });
    
    is_initialized = true;
  }
  
  /**
   * Show a modal with the given parameters
   *
   * @param {Object} options - Modal options
   * @param {string} options.type - Modal type (notice, confirm, error, success)
   * @param {string} options.title - Modal title
   * @param {string} options.message - Modal message content
   * @param {Object} options.buttons - Button configuration
   * @param {Function} options.on_confirm - Callback when confirmed
   * @param {Function} options.on_cancel - Callback when canceled
   */
  function show_modal(options) {
    init_modal();
    
    const type = options.type || 'notice';
    const title = options.title || '';
    const message = options.message || '';
    const buttons = options.buttons || {};
    const on_confirm = options.on_confirm || function() {};
    const on_cancel = options.on_cancel || function() {};
    
    $modal.removeClass('affililabs-modal--notice affililabs-modal--confirm affililabs-modal--error affililabs-modal--success');
    $buttons_container.empty();
    
    $modal.addClass(`affililabs-modal--${type}`);
    
    $title.text(title);
    $content.html(message);
    
    if (type === 'notice' || type === 'error' || type === 'success') {
      const ok_text = buttons.ok || i18n.ok || 'OK';
      const $ok_button = $('<button>', {
        class: `affililabs-modal__button affililabs-modal__button--${type === 'error' ? 'danger' : (type === 'success' ? 'success' : 'primary')}`,
        text: ok_text
      }).on('click', function() {
        close_modal();
        on_confirm();
      });
      
      $buttons_container.append($ok_button);
    } else if (type === 'confirm') {
      const confirm_text = buttons.confirm || i18n.confirm || 'Confirm';
      const cancel_text = buttons.cancel || i18n.cancel || 'Cancel';
      
      const $confirm_button = $('<button>', {
        class: 'affililabs-modal__button affililabs-modal__button--primary',
        text: confirm_text
      }).on('click', function() {
        close_modal();
        on_confirm();
      });
      
      const $cancel_button = $('<button>', {
        class: 'affililabs-modal__button affililabs-modal__button--neutral',
        text: cancel_text
      }).on('click', function() {
        close_modal();
        on_cancel();
      });
      
      $buttons_container.append($cancel_button, $confirm_button);
    }
    
    $modal.addClass('affililabs-modal--show');
  }
  
  function close_modal() {
    $modal.removeClass('affililabs-modal--show');
  }
  
  /**
   * Show a notice modal
   *
   * @param {string} title - Modal title
   * @param {string} message - Modal message
   * @param {Object} options - Additional options
   */
  function show_notice(title, message, options = {}) {
    show_modal({
      type: 'notice',
      title: title,
      message: message,
      buttons: options.buttons,
      on_confirm: options.on_confirm
    });
  }
  
  /**
   * Show a confirmation modal
   *
   * @param {string} title - Modal title
   * @param {string} message - Modal message
   * @param {Function} on_confirm - Callback when confirmed
   * @param {Object} options - Additional options
   */
  function show_confirm(title, message, on_confirm, options = {}) {
    show_modal({
      type: 'confirm',
      title: title,
      message: message,
      buttons: options.buttons,
      on_confirm: on_confirm,
      on_cancel: options.on_cancel
    });
  }
  
  /**
   * Show an error modal
   *
   * @param {string} title - Modal title
   * @param {string} message - Modal message
   * @param {Object} options - Additional options
   */
  function show_error(title, message, options = {}) {
    show_modal({
      type: 'error',
      title: title || i18n.error || 'Error',
      message: message,
      buttons: options.buttons,
      on_confirm: options.on_confirm
    });
  }
  
  /**
   * Show a success modal
   *
   * @param {string} title - Modal title
   * @param {string} message - Modal message
   * @param {Object} options - Additional options
   */
  function show_success(title, message, options = {}) {
    show_modal({
      type: 'success',
      title: title || i18n.success || 'Success',
      message: message,
      buttons: options.buttons,
      on_confirm: options.on_confirm
    });
  }
  
  /**
   * Show a delete confirmation modal
   *
   * @param {string} message - Custom message (optional)
   * @param {Function} on_confirm - Callback when confirmed
   * @param {Object} options - Additional options
   */
  function show_delete_confirm(message, on_confirm, options = {}) {
    const default_message = i18n.confirm_delete + ' ' + i18n.action_cannot_be_undone;
    
    show_modal({
      type: 'confirm',
      title: i18n.warning || 'Warning',
      message: message || default_message,
      buttons: {
        confirm: i18n.delete || 'Delete',
        cancel: i18n.cancel || 'Cancel'
      },
      on_confirm: on_confirm,
      on_cancel: options.on_cancel
    });
  }

  window.affililabs_modal = {
    show: show_modal,
    notice: show_notice,
    confirm: show_confirm,
    error: show_error,
    success: show_success,
    delete_confirm: show_delete_confirm,
    close: close_modal
  };
}); 
