import React from 'react';
import ReactDOM from 'react-dom';
import UpgradeModal from '../blocks/ui-components/UpgradeModal';
import { __ } from '@wordpress/i18n';
import { initElementorTemplateLibrary } from './TemplateLibraryModalElementor';

( ( $ ) => {
	'use strict';

	const ElementorAmazonFree = {
		// Initialize the plugin.
		init() {
			if ( window.AffiliateX && window.AffiliateX.proActive !== 'true' ) {
				ElementorAmazonFree.initUpgradeModal();
			}

			ElementorAmazonFree.bindEvents();
			ElementorAmazonFree.initTemplateLibrary();
		},

		// Bind event listeners.
		bindEvents() {
			if ( window.AffiliateX && window.AffiliateX.proActive !== 'true' ) {
				$( document ).on( 'click', '.affx-action-button__amazon', ElementorAmazonFree.triggerUpgradeModal );
				$( document ).on( 'click', '.affx-connect-all-wrapper', ElementorAmazonFree.triggerUpgradeModal );
			}

			$( window ).on( 'elementor:init', ElementorAmazonFree.addConnectAllButton );
			$( window ).on( 'elementor:init', ElementorAmazonFree.addTemplateLibraryButton );

			$( document ).on( 'click', '.affx-template-library-button', ElementorAmazonFree.openTemplateLibrary );

			// Ignore wp-auth-check.js error
			$( document ).on( 'error', function ( e ) {
				if ( e.originalEvent?.filename?.includes( 'wp-auth-check.js' ) ) {
					return false;
				}
			} );
		},

		// Initialize upgrade modal.
		initUpgradeModal() {
			$( 'body' ).append( '<div id="affx-upgrade-modal-root"></div>' );

			const upgradeRootElement = document.getElementById( 'affx-upgrade-modal-root' );

			if ( upgradeRootElement && ! upgradeRootElement.hasChildNodes() ) {
				ReactDOM.createRoot( upgradeRootElement ).render( <UpgradeModal /> );
			}
		},

		// Initialize template library.
		initTemplateLibrary() {
			$( window ).on( 'elementor:init', () => {
				// Wait for Elementor to be fully loaded
				setTimeout( () => {
					initElementorTemplateLibrary();
				}, 1000 );
			} );
		},

		openTemplateLibrary( e ) {
			e.preventDefault();

			let widgetType = null;
			let widgetModel = null;
			try {
				const currentPageView = elementor?.getPanelView()?.getCurrentPageView();
				const model = currentPageView?.model;
				if ( model && model.get( 'widgetType' ) ) {
					widgetType = model.get( 'widgetType' ).replace( 'affiliatex-', '' );
					widgetModel = model;
				}
			} catch ( error ) {
				console.error( 'Error getting widget type:', error );
			}

			if ( window.AffiliateXElementorTemplateLibrary ) {
				window.AffiliateXElementorTemplateLibrary.open( widgetType, widgetModel );
			} else {
				console.error( 'Template library not initialized' );
			}
		},

		// Trigger modal
		triggerUpgradeModal() {
			window.wp.data.dispatch( 'affiliatex' ).setActiveModal( 'upgrade-modal' );
			window.wp.data.dispatch( 'affiliatex' ).setUpgradeModal( {
				modalType: 'amazon',
				modalTitle: __( 'Amazon Integration', 'affiliatex' ),
				blockTitle: __( 'Amazon Integration', 'affiliatex' )
			} );
		},

		// Handle the active widget.
		addConnectAllButton() {
			// List of widgets that don't need the connect all button.
			const excludeWidgets = [
				'affiliatex-product-comparison',
				'affiliatex-product-table',
				'affiliatex-pros-and-cons',
				'affiliatex-specifications',
				'affiliatex-verdict',
				'affiliatex-versus-line',
				'affiliatex-versus',
				'affiliatex-dynamic-listing'
			];

			const initiallyHiddenWidgets = [ 'affiliatex-top-products', 'affiliatex-coupon-listing', 'affiliatex-coupon-grid' ];

			const addButton = ( panel, model ) => {
				if ( panel && model ) {
					if (
						model?.get( 'widgetType' ) &&
						model.get( 'widgetType' ).includes( 'affiliatex' ) &&
						! excludeWidgets.includes( model.get( 'widgetType' ) )
					) {
						if ( panel.getOption( 'tab' ) === 'content' ) {
							const navigationPanel = $( panel.el ).find( '.elementor-panel-navigation' );
							if ( $( panel.el ).find( '.affx-connect-all-wrapper' ).length === 0 ) {
								navigationPanel.after( AffiliateX.connectAllButton );
								if ( initiallyHiddenWidgets.includes( model.get( 'widgetType' ) ) ) {
									$( panel.el ).find( '.affx-connect-all-wrapper' ).hide();
								}
							}
						}
					}
				}
			};

			// Triggers when a widget is activated.
			elementor.hooks.addAction( 'panel/open_editor/widget', addButton );

			const processedViews = new WeakSet();

			elementor.hooks.addFilter( 'controls/base/behaviors', ( behaviors ) => {
				const currentPageView = elementor.getPanelView().getCurrentPageView();

				// Only add button once per unique PageView
				if ( currentPageView && ! processedViews.has( currentPageView ) ) {
					processedViews.add( currentPageView );
					addButton( currentPageView, currentPageView.model );
				}

				return behaviors;
			} );
		},

		addTemplateLibraryButton() {
			// Only add the button if Elementor templates are available
			if ( window.AffiliateX && window.AffiliateX.hasElementorTemplates !== 'true' ) {
				return;
			}

			const affxWidgets = [
				'affiliatex-buttons',
				'affiliatex-cta',
				'affiliatex-notice',
				'affiliatex-product-comparison',
				'affiliatex-product-table',
				'affiliatex-pros-and-cons',
				'affiliatex-single-product',
				'affiliatex-specifications',
				'affiliatex-verdict',
				'affiliatex-versus-line',
				'affiliatex-dynamic-listing'
			];

			const addButton = ( panel, model ) => {
				if ( panel && model ) {
					if ( model?.get( 'widgetType' ) && affxWidgets.includes( model.get( 'widgetType' ) ) ) {
						if ( panel.getOption( 'tab' ) === 'content' ) {
							const navigationPanel = $( panel.el ).find( '.elementor-panel-navigation' );
							// Add button after navigation if it doesn't exist
							if ( $( panel.el ).find( '.affx-template-library-button-wrapper' ).length === 0 ) {
								navigationPanel.after( AffiliateX.templateLibraryButton );
							}
						}
					}
				}
			};

			elementor.hooks.addAction( 'panel/open_editor/widget', addButton );
			const processedViews = new WeakSet();

			elementor.hooks.addFilter( 'controls/base/behaviors', ( behaviors ) => {
				const currentPageView = elementor.getPanelView().getCurrentPageView();
				if ( currentPageView && ! processedViews.has( currentPageView ) ) {
					processedViews.add( currentPageView );
					addButton( currentPageView, currentPageView.model );
				}

				return behaviors;
			} );
		}
	};

	ElementorAmazonFree.init();
} )( jQuery );
