<?php
defined('ABSPATH') || exit;
const AEGIS_SHIELD_BLOCKED_IPS_FILE = AEGIS_SHIELD_LOG_DIR . 'blocked-ips.json';

//  Fallback storage for blocked IPs
function aegis_shield_get_blocked_ips_fallback() {
    if (!file_exists(AEGIS_SHIELD_BLOCKED_IPS_FILE)) return [];
    $json = file_get_contents(AEGIS_SHIELD_BLOCKED_IPS_FILE);
    return json_decode($json, true) ?? [];
}

function aegis_shield_save_blocked_ips_fallback($ips) {
    wp_mkdir_p(dirname(AEGIS_SHIELD_BLOCKED_IPS_FILE));
    file_put_contents(AEGIS_SHIELD_BLOCKED_IPS_FILE, json_encode(array_values(array_unique($ips))));
}

//  Middleware-based IP blocking
add_action('init', function () {
    $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
    $ip = filter_var($ip, FILTER_VALIDATE_IP);
    if (!$ip) {
        return;
    }

    $blocked_ips = aegis_shield_get_blocked_ips_fallback();
    if (in_array($ip, $blocked_ips, true)) {
        status_header(403);
        exit('Access denied.');
    }
});


// Log failed logins
add_action('wp_login_failed', function ($username) {
    $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
    $ip = filter_var($ip, FILTER_VALIDATE_IP);

    if (!$ip) {
        return;
    }

    $timestamp = current_time('mysql');
    $log_entry = [
        'username' => sanitize_text_field($username),
        'ip'       => $ip,
        'time'     => $timestamp,
    ];

    $logs = get_option('aegis_shield_failed_logins', []);
    $logs[] = $log_entry;

    if (count($logs) > 100) {
        $logs = array_slice($logs, -100);
    }

    update_option('aegis_shield_failed_logins', $logs);
});


//  404 throttling
add_action('template_redirect', function () {
    if (is_404()) {
        $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
        $ip = filter_var($ip, FILTER_VALIDATE_IP);

        if (!$ip) {
            return;
        }

        $key = 'aegis_404_' . md5($ip);
        $timestamps = get_transient($key) ?: [];

        $now = time();
        $timestamps[] = $now;
        $timestamps = array_filter($timestamps, fn($t) => ($now - $t) < 60);

        if (count($timestamps) > 10) {
            aegis_shield_block_ip($ip);
            aegis_shield_save_blocked_ips_fallback(array_merge(aegis_shield_get_blocked_ips_fallback(), [$ip]));
        }

        set_transient($key, $timestamps, 2 * MINUTE_IN_SECONDS);
    }
});


//  Block IPs via .htaccess or fallback
function aegis_shield_block_ip($ip) {
    if (!filter_var($ip, FILTER_VALIDATE_IP)) return false;

    $blocked_ips = aegis_shield_get_blocked_ips_fallback();
    if (!in_array($ip, $blocked_ips, true)) {
        $blocked_ips[] = $ip;
        aegis_shield_save_blocked_ips_fallback($blocked_ips);
    }
    return true;
}


