<?php
// admin-ui.php with tabs

defined('ABSPATH') || exit;

/**
 * Register Aegis Shield admin menu.
 */
add_action('admin_menu', function () {
    add_menu_page(
        __('Aegis Shield', 'aegis-shield'),
        __('Aegis Shield', 'aegis-shield'),
        'manage_options',
        'aegis-shield',
        'aegis_shield_render_admin_page',
        'dashicons-shield-alt',
        80
    );
});

add_action('admin_enqueue_scripts', function ($hook) {
    if ($hook === 'toplevel_page_aegis-shield') {
        wp_enqueue_script('aegis-scan-ui', plugin_dir_url(__FILE__) . '/js/aegis-scan-ui.js', ['jquery'], '1.0', true);
        wp_localize_script('aegis-scan-ui', 'AegisAjax', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('aegis_run_scan')
        ]);
    }
});

add_action('wp_ajax_aegis_run_scan', function () {
    check_ajax_referer('aegis_run_scan');

    $results = aegis_shield_run_scan();
    ob_start();

    echo '<h2>' . esc_html__('Scan Results', 'aegis-shield') . '</h2>';

    echo '<p>' . sprintf(
        /* translators: %d is the number of PHP files scanned. */
            esc_html__('%d PHP files scanned.', 'aegis-shield'),
            esc_html($results['scanned_files'])
        ) . '</p>';

    if (!empty($results['suspicious_matches'])) {
        echo '<h3 style="color: red;">' . esc_html__('Suspicious Code Found:', 'aegis-shield') . '</h3><ul>';
        foreach ($results['suspicious_matches'] as $match) {
            echo '<li><strong>' . esc_html($match['file']) . '</strong><ul>';
            foreach ($match['matches'] as $m) {
                echo '<li>Line ' . esc_html($m['line']) . ': ' . esc_html($m['description']) . ' <code>' . esc_html($m['snippet']) . '</code></li>';
            }
            echo '</ul></li>';
        }
        echo '</ul>';
    } else {
        echo '<p style="color:green;">' . esc_html__('No suspicious code found.', 'aegis-shield') . '</p>';
    }

    if (!empty($results['integrity_issues'])) {
        echo '<h3>' . esc_html__('Integrity Issues:', 'aegis-shield') . '</h3><ul>';
        foreach ($results['integrity_issues'] as $issue) {
            echo '<li><strong>' . esc_html($issue['file']) . '</strong>: ' . esc_html($issue['issue']) . '</li>';
        }
        echo '</ul>';
    }

    wp_send_json_success(['html' => ob_get_clean()]);
});


function aegis_shield_render_admin_page() {
    $tabs = [
        'scan' => __('Scan', 'aegis-shield'),
        'ips' => __('IP Blocking', 'aegis-shield'),
        'cron' => __('Automation Settings', 'aegis-shield'),
    ];

    if (!defined('AEGIS_SHIELD_PRO')) {
        $tabs['get-pro'] = __('Get Pro', 'aegis-shield');
    }

    if (defined('AEGIS_SHIELD_PRO') && AEGIS_SHIELD_PRO === true) {
        $tabs['scan-history'] = __('Scan History', 'aegis-shield');
        $tabs['license'] = __('License', 'aegis-shield');
        $tabs['support'] = __('Support', 'aegis-shield');
    }

    $current_tab = isset($_GET['tab']) ? sanitize_key(wp_unslash($_GET['tab'])) : 'scan';
    $current_tab = array_key_exists($current_tab, $tabs) ? $current_tab : 'scan';
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Aegis Shield - Security Scanner', 'aegis-shield'); ?></h1>

        <h2 class="nav-tab-wrapper">
            <?php foreach ($tabs as $slug => $label): ?>
                <?php
                $tab_classes = 'nav-tab' . ($current_tab === $slug ? ' nav-tab-active' : '');

                // Support tab styling
                $custom_style = '';
                if ($slug === 'support' && defined('AEGIS_SHIELD_PRO') && AEGIS_SHIELD_PRO === true) {
                    $custom_style = 'background-color:#0073aa; color:white; border-color:#0073aa;';
                }

                // Get Pro tab styling
                if ($slug === 'get-pro' && (!defined('AEGIS_SHIELD_PRO') || AEGIS_SHIELD_PRO !== true)) {
                    $custom_style = 'background-color:#d63638; color:white; border-color:#d63638;';
                }
                ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=aegis-shield&tab=' . $slug)); ?>"
                   class="<?php echo esc_attr($tab_classes); ?>"
                    <?php echo $custom_style ? 'style="' . esc_attr($custom_style) . '"' : ''; ?>>
                    <?php echo esc_html($label); ?>
                </a>
            <?php endforeach; ?>
        </h2>

        <?php
        switch ($current_tab) {
            case 'scan':
                $has_scanned = false;
                $scan_dir = AEGIS_SHIELD_LOG_DIR . '/scans/';
                if (is_dir($scan_dir)) {
                    $scan_files = glob($scan_dir . '*.json');
                    $has_scanned = !empty($scan_files);
                }
                $logo_url = plugins_url('../assets/shield-logo-tp.png', __FILE__);
                ?>
                <div id="aegis-scan-ui" style="margin-top: 1.5em; background: #f8f9fa; padding: 2em; border-radius: 8px; border: 1px solid #dee2e6;">
                    <div style="display: flex; align-items: center; gap: 12px; margin-bottom: 1rem;">
                        <img src="<?php echo esc_url($logo_url); ?>" alt="Aegis Shield Logo" style="height: 48px; width: auto;">
                        <h2 style="font-size: 1.5rem; margin: 0; color: #0073aa;">
                            <?php echo $has_scanned ? 'Run Another Security Scan' : 'Welcome to Aegis Shield'; ?>
                        </h2>
                    </div>

                    <p style="font-size: 1rem; max-width: 700px; line-height: 1.6; color: #333;">
                        <?php if (!$has_scanned): ?>
                            Your journey to a more secure WordPress site starts here. By clicking the scan button below, Aegis Shield will perform an initial baseline scan of your entire site.
                            This scan helps us understand the current state of your files, and it forms the foundation for all future integrity checks.
                            <br><br>
                            Let’s lock things down and keep the bad guys out — one scan at a time.
                        <?php else: ?>
                            Welcome back! You're just one click away from making sure your site is still clean and secure.
                            This scan checks your site for any file changes or suspicious activity by comparing against your most recent scan.
                            Stay proactive — run scans regularly for maximum peace of mind.
                        <?php endif; ?>
                    </p>

                    <button id="aegis-run-scan" class="button button-primary" style="margin-top: 1.5em; padding: 0.75rem 2rem; font-size: 1rem;">
                        <?php esc_html_e('Run Scan Now', 'aegis-shield'); ?>
                    </button>

                    <div id="aegis-scan-progress" style="margin-top: 1.5em; display: none;">
                        <div style="background: #e9ecef; border-radius: 6px; overflow: hidden;">
                            <div id="aegis-scan-bar" style="width: 0%; background: #0073aa; height: 24px; transition: width 0.3s;"></div>
                        </div>
                        <p id="aegis-scan-status" style="margin-top: 0.5em; color: #555; font-style: italic;"></p>
                    </div>

                    <div id="aegis-scan-results" style="margin-top: 2em;"></div>
                </div>
                <?php
                break;



            case 'ips':
                // Update full IP list from textarea POST
                if (isset($_POST['aegis_shield_update_htaccess']) && check_admin_referer('aegis_shield_htaccess_update')) {
                    $raw_input = isset($_POST['blocked_ips']) ? sanitize_textarea_field(wp_unslash($_POST['blocked_ips'])) : '';
                    $raw_ips = array_filter(array_map('trim', explode("\n", $raw_input)));
                    $clean_ips = [];
                    foreach ($raw_ips as $ip) {
                        if (filter_var($ip, FILTER_VALIDATE_IP)) {
                            $clean_ips[] = $ip;
                        }
                    }
                    aegis_shield_save_blocked_ips_fallback($clean_ips);
                    echo '<div class="updated notice"><p>' . esc_html__('Blocked IP list updated successfully.', 'aegis-shield') . '</p></div>';
                }

                // Block a single IP from GET
                if (isset($_GET['block_ip']) && check_admin_referer('aegis_shield_block_ip')) {
                    $ip_to_block = sanitize_text_field(wp_unslash($_GET['block_ip']));
                    if (filter_var($ip_to_block, FILTER_VALIDATE_IP)) {
                        aegis_shield_block_ip($ip_to_block); // Handles saving to fallback
                        echo '<div class="updated notice"><p>' . esc_html__('IP blocked successfully.', 'aegis-shield') . '</p></div>';
                    }
                }

                // Remove (unblock) a single IP from GET
                if (isset($_GET['remove_ip']) && check_admin_referer('aegis_shield_remove_ip')) {
                    $ip_to_remove = sanitize_text_field(wp_unslash($_GET['remove_ip']));
                    if (filter_var($ip_to_remove, FILTER_VALIDATE_IP)) {
                        $blocked = aegis_shield_get_blocked_ips_fallback();
                        $updated = array_diff($blocked, [$ip_to_remove]);
                        aegis_shield_save_blocked_ips_fallback($updated);
                        echo '<div class="updated notice"><p>' . esc_html__('IP unblocked successfully.', 'aegis-shield') . '</p></div>';
                    }
                }

                // Delete a single failed login log entry
                if (isset($_GET['delete_log']) && check_admin_referer('aegis_shield_delete_log')) {
                    $index = (int) sanitize_text_field(wp_unslash($_GET['delete_log']));
                    $logs = get_option('aegis_shield_failed_logins', []);
                    if (isset($logs[$index])) {
                        unset($logs[$index]);
                        $logs = array_values($logs);
                        update_option('aegis_shield_failed_logins', $logs);
                        echo '<div class="updated notice"><p>' . esc_html__('Log entry deleted.', 'aegis-shield') . '</p></div>';
                    }
                }

                echo '<h3>' . esc_html__('Blocked IP Addresses', 'aegis-shield') . '</h3>';

                $blocked_ips = aegis_shield_get_blocked_ips_fallback();
                if (!empty($blocked_ips)) {
                    echo '<table class="widefat striped"><thead><tr><th>' . esc_html__('IP Address', 'aegis-shield') . '</th><th>' . esc_html__('Action', 'aegis-shield') . '</th></tr></thead><tbody>';
                    foreach ($blocked_ips as $ip) {
                        $remove_url = wp_nonce_url(admin_url('admin.php?page=aegis-shield&tab=ips&remove_ip=' . urlencode($ip)), 'aegis_shield_remove_ip');
                        echo '<tr><td>' . esc_html($ip) . '</td><td><a href="' . esc_url($remove_url) . '" class="button small button-danger">' . esc_html__('Unblock', 'aegis-shield') . '</a></td></tr>';
                    }
                    echo '</tbody></table>';
                } else {
                    echo '<p>' . esc_html__('No IP addresses are currently blocked.', 'aegis-shield') . '</p>';
                }

                echo '<hr><h3>' . esc_html__('Recent Failed Login Attempts', 'aegis-shield') . '</h3>';

                $failed_logins = get_option('aegis_shield_failed_logins', []);
                $blocked = aegis_shield_get_blocked_ips_fallback();

                if (!empty($failed_logins)) {
                    echo '<table class="widefat striped"><thead><tr><th>' . esc_html__('Time', 'aegis-shield') . '</th><th>' . esc_html__('Username', 'aegis-shield') . '</th><th>' . esc_html__('IP Address', 'aegis-shield') . '</th><th>' . esc_html__('Action', 'aegis-shield') . '</th></tr></thead><tbody>';
                    foreach (array_reverse($failed_logins, true) as $index => $log) {
                        echo '<tr>';
                        echo '<td>' . esc_html($log['time']) . '</td>';
                        echo '<td>' . esc_html($log['username']) . '</td>';
                        echo '<td>' . esc_html($log['ip']) . '</td>';
                        echo '<td>';
                        if (!in_array($log['ip'], $blocked, true)) {
                            $block_url = wp_nonce_url(admin_url('admin.php?page=aegis-shield&tab=ips&block_ip=' . urlencode($log['ip'])), 'aegis_shield_block_ip');
                            echo '<a href="' . esc_url($block_url) . '" class="button small">' . esc_html__('Block', 'aegis-shield') . '</a> ';
                        } else {
                            echo '<span style="color: green;">' . esc_html__('Blocked', 'aegis-shield') . '</span> ';
                        }
                        $delete_url = wp_nonce_url(admin_url('admin.php?page=aegis-shield&tab=ips&delete_log=' . $index), 'aegis_shield_delete_log');
                        echo '<a href="' . esc_url($delete_url) . '" class="button small button-link-delete" style="margin-left: 8px;">' . esc_html__('Delete', 'aegis-shield') . '</a>';
                        echo '</td></tr>';
                    }
                    echo '</tbody></table>';
                } else {
                    echo '<p>' . esc_html__('No failed login attempts recorded yet.', 'aegis-shield') . '</p>';
                }
                break;




            case 'cron':
                if (isset($_POST['save_cron_settings']) && check_admin_referer('aegis_shield_cron_settings')) {
                    $enabled = isset($_POST['enable_cron']) ? '1' : '0';

                    $frequency = 'weekly'; // Default
                    if (isset($_POST['cron_frequency'])) {
                        $raw_frequency = sanitize_text_field(wp_unslash($_POST['cron_frequency']));
                        $allowed_frequencies = ['daily', 'weekly', 'monthly', 'manual'];
                        if (in_array($raw_frequency, $allowed_frequencies, true)) {
                            $frequency = $raw_frequency;
                        }
                    }

                    update_option('aegis_shield_cron_enabled', $enabled);
                    update_option('aegis_shield_cron_frequency', $frequency);

                    wp_clear_scheduled_hook('aegis_shield_cron_scan');

                    if ($enabled === '1') {
                        wp_schedule_event(time(), $frequency, 'aegis_shield_cron_scan');
                    }

                    echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Cron settings saved.', 'aegis-shield') . '</p></div>';
                }

                $cron_enabled = get_option('aegis_shield_cron_enabled', '0');
                $cron_frequency = get_option('aegis_shield_cron_frequency', 'daily');
                $last_run = get_option('aegis_shield_last_cron_scan');
                ?>

                <div style="max-width: 600px; background: #fff; padding: 20px; border-radius: 6px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); margin-top: 20px;">
                    <form method="post">
                        <?php wp_nonce_field('aegis_shield_cron_settings'); ?>

                        <h2 style="margin-top: 0;"><?php esc_html_e('Automation Settings', 'aegis-shield'); ?></h2>

                        <p>
                            <label style="font-weight: 500;">
                                <input type="checkbox" name="enable_cron" value="1" <?php checked($cron_enabled, '1'); ?>>
                                <?php esc_html_e('Enable automatic scans & email reports', 'aegis-shield'); ?>
                            </label>
                        </p>

                        <p>
                            <label for="cron_frequency" style="font-weight: 500;">
                                <?php esc_html_e('Scan Frequency:', 'aegis-shield'); ?>
                            </label><br>
                            <select name="cron_frequency" id="cron_frequency" style="margin-top: 5px; min-width: 150px;">
                                <option value="daily" <?php selected($cron_frequency, 'daily'); ?>><?php esc_html_e('Daily', 'aegis-shield'); ?></option>
                                <option value="weekly" <?php selected($cron_frequency, 'weekly'); ?>><?php esc_html_e('Weekly', 'aegis-shield'); ?></option>
                                <option value="monthly" <?php selected($cron_frequency, 'monthly'); ?>><?php esc_html_e('Monthly', 'aegis-shield'); ?></option>
                                <option value="manual" <?php selected($cron_frequency, 'manual'); ?>><?php esc_html_e('Manual', 'aegis-shield'); ?></option>
                            </select>
                        </p>

                        <p>
                            <input type="submit" name="save_cron_settings" class="button button-primary" value="<?php esc_attr_e('Save Settings', 'aegis-shield'); ?>">
                        </p>

                        <?php if ($last_run): ?>
                            <p style="margin-top: 20px;"><strong><?php esc_html_e('Last automated scan:', 'aegis-shield'); ?></strong><br>
                                <?php echo esc_html($last_run); ?></p>
                        <?php endif; ?>
                    </form>
                </div>

                <?php
                break;



            case 'scan-history':
                do_action('aegis_shield_pro_scan_history_tab');
                break;

            case 'license':
                do_action('aegis_shield_pro_license_tab');
                break;

            case 'support':
                if (defined('AEGIS_SHIELD_PRO')) {
                    do_action('aegis_shield_pro_support_tab');
                }
                break;
            case 'get-pro':
                if (!defined('AEGIS_SHIELD_PRO')) {
                    ?>
                    <div class="wrap" style="max-width: 700px; padding-top: 1em;">
                        <h2 style="font-size: 26px; color: #222;">🚀 Unlock Pro Security Tools</h2>
                        <p style="font-size: 16px; line-height: 1.6;">
                            Get access to advanced scan history, suspicious code insights, premium support, and more — all backed by our expert security team.
                        </p>

                        <ul style="margin-left: 1.25em; list-style: disc; font-size: 15px;">
                            <li><strong>Detailed Scan History</strong> — Review, export, and manage past scans.</li>
                            <li><strong>Expert Code Insights</strong> — See what triggered scan alerts.</li>
                            <li><strong>Pro Support</strong> — Our security team will help review and advise.</li>
                            <li><strong>Plugin Update Delivery</strong> — Via secure license activation.</li>
                        </ul>

                        <a href="https://aegisshield.pro" target="_blank" style="display: inline-block; margin-top: 2em; background-color: #0073aa; color: #fff; font-weight: bold; padding: 12px 24px; border-radius: 4px; text-decoration: none;">
                            Upgrade to Aegis Shield Pro
                        </a>
                    </div>
                    <?php
                }
                break;

        }
        ?>
    </div>
    <?php
}
