<?php
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;
/**
 * EOF_Import_Export Class
 *
 * This class is for importing and exporting 
 * option settings.
 * 
 * @package EOF
 * @since 1.0.0
 */
if( !class_exists('EOF_Import_Export') ) :

class EOF_Import_Export {

	public function __construct( $parent ) {
		$this->parent = $parent;
		add_filter( "eof_sections", array($this, 'port_section') );
		add_action( "wp_ajax_eof_download_options-" . $this->parent->configs['opt_name'], array($this, "download_options") );
        add_action( "wp_ajax_nopriv_eof_download_options-" . $this->parent->configs['opt_name'], array($this, "download_options") );
		add_action( 'admin_init', array($this, 'run') );
		add_filter( 'upload_mimes', array($this, 'custom_upload_mimes') );
	}

	function run() {
		if( isset($this->parent->configs['options_page_id']) ) {
			add_action( "load-" . $this->parent->configs['options_page_id'], array($this, 'upload_import_file') );
		}
	}

	function port_section( $sections ) {
		$sections['eof_port'] = array(
			'title'	=> __('Import/Export', 'eof'),
			'id'	=> 'eof_port',
			'type'	=> 'blank',
			'callback' => array($this,'display'),
			'priority'	=> 41,
		);

		return $sections;
	} 

	public function display() {
		
		$output = '';

		?>

		<h3 class="title"><?php _e( 'Import Settings', 'eof' ); ?></h3>

		<p>
			<?php _e( 'Please select a <b>.json</b> file generated by this plugin.', 'eof' ); ?>
		</p>

		<form method="post" enctype="multipart/form-data">

			<?php wp_nonce_field( 'eof_import', 'eof_import_nonce' ); ?>

			<input type="file" name="eof_import_file" id="eof-import-file" />

			<p class="submit">
				<input type="submit" name="submit" id="submit" class="button button-primary" value="<?php _e( 'Import from File', 'eof' ); ?>">
			</p>
		</form>

		<h3 class="title"><?php _e( 'Export Settings', 'eof' ); ?></h3>

		<p>
			<?php _e( 'Click below to generate a <b>.json</b> file for your current option settings.', 'eof' ); ?>
		</p>

		<p class="submit">
			<?php $link = admin_url( 'admin-ajax.php?action=eof_download_options-' . $this->parent->configs['opt_name'] . '&nonce=' . wp_create_nonce( 'eof-export' ) ); ?>
			<a href="<?php echo $link; ?>" id="eof-export-button" class="button-primary"><?php _e( 'Export Settings', 'eof' ); ?></a>
		</p>
		<?php

	}

	/**
     * Adds the appropriate mime types to WordPress
     *
     * @since  1.0.0
     * @param array $existing_mimes
     * @return array
     */
    function custom_upload_mimes( $existing_mimes = array() ) {
        $existing_mimes['json']  = 'application/json';
        return $existing_mimes;
    }

	public function upload_import_file() {
		// Check nonce for security since form was posted
		if ( ! empty( $_POST ) && ! empty( $_FILES['eof_import_file'] ) && check_admin_referer( 'eof_import', 'eof_import_nonce' ) ) { 
			// Uploaded file
			$uploaded_file = $_FILES['eof_import_file'];
			// Check file type
			// This will also fire if no file uploaded
			$filetype = wp_check_filetype_and_ext( $uploaded_file['tmp_name'], $uploaded_file['name'], false );
			if ( 'json' != $filetype['ext'] && ! wp_match_mime_types( 'json', $wp_filetype['type'] ) ) {
				wp_die(
					__( 'You must upload a <b>.json</b> file generated by this plugin.', 'eof' ),
				'',
					array( 'back_link' => true )
				);
			}
			// Check and move file to uploads dir, get file data
			// Will show die with WP errors if necessary (file too large, quota exceeded, etc.)
			$overrides = array( 'test_form' => false );
			$file_data = wp_handle_upload( $uploaded_file, $overrides );
			if ( isset( $file_data['error'] ) ) {
				wp_die(
					$file_data['error'],
					'',
					array( 'back_link' => true )
				);
			}

			// Process import file
			$this->process_import_file( $file_data );
		}
	}
	function process_import_file( $file_data ) {
		global $eof_import_results;

		// File exists?
		if ( ! file_exists( $file_data['file'] ) ) {
			wp_die(
				__( 'Import file could not be found. Please try again.', 'eof' ),
				'',
				array( 'back_link' => true )
			);
		}
	
		// Get file contents and decode
		$data = wp_remote_get( $file_data['url'] );
		$data = json_decode( $data['body'], true );

		// Delete import file
		unlink( $file_data['file'] );

		// Import the options data
		$this->import_data( $data );
	}

	function import_data( $data ) {

		if ( ! empty ( $data ) && is_array( $data ) && isset ( $data['eof-backup'] ) && $data['eof-backup'] == '1' ) {

			$new_options = wp_parse_args( $data, $this->parent->null_options );

            update_option( $this->parent->configs['opt_name'], $new_options );
		}
	}

	public function download_options() {

		if ( ! isset( $_GET['nonce'] ) || !wp_verify_nonce( $_GET['nonce'], 'eof-export' ) ) {
            wp_die( __('Invalid nonce for options use', 'eof'), '' );
            exit;
        }

        $backup_options = $this->parent->options;
        $backup_options['eof-backup'] = '1';

        $content = json_encode( $backup_options );

        if ( isset( $_GET['action'] ) && $_GET['action'] == 'eof_download_options-'.$this->parent->configs['opt_name'] ) {
            header( 'Content-Description: File Transfer' );
            header( 'Content-type: application/txt' );
            header( 'Content-Disposition: attachment; filename="eof_options_' . $this->parent->configs['opt_name'] . '_backup_' . date( 'd-m-Y' ) . '.json"' );
            header( 'Content-Transfer-Encoding: binary' );
            header( 'Expires: 0' );
            header( 'Cache-Control: must-revalidate' );
            header( 'Pragma: public' );

            echo $content;

            exit;
        } else {
            header( "Expires: Mon, 26 Jul 1997 05:00:00 GMT" );
            header( "Last-Modified: " . gmdate( "D, d M Y H:i:s" ) . "GMT" );
            header( 'Expires: Sat, 26 Jul 1997 05:00:00 GMT' );
            header( 'Cache-Control: no-store, no-cache, must-revalidate' );
            header( 'Cache-Control: post-check=0, pre-check=0', false );
            header( 'Pragma: no-cache' );
            
            echo $content;
            exit;
        }
	}
}

endif;
