<?php
/**
 * Wallet transaction template
 *
 * This template displays the transaction details in the wallet section of the user's account.
 *
 * @package Advanced Wallet for WooCommerce
 * @since 1.0.0
 */

defined('ABSPATH') || exit;

// Variables expected from the template call:
// $wallet_balance, $transactions, $total_count, $total_pages, $current_page, $per_page
?>

<h2><?php echo esc_html__('Wallet Balance', 'advanced-wallet-for-woocommerce'); ?></h2>
<p class="wallet-balance"><?php echo esc_html__('Your current balance: ', 'advanced-wallet-for-woocommerce') . wp_kses_post(wc_price($wallet_balance)); ?></p>

<?php if (empty($transactions)) : ?>
    <p><?php echo esc_html__('No transactions found.', 'advanced-wallet-for-woocommerce'); ?></p>
<?php else : ?>
    <h3><?php echo esc_html__('Recent Transactions', 'advanced-wallet-for-woocommerce'); ?></h3>
    <table class="wallet-transactions">
        <thead>
            <tr>
                <th><?php echo esc_html__('Date', 'advanced-wallet-for-woocommerce'); ?></th>
                <th><?php echo esc_html__('Type', 'advanced-wallet-for-woocommerce'); ?></th>
                <th><?php echo esc_html__('Amount', 'advanced-wallet-for-woocommerce'); ?></th>
                <th><?php echo esc_html__('Description', 'advanced-wallet-for-woocommerce'); ?></th>
                <th><?php echo esc_html__('Expires', 'advanced-wallet-for-woocommerce'); ?></th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($transactions as $transaction) : 
                // Format the date
                $date = date_i18n(get_option('date_format'), strtotime($transaction['created_at']));
                
                // Determine CSS class based on transaction type
                $type_class = ($transaction['type'] === 'credit') ? 'credit' : 'debit';
                
                // Format the type label
                $type_label = ($transaction['type'] === 'credit') 
                    ? __('Credit', 'advanced-wallet-for-woocommerce') 
                    : __('Debit', 'advanced-wallet-for-woocommerce');
                
                // Format the amount
                if ($transaction['type'] === 'credit') {
                    $amount = wc_price($transaction['amount'] - $transaction['used_amount']);
                    if ($transaction['used_amount'] > 0) {
                        $amount .= ' <small>(' . wc_price($transaction['amount']) . ' ' . __('total', 'advanced-wallet-for-woocommerce') . ')</small>';
                    }
                } else {
                    $amount = wc_price($transaction['amount']);
                }
                
                // Format the description
                $description = $transaction['note'];
                
                // Get and format expiry date (only for credits)
                $expiry_html = '&mdash;';
                if ($transaction['type'] === 'credit' && !empty($transaction['expiry_date'])) {
                    $today = current_time('Y-m-d');
                    $expiry_date = $transaction['expiry_date'];
                    
                    // Format the date for display
                    $formatted_expiry = date_i18n(get_option('date_format'), strtotime($expiry_date));
                    
                    // Show different formatting based on expiry status
                    if ($expiry_date < $today) {
                        // Expired
                        $expiry_html = '<span class="expired">' . $formatted_expiry . ' (' . __('Expired', 'advanced-wallet-for-woocommerce') . ')</span>';
                    } elseif ($expiry_date === $today) {
                        // Expires today
                        $expiry_html = '<span class="expires-today">' . $formatted_expiry . ' (' . __('Today', 'advanced-wallet-for-woocommerce') . ')</span>';
                    } else {
                        // Future expiry
                        $days_left = (int)((strtotime($expiry_date) - strtotime($today)) / DAY_IN_SECONDS);
                        if ($days_left <= 7) {
                            // Highlight if expiring within 7 days
                            $expiry_html = '<span class="expiring-soon">' . $formatted_expiry . ' (' . sprintf(
                                // translators: %d: Number of days left until the wallet credit expires
                                _n('%d day left', '%d days left', $days_left, 'advanced-wallet-for-woocommerce'),
                                $days_left
                            ) . ')</span>';
                        } else {
                            $expiry_html = '<span class="valid">' . $formatted_expiry . '</span>';
                        }
                    }
                }
                
                // Special format for cashback credits
                $source_class = '';
                if ($transaction['type'] === 'credit' && $transaction['source'] === 'cashback') {
                    $source_class = ' cashback';
                    $type_label = __('Cashback', 'advanced-wallet-for-woocommerce');
                }
            ?>
                <tr class="<?php echo esc_attr($type_class . $source_class); ?>">
                    <td><?php echo esc_html($date); ?></td>
                    <td><?php echo esc_html($type_label); ?></td>
                    <td><?php echo wp_kses_post($amount); ?></td>
                    <td><?php echo wp_kses_post($description); ?></td>
                    <td><?php echo wp_kses_post($expiry_html); ?></td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>    
    
    <?php if ($total_pages > 1) : ?>
        <div class="tablenav bottom">
            <div class="tablenav-pages">
                <span class="displaying-num">
                    <?php 
                    printf(
                        // translators: %d: number of items
                        esc_html(_n('%d item', '%d items', $total_count, 'advanced-wallet-for-woocommerce')),
                        esc_html($total_count)
                    ); 
                    ?>
                </span>
                
                <nav class="pagination-container">
                    <ul class="pagination">
                        <?php
                        // Previous page
                        if ($current_page > 1) {
                            $prev_url = add_query_arg('page_no', ($current_page - 1), $base_url);
                            echo '<li class="page-item"><a class="page-link" href="' . esc_url($prev_url) . '">&laquo; ';
                            esc_html_e('Previous', 'advanced-wallet-for-woocommerce');
                            echo '</a></li>';
                        } else {
                            echo '<li class="page-item disabled"><span class="page-link">&laquo; ';
                            esc_html_e('Previous', 'advanced-wallet-for-woocommerce');
                            echo '</span></li>';
                        }
                        
                        // Page numbers
                        $start_page = max(1, $current_page - 2);
                        $end_page = min($total_pages, $current_page + 2);
                        
                        // First page
                        if ($start_page > 1) {
                            $first_url = add_query_arg('page_no', 1, $base_url);
                            echo '<li class="page-item"><a class="page-link" href="' . esc_url($first_url) . '">1</a></li>';
                            if ($start_page > 2) {
                                echo '<li class="page-item disabled"><span class="page-link">&hellip;</span></li>';
                            }
                        }
                        
                        // Page links
                        for ($i = $start_page; $i <= $end_page; $i++) {
                            $page_url = add_query_arg('page_no', $i, $base_url);
                            if ($i == $current_page) {
                                echo '<li class="page-item active"><span class="page-link">' . esc_html($i) . '</span></li>';
                            } else {
                                echo '<li class="page-item"><a class="page-link" href="' . esc_url($page_url) . '">' . esc_html($i) . '</a></li>';
                            }
                        }
                        
                        // Last page
                        if ($end_page < $total_pages) {
                            if ($end_page < $total_pages - 1) {
                                echo '<li class="page-item disabled"><span class="page-link">&hellip;</span></li>';
                            }
                            $last_url = add_query_arg('page_no', $total_pages, $base_url);
                            echo '<li class="page-item"><a class="page-link" href="' . esc_url($last_url) . '">' . esc_html($total_pages) . '</a></li>';
                        }
                        
                        // Next page
                        if ($current_page < $total_pages) {
                            $next_url = add_query_arg('page_no', ($current_page + 1), $base_url);
                            echo '<li class="page-item"><a class="page-link" href="' . esc_url($next_url) . '">';
                            esc_html_e('Next', 'advanced-wallet-for-woocommerce');
                            echo ' &raquo;</a></li>';
                        } else {
                            echo '<li class="page-item disabled"><span class="page-link">';
                            esc_html_e('Next', 'advanced-wallet-for-woocommerce');
                            echo ' &raquo;</span></li>';
                        }
                        ?>
                    </ul>
                </nav>
            </div>
        </div>
    <?php endif; ?>
<?php endif; ?>