(function($) {
    'use strict';

    /**
     * Handle wallet partial payment checkbox events at checkout
     */
    var PisolWalletCheckout = {
        init: function() {
            // Initialize event listeners
            $(document.body).on('change', '#use_wallet_partial', this.handleWalletPartialCheckbox);
            
            // Also trigger when the checkout is updated
            $(document.body).on('updated_checkout', this.initializeWalletPartialCheckbox);
            
            // Setup state persistence with cookies for better reliability
            this.setupStatePersistence();
            
            // Initialize on page load
            this.initializeWalletPartialCheckbox();
        },

        /**
         * Setup state persistence using both sessionStorage and cookies
         * for better cross-page reliability
         */
        setupStatePersistence: function() {
            // Check if we need to restore state from cookie (more reliable than sessionStorage)
            if (!sessionStorage.getItem('wallet_partial_checked') && this.getCookie('wallet_partial_checked')) {
                sessionStorage.setItem('wallet_partial_checked', this.getCookie('wallet_partial_checked'));
            }

            // Send state to server on page unload for better persistence
            $(window).on('beforeunload', function() {
                var checkboxState = sessionStorage.getItem('wallet_partial_checked');
                if (checkboxState) {
                    // Set cookie with 1 hour expiry for better persistence
                    PisolWalletCheckout.setCookie('wallet_partial_checked', checkboxState, 1);
                }
            });
        },

        /**
         * Handle wallet partial payment checkbox change
         */
        handleWalletPartialCheckbox: function() {
            // Store checkbox state in session storage and cookies
            var isChecked = $(this).is(':checked');
            var stateValue = isChecked ? 'true' : 'false';
            
            sessionStorage.setItem('wallet_partial_checked', stateValue);
            PisolWalletCheckout.setCookie('wallet_partial_checked', stateValue, 1);
            
            // Set hidden input value to pass state during AJAX
            $('#wallet_partial_state').val(isChecked ? '1' : '0');
            
            // Send an AJAX request to update the server session immediately
            $.ajax({
                type: 'POST',
                url: wc_checkout_params.ajax_url,
                data: {
                    action: 'pisol_update_wallet_state',
                    state: isChecked ? '1' : '0',
                    security: wc_checkout_params.update_order_review_nonce
                },
                success: function() {
                    // Update checkout when checkbox state changes to apply/remove wallet payment
                    $('body').trigger('update_checkout', { update_shipping_method: false });
                }
            });
        },

        /**
         * Initialize the wallet partial checkbox behavior after checkout is updated
         */
        initializeWalletPartialCheckbox: function() {
            // Get checkbox element
            var checkbox = $('#use_wallet_partial');
            if (checkbox.length > 0) {
                // Get state from sessionStorage or cookie
                var savedState = sessionStorage.getItem('wallet_partial_checked') || 
                                PisolWalletCheckout.getCookie('wallet_partial_checked');
                
                // Only set checked state if the saved state is 'true'
                // Otherwise ensure it's unchecked
                if (savedState === 'true') {
                    checkbox.prop('checked', true);
                    $('#wallet_partial_state').val('1');
                } else {
                    checkbox.prop('checked', false);
                    $('#wallet_partial_state').val('0');
                }
                
                // If checkbox exists but was not previously initialized
                // trigger an update to ensure the fee is applied or removed correctly
                if (!checkbox.data('initialized')) {
                    checkbox.data('initialized', true);
                    
                    // Only trigger update if there's a saved state
                    if (typeof savedState !== 'undefined' && savedState !== null) {
                        setTimeout(function() {
                            $('body').trigger('update_checkout', { update_shipping_method: false });
                        }, 100);
                    }
                }
            }
        },

        /**
         * Set a cookie with name, value and expiry in hours
         */
        setCookie: function(name, value, hours) {
            var expires = '';
            if (hours) {
                var date = new Date();
                date.setTime(date.getTime() + (hours * 60 * 60 * 1000));
                expires = '; expires=' + date.toUTCString();
            }
            document.cookie = name + '=' + value + expires + '; path=/; SameSite=Lax';
        },

        /**
         * Get cookie value by name
         */
        getCookie: function(name) {
            var nameEQ = name + '=';
            var ca = document.cookie.split(';');
            for (var i = 0; i < ca.length; i++) {
                var c = ca[i];
                while (c.charAt(0) === ' ') c = c.substring(1, c.length);
                if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length, c.length);
            }
            return null;
        }
    };

    // Initialize on document ready
    $(document).ready(function() {
        PisolWalletCheckout.init();
    });

})(jQuery);