<?php
/**
 * Wallet Transactions Class
 * Handles displaying wallet transactions in account area
 *
 * @package Advanced Wallet for WooCommerce
 * @since 1.0.0
 */

namespace PISOL\AWW\FRONT;

use PISOL\AWW\CLASSES\Wallet_Manager;

defined('ABSPATH') || exit;

class Wallet_Transactions {
    
    /**
     * The single instance of the class
     *
     * @var Wallet_Transactions
     */
    private static $instance = null;
    
    /**
     * Singleton pattern - ensures only one instance is created
     *
     * @return Wallet_Transactions
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Add wallet endpoint to My Account
        add_action('init', [$this, 'add_wallet_endpoint']);
        
        // Add wallet to My Account menu
        add_filter('woocommerce_account_menu_items', [$this, 'add_wallet_menu_item']);
        
        // Add content for wallet page
        add_action('woocommerce_account_wallet_endpoint', [$this, 'wallet_content']);
        
        // Add scripts and styles
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
    }
    
    /**
     * Add wallet endpoint to My Account
     */
    public function add_wallet_endpoint() {
        add_rewrite_endpoint('wallet', EP_ROOT | EP_PAGES);
    }
    
    /**
     * Add wallet to My Account menu
     *
     * @param array $items Menu items
     * @return array Modified menu items
     */
    public function add_wallet_menu_item($items) {
        // Insert after dashboard
        $new_items = array();
        
        foreach ($items as $key => $item) {
            $new_items[$key] = $item;
            
            if ($key === 'dashboard') {
                $new_items['wallet'] = __('My Wallet', 'advanced-wallet-for-woocommerce');
            }
        }
        
        return $new_items;
    }
    
    /**
     * Display wallet content in My Account page
     */
    public function wallet_content() {
        // Only for logged in users
        if (!is_user_logged_in()) {
            return;
        }

        // Display navigation menu
        $this->navigation_menu();

        $sub_page = isset( $_GET['action'] ) ? sanitize_text_field( $_GET['action'] ) : '';
        
        if( $sub_page === 'transactions' || empty( $sub_page ) ) {
            $this->transaction();
        } elseif( $sub_page === 'add' ) {

            $enable_recharge = get_option('pisol_aww_enable_recharge', '0');
            if(empty($enable_recharge)){
                return;
            }

            $this->add_fund();
        } 
        
    }

    function add_fund(){
        // Load the add fund template
        $min_amount = absint( get_option('pisol_aww_min_recharge_amount') );
        $max_amount = absint( get_option('pisol_aww_max_recharge_amount') );
        // Set a default max amount if not set or invalid
        if ( $max_amount <= 0 ) {
            $max_amount = ''; // Set your desired default max value
        }
        
        wc_get_template(
            'add-fund.php',
            array(
                'min_amount' => $min_amount,
                'max_amount' => $max_amount,
            ),
            '',
            plugin_dir_path(dirname(__FILE__)) . 'public/template/'
        );
    }

    function navigation_menu(){
        // Load the navigation menu template
        $navigation = array(
            'transactions' => __('Transactions', 'advanced-wallet-for-woocommerce'),
            'add' => __('Add Funds', 'advanced-wallet-for-woocommerce'),
        );

        $enable_recharge = get_option('pisol_aww_enable_recharge', '0');
        if(empty($enable_recharge)){
            unset($navigation['add']);
        }

        $current_action = isset($_GET['action']) ? sanitize_text_field($_GET['action']) : 'transactions';

        wc_get_template(
            'navigation.php',
            array(
                'navigation' => $navigation,
                'current_action' => $current_action,
            ),
            '',
            plugin_dir_path(dirname(__FILE__)) . 'public/template/'
        );
    }    
      
    function transaction(){
        $user_id = get_current_user_id();
        $wallet_manager = Wallet_Manager::get_instance();
        $wallet_balance = $wallet_manager->get_user_balance($user_id);
        
        // Get transaction data
        $result = $this->get_transactions($wallet_manager, $user_id);
        
        // Get base URL for pagination
        $base_url = wc_get_account_endpoint_url('wallet');
        
        // Load the transaction template
        wc_get_template(
            'transaction.php',
            array(
                'wallet_balance' => $wallet_balance,
                'transactions' => $result['transactions'],
                'total_count' => $result['total_count'],
                'total_pages' => $result['total_pages'],
                'current_page' => $result['current_page'],
                'per_page' => $result['per_page'],
                'base_url' => $base_url,
            ),
            '',
            plugin_dir_path(dirname(__FILE__)) . 'public/template/'
        );
    }
    
    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts() {
        // Only on account page
        if (!is_account_page()) {
            return;
        }
        
        wp_enqueue_style(
            'pisol-wallet-transactions',
            plugin_dir_url(dirname(__FILE__)) . 'public/css/wallet-transactions.css',
            [],
            PISOL_AWW_VERSION
        );
    }

    /**
     * Get wallet transactions based on filters
     *
     * @param Wallet_Manager $wallet_manager Wallet manager instance
     * @param int $user_id Optional user ID filter
     * @return array Transactions
     */
    private function get_transactions($wallet_manager, $user_id = 0) {
        // Get current page number
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- No sensitive action; page switching only.
        $current_page = isset($_GET['page_no']) ? absint($_GET['page_no']) : 1;
        if ($current_page < 1) $current_page = 1;
        
        // Items per page
        $per_page = 10;
        
        // Get paginated transactions directly using the combined function
        return Wallet_Manager::get_paginated_transactions_combined($user_id, $current_page, $per_page);
    }
    
    
}