<?php
/**
 * Wallet Payment Gateway Class
 * Handles wallet payment gateway functionality when wallet balance is sufficient for full payment
 *
 * @package Advanced Wallet for WooCommerce
 * @since 1.0.0
 */

namespace PISOL\AWW\FRONT;

use PISOL\AWW\CLASSES\Wallet_Manager;

defined('ABSPATH') || exit;

class Wallet_Gateway extends \WC_Payment_Gateway {
    
    public $instructions;
    /**
     * Constructor for the gateway.
     */
    public function __construct() {
        $this->id                 = 'pisol_wallet';
        $this->icon               = '';
        $this->has_fields         = false;
        $this->method_title       = __('Wallet Balance', 'advanced-wallet-for-woocommerce');
        $this->method_description = __('Pay directly from your wallet balance.', 'advanced-wallet-for-woocommerce');
        
        // Load form fields
        $this->init_form_fields();
        
        // Load settings
        $this->init_settings();
        
        // Define properties
        $this->title        = $this->get_option('title', __('Pay from Wallet', 'advanced-wallet-for-woocommerce'));
        $this->description  = $this->get_option('description', __('Pay using your wallet balance.', 'advanced-wallet-for-woocommerce'));
        $this->instructions = $this->get_option('instructions', $this->description);
        $this->enabled      = $this->get_option('enabled', 'yes');
        
        // Actions
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
        add_action('woocommerce_thankyou_' . $this->id, [$this, 'thankyou_page']);
        
        // Add customer order email instructions
        add_action('woocommerce_email_before_order_table', [$this, 'email_instructions'], 10, 3);
        
        // Process the payment
        add_action('woocommerce_payment_complete_order_status', [$this, 'change_payment_complete_order_status'], 10, 3);
    }
    
    /**
     * Initialize Gateway Settings Form Fields
     */
    public function init_form_fields() {
        $this->form_fields = [
            'enabled' => [
                'title'   => __('Enable/Disable', 'advanced-wallet-for-woocommerce'),
                'type'    => 'checkbox',
                'label'   => __('Enable Wallet Payments', 'advanced-wallet-for-woocommerce'),
                'default' => 'yes'
            ],
            'title' => [
                'title'       => __('Title', 'advanced-wallet-for-woocommerce'),
                'type'        => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'advanced-wallet-for-woocommerce'),
                'default'     => __('Pay from Wallet', 'advanced-wallet-for-woocommerce'),
                'desc_tip'    => true,
            ],
            'description' => [
                'title'       => __('Description', 'advanced-wallet-for-woocommerce'),
                'type'        => 'textarea',
                'description' => __('Payment method description that the customer will see on your checkout.', 'advanced-wallet-for-woocommerce'),
                'default'     => __('Pay using your wallet balance.', 'advanced-wallet-for-woocommerce'),
                'desc_tip'    => true,
            ],
            'instructions' => [
                'title'       => __('Instructions', 'advanced-wallet-for-woocommerce'),
                'type'        => 'textarea',
                'description' => __('Instructions that will be added to the thank you page and emails.', 'advanced-wallet-for-woocommerce'),
                'default'     => __('Your wallet balance has been debited for this order.', 'advanced-wallet-for-woocommerce'),
                'desc_tip'    => true,
            ],
        ];
    }
    
    /**
     * Check if the gateway is available for use
     *
     * @return bool
     */
    public function is_available() {
        $is_available = parent::is_available();

        if (!$is_available || !is_user_logged_in()) {
            return false;
        }

        if(Add_Fund::is_wallet_rechargeable_cart()){
            return false; // Don't show this gateway if wallet recharge product is in the cart
        }

        if (WC()->cart && WC()->cart->needs_payment()) {
            $user_id = get_current_user_id();
            $wallet_manager = Wallet_Manager::get_instance();
            $wallet_balance = $wallet_manager->get_user_balance($user_id);

            $formatted_balance = html_entity_decode(
                wp_strip_all_tags( wc_price( $wallet_balance ) )
            );

            $sub_text = sprintf(
                // translators: %s: the user's current wallet balance formatted as price.
                __('(💼 Balance: %s)', 'advanced-wallet-for-woocommerce'),
                $formatted_balance
            );

            if(strpos($this->title, $sub_text) === false) {
                // Append the wallet balance to the title if not already present
                $this->title .= ' ' . $sub_text;
            }
            
            // Get the raw cart total without any wallet fees
            $raw_cart_total = 0;
            
            // Check if wallet checkout class exists and get raw total from there
            $checkout_handler = Wallet_Checkout::get_instance();
            if (method_exists($checkout_handler, 'get_raw_cart_total')) {
                $raw_cart_total = $checkout_handler->get_raw_cart_total();
            } else {
                // Fallback to cart total if method doesn't exist
                $raw_cart_total = WC()->cart->get_total('numeric');
            }
            
            // Check if partial payment option is being used
            $partial_payment_active = false;
            if (WC()->session) {
                $partial_payment_active = (bool) WC()->session->get('wallet_partial_checked', false);
            }
            
            // Only show this gateway if the wallet balance is sufficient for the entire order
            // AND partial payment option is not being used
            return ($wallet_balance >= $raw_cart_total) && !$partial_payment_active;
        }

        return $is_available;
    }
    
    /**
     * Output for the order received page.
     *
     * @param int $order_id Order ID
     */
    public function thankyou_page($order_id) {
        if ($this->instructions) {
            echo wp_kses_post(wpautop(wptexturize($this->instructions)));
        }
    }
    
    /**
     * Add content to the WC emails.
     *
     * @param WC_Order $order Order object
     * @param bool $sent_to_admin Sent to admin
     * @param bool $plain_text Email format: plain text or HTML
     */
    public function email_instructions($order, $sent_to_admin, $plain_text = false) {
        if ($this->instructions && !$sent_to_admin && $this->id === $order->get_payment_method()) {
            echo wp_kses_post(wpautop(wptexturize($this->instructions)) . PHP_EOL);
        }
    }
    
    /**
     * Process the payment and return the result
     *
     * @param int $order_id Order ID
     * @return array
     */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
        $user_id = $order->get_user_id();
        
        if (!$user_id) {
            wc_add_notice(__('User account is required for wallet payment.', 'advanced-wallet-for-woocommerce'), 'error');
            return;
        }
        
        $wallet_manager = Wallet_Manager::get_instance();
        $order_total = $order->get_total();
        $wallet_balance = $wallet_manager->get_user_balance($user_id);
        
        // Check if user has enough balance
        if ($wallet_balance < $order_total) {
            wc_add_notice(
                sprintf(
                    // translators: %s: the user's current wallet balance formatted as price.
                    __('Insufficient wallet balance. Your current balance is %s.', 'advanced-wallet-for-woocommerce'),
                    wc_price($wallet_balance)
                ),
                'error'
            );
            return;
        }
        
        // Process the debit from wallet
        $debit_result = $wallet_manager->add_debit([
            'user_id' => $user_id,
            'amount' => $order_total,
            'reference_id' => $order_id,
            'reference_type' => 'order',
            'note' => sprintf(
                // translators: %s: the order number.
                __('Payment for order #%s', 'advanced-wallet-for-woocommerce'),
                $order->get_order_number()
            )
        ]);
        
        if (!$debit_result) {
            wc_add_notice(__('There was an error processing your wallet payment. Please try again.', 'advanced-wallet-for-woocommerce'), 'error');
            return;
        }
        
        // Add order note
        $order->add_order_note(
            sprintf(
                // translators: %s: the order total formatted as price.
                __('Payment of %s made from wallet balance.', 'advanced-wallet-for-woocommerce'),
                wc_price($order_total)
            )
        );
        
        // Mark as paid
        $order->payment_complete();
        
        // Save wallet payment info in order meta
        update_post_meta($order_id, '_wallet_payment', 'yes');
        update_post_meta($order_id, '_wallet_payment_amount', $order_total);
        
        // Reduce stock levels
        wc_reduce_stock_levels($order_id);
        
        // Remove cart
        WC()->cart->empty_cart();
        
        // Return thank you page redirect
        return [
            'result'   => 'success',
            'redirect' => $this->get_return_url($order)
        ];
    }
    
    /**
     * Change the order status for paid orders
     *
     * @param string $status New order status
     * @param int $order_id Order ID
     * @param WC_Order $order Order object
     * @return string Modified order status
     */
    public function change_payment_complete_order_status($status, $order_id, $order) {
        if ($order->get_payment_method() === $this->id) {
            $status = 'processing';
        }
        return $status;
    }
}