<?php
/**
 * Wallet Checkout Class
 * Handles the application of wallet balance at checkout
 *
 * @package Advanced Wallet for WooCommerce
 * @since 1.0.0
 */

namespace PISOL\AWW\FRONT;

use PISOL\AWW\CLASSES\Wallet_Manager;

defined('ABSPATH') || exit;

class Wallet_Checkout {
    
    /**
     * The single instance of the class
     *
     * @var Wallet_Checkout
     */
    private static $instance = null;
    
    /**
     * Singleton pattern - ensures only one instance is created
     *
     * @return Wallet_Checkout
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Add checkbox to checkout when partial payment is possible
        add_action('woocommerce_review_order_before_payment', [$this, 'add_wallet_partial_payment_option']);
        
        // Process partial payment option
        add_action('woocommerce_checkout_process', [$this, 'validate_wallet_partial_payment']);
        
        // Apply partial payment from wallet as fee
        add_action('woocommerce_cart_calculate_fees', [$this, 'apply_wallet_partial_payment'], 20);
        
        // Save wallet partial payment option to order
        add_action('woocommerce_checkout_update_order_meta', [$this, 'save_wallet_partial_payment_option']);
        
        // Process the wallet deduction after order is placed
        add_action('woocommerce_checkout_order_processed', [$this, 'process_wallet_payment'], 10, 3);
        
        // Filter to check if user has sufficient balance (used by gateway)
        add_filter('pisol_aww_user_has_sufficient_balance', [$this, 'check_sufficient_balance'], 10, 2);
        
        // Enqueue scripts
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        
        // Add hidden field to store checkbox state
        add_action('woocommerce_after_checkout_form', [$this, 'add_hidden_fields']);
        
        // Handle AJAX update for wallet partial payment
        add_action('woocommerce_checkout_update_order_review', [$this, 'handle_checkout_update']);
        
        // Add AJAX handler for wallet state updates
        add_action('wp_ajax_pisol_update_wallet_state', [$this, 'ajax_update_wallet_state']);
        add_action('wp_ajax_nopriv_pisol_update_wallet_state', [$this, 'ajax_update_wallet_state']);
        
        // Initialize state from cookie if present
        add_action('init', [$this, 'initialize_wallet_state_from_cookie'], 20);
    }
    
    /**
     * Initialize wallet state from cookie if present
     * This runs early to ensure consistency across page loads
     */
    public function initialize_wallet_state_from_cookie() {
        // Only for checkout page and only for logged in users
        if (!is_checkout() || !is_user_logged_in() || !function_exists('WC')) {
            return;
        }
        
        // Check if WC session is initialized
        if (!WC()->session) {
            return;
        }
        
        // Check for cookie state - prioritize cookie state on page load
        if (isset($_COOKIE['wallet_partial_checked'])) {
            $cookie_state = sanitize_text_field(wp_unslash($_COOKIE['wallet_partial_checked']));
            $use_wallet = ($cookie_state === 'true');
            
            // Set session state to match cookie
            WC()->session->set('wallet_partial_checked', $use_wallet);
            
            // Force recalculation of fees if needed
            if (WC()->cart) {
                WC()->cart->calculate_totals();
            }
        }
    }
    
    /**
     * AJAX handler for wallet state updates
     * Directly updates session state without full checkout refresh
     */
    public function ajax_update_wallet_state() {
        // Verify security nonce
        if (!isset($_POST['security']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['security'])), 'woocommerce-process_checkout')) {
            wp_send_json_error('Invalid security token');
            exit;
        }
        
        // Get state value
        $state = (isset($_POST['state']) && $_POST['state'] === '1');
        
        // Update session
        if (WC()->session) {
            WC()->session->set('wallet_partial_checked', $state);
            if (!$state) {
                WC()->session->set('wallet_partial_payment_amount', 0);
            }
            
            // Set cookie for better persistence
            $expire = time() + HOUR_IN_SECONDS;
            setcookie('wallet_partial_checked', $state ? 'true' : 'false', $expire, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);
        }
        
        wp_send_json_success();
        exit;
    }
    
    /**
     * Add hidden fields to store checkbox state
     */
    public function add_hidden_fields() {
        // Only on checkout page
        if (!is_checkout()) {
            return;
        }
        
        // Check if we need to set initial state from session or cookie
        $checked = false;
        
        // Check session first
        if (WC()->session) {
            $checked = (bool) WC()->session->get('wallet_partial_checked', false);
        }
        
        // Check cookie as fallback
        if (!$checked && isset($_COOKIE['wallet_partial_checked'])) {
            $checked = (sanitize_text_field(wp_unslash($_COOKIE['wallet_partial_checked'])) === 'true');
        }
        
        // Add hidden field with proper initial value
        echo '<input type="hidden" id="wallet_partial_state" name="wallet_partial_state" value="' . esc_attr($checked ? '1' : '0') . '">';
    }
    
    /**
     * Handle checkout update to process wallet partial payment
     * 
     * @param string $post_data Posted data from checkout
     */
    public function handle_checkout_update($post_data) {
        parse_str($post_data, $posted_data);
        
        // Check for the wallet_partial_state hidden field
        if (isset($posted_data['wallet_partial_state'])) {
            $use_wallet = ($posted_data['wallet_partial_state'] === '1');
            
            // Store in session - explicitly set to true or false (not null)
            if (WC()->session) {
                WC()->session->set('wallet_partial_checked', $use_wallet);
                
                // If unchecked, also clear the payment amount to ensure fee is removed
                if (!$use_wallet) {
                    WC()->session->set('wallet_partial_payment_amount', 0);
                }
                
                // Set cookie for better persistence
                $expire = time() + HOUR_IN_SECONDS;
                setcookie('wallet_partial_checked', $use_wallet ? 'true' : 'false', $expire, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);
            }
        }
        // Also check for direct checkbox state
        elseif (isset($posted_data['use_wallet_partial'])) {
            $use_wallet = ($posted_data['use_wallet_partial'] === '1');
            
            if (WC()->session) {
                WC()->session->set('wallet_partial_checked', $use_wallet);
                
                if (!$use_wallet) {
                    WC()->session->set('wallet_partial_payment_amount', 0);
                }
                
                // Set cookie for better persistence
                $expire = time() + HOUR_IN_SECONDS;
                setcookie('wallet_partial_checked', $use_wallet ? 'true' : 'false', $expire, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);
            }
        }
    }
    
    /**
     * Enqueue scripts and styles needed for wallet checkout functionality
     */
    public function enqueue_scripts() {
        // Only enqueue script on checkout page
        if (!is_checkout()) {
            return;
        }
        
        wp_enqueue_script(
            'pisol-wallet-checkout',
            plugin_dir_url(dirname(__FILE__)) . 'public/js/script.js',
            ['jquery', 'wc-checkout'],
            PISOL_AWW_VERSION,
            true
        );
        
        // Pass some data to the script
        wp_localize_script(
            'pisol-wallet-checkout',
            'pisol_wallet_params',
            [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('pisol-wallet-nonce')
            ]
        );
    }
    
    /**
     * Get raw cart total without wallet fee
     * 
     * @return float Raw cart total
     */
    public function get_raw_cart_total() {
        $cart = WC()->cart;
        
        if (!$cart) {
            return 0;
        }
        
        // Get cart total before fees
        $cart_subtotal = $cart->get_subtotal() + $cart->get_subtotal_tax();
        $cart_shipping = $cart->get_shipping_total() + $cart->get_shipping_tax();
        $cart_total = $cart_subtotal + $cart_shipping;
        
        // Add any other fees (except our own wallet fee)
        foreach ($cart->get_fees() as $fee) {
            if (strpos($fee->name, __('Wallet Balance Applied', 'advanced-wallet-for-woocommerce')) === false) {
                $cart_total += $fee->amount;
            }
        }
        
        return $cart_total;
    }
    
    /**
     * Store raw cart total in session for consistent checkout handling
     */
    public function store_raw_cart_total() {
        if (!is_checkout() || !function_exists('WC') || !WC()->session) {
            return;
        }
        
        // Store raw cart total in session for access by other methods
        $raw_cart_total = $this->get_raw_cart_total();
        WC()->session->set('wallet_raw_cart_total', $raw_cart_total);
    }
    
    /**
     * Add wallet partial payment checkbox at checkout
     */
    public function add_wallet_partial_payment_option() {
        // Only for logged in users
        if (!is_user_logged_in()) {
            return;
        }
        
        $user_id = get_current_user_id();
        $wallet_manager = Wallet_Manager::get_instance();
        $wallet_balance = $wallet_manager->get_user_balance($user_id);
        
        // If wallet balance is zero or negative, return
        if ($wallet_balance <= 0) {
            return;
        }

        if(Add_Fund::is_wallet_rechargeable_cart()){
            return false; // Don't show this gateway if wallet recharge product is in the cart
        }
        
        // Store the raw cart total in session (without wallet fee)
        $this->store_raw_cart_total();
        
        // Get raw cart total (before wallet fee)
        $raw_cart_total = WC()->session->get('wallet_raw_cart_total', $this->get_raw_cart_total());
        
        // If wallet balance is less than raw cart total, show partial payment option
        if ($wallet_balance < $raw_cart_total) {
            $formatted_balance = wc_price($wallet_balance);
            
            // Check if the checkbox was previously checked
            $checked = false;
            
            // First check session
            if (WC()->session) {
                $checked = (bool) WC()->session->get('wallet_partial_checked', false);
            }
            
            // Then check cookie as fallback
            if (!$checked && isset($_COOKIE['wallet_partial_checked'])) {
                $checked = (sanitize_text_field(wp_unslash($_COOKIE['wallet_partial_checked'])) === 'true');
                
                // Update session from cookie if needed
                if ($checked && WC()->session) {
                    WC()->session->set('wallet_partial_checked', true);
                }
            }
            
            woocommerce_form_field('use_wallet_partial', [
                'type' => 'checkbox',
                'class' => ['use-wallet-partial form-row-wide'],
                
                'label' => sprintf(
                    // translators: %s: Wallet balance amount
                    __('Use wallet balance (%s) for partial payment', 'advanced-wallet-for-woocommerce'),
                    $formatted_balance
                ),
                'default' => $checked ? 1 : 0
            ]);
        }
    }
    
    /**
     * Validate wallet partial payment option
     */
    public function validate_wallet_partial_payment() {
        // No validation needed, just ensuring the function exists
    }
    
    /**
     * Apply wallet balance as a fee (negative amount) when partial payment option is selected
     *
     * @param object $cart WC_Cart object
     */
    public function apply_wallet_partial_payment($cart) {
        if (is_admin() && !defined('DOING_AJAX')) {
            return;
        }

        if (defined('DOING_AJAX') && DOING_AJAX && isset($_POST['security'])) {
            if (!check_ajax_referer('update-order-review', 'security', false)) {
                return; // Nonce check failed, do not process
            }
        }


        // First remove any existing wallet fee to prevent duplicates
        $this->remove_existing_wallet_fees($cart);
        
        // Don't apply a fee if the wallet option isn't checked
        $use_wallet = false;
        
        // Check hidden field first (most reliable during AJAX)
        if (isset($_POST['wallet_partial_state'])) {
            $use_wallet = sanitize_text_field(wp_unslash($_POST['wallet_partial_state'])) == '1';
        }
        // Check in POST data (when updating checkout)
        elseif (isset($_POST['post_data'])) {
            $post_data = sanitize_text_field(wp_unslash($_POST['post_data']));
            parse_str($post_data, $post_data);
            
            // First check hidden field
            if (isset($post_data['wallet_partial_state'])) {
                $use_wallet = $post_data['wallet_partial_state'] === '1';
            }
            // Then check checkbox
            elseif (isset($post_data['use_wallet_partial'])) {
                $use_wallet = $post_data['use_wallet_partial'] === '1';
            }
        }
        // Check direct POST (when submitting checkout)
        elseif (isset($_POST['use_wallet_partial'])) {
            $use_wallet = sanitize_text_field(wp_unslash($_POST['use_wallet_partial'])) == '1';
        }
        
        // Check session (for persistence) - only if previous checks didn't yield a result
        if (WC()->session && !isset($_POST['wallet_partial_state']) && !isset($_POST['post_data']) && !isset($_POST['use_wallet_partial'])) {
            $use_wallet = (bool) WC()->session->get('wallet_partial_checked', false);
        }
        
        // Store the current state in the session for future requests
        if (WC()->session) {
            WC()->session->set('wallet_partial_checked', $use_wallet);
        }
        
        // If checkbox isn't checked or user not logged in, no need to continue
        if (!$use_wallet || !is_user_logged_in()) {
            return;
        }
        
        $user_id = get_current_user_id();
        $wallet_manager = Wallet_Manager::get_instance();
        $wallet_balance = $wallet_manager->get_user_balance($user_id);
        
        // If wallet balance is zero or negative, return
        if ($wallet_balance <= 0) {
            return;
        }
        
        // Get raw cart total (before wallet fee) - this is important for consistent checkout experience
        $raw_cart_total = WC()->session->get('wallet_raw_cart_total', $this->get_raw_cart_total());
        
        // If wallet balance is greater than or equal to raw cart total, don't apply as fee
        // In this case, the full payment gateway should be used
        if ($wallet_balance >= $raw_cart_total) {
            return;
        }
        
        // Amount to apply - use entire wallet balance
        $amount_to_use = $wallet_balance;
        
        // Apply wallet balance as a negative fee
        $cart->add_fee(
            sprintf(__('Wallet Balance Applied', 'advanced-wallet-for-woocommerce')),
            -$amount_to_use,
            false // Non-taxable
        );
        
        // Store wallet amount and raw cart total in session for later use
        WC()->session->set('wallet_partial_payment_amount', $amount_to_use);
        WC()->session->set('wallet_raw_cart_total', $raw_cart_total);
    }
    
    /**
     * Remove any existing wallet fees from the cart
     * 
     * @param WC_Cart $cart Cart object
     */
    private function remove_existing_wallet_fees($cart) {
        if (!method_exists($cart, 'get_fees') || empty($cart->get_fees())) {
            return;
        }
        
        $fee_label = __('Wallet Balance Applied', 'advanced-wallet-for-woocommerce');
        
        // Get all current fees and store IDs of wallet fees to remove
        $fees_to_remove = [];
        foreach ($cart->get_fees() as $fee_key => $fee) {
            if (strpos($fee->name, $fee_label) !== false) {
                $fees_to_remove[] = $fee_key;
            }
        }
        
        // Remove each fee using the proper API method
        foreach ($fees_to_remove as $fee_id) {
            $cart->remove_fee($fee_id);
        }
    }
    
    /**
     * Save the wallet partial payment option to the order meta
     *
     * @param int $order_id Order ID
     */
    public function save_wallet_partial_payment_option($order_id) {
        $use_wallet = false;

        $nonce_value    = sanitize_text_field(wp_unslash( $_REQUEST['woocommerce-process-checkout-nonce'] ?? '' ) ); 

        if ( empty( $nonce_value ) || ! wp_verify_nonce( $nonce_value, 'woocommerce-process_checkout' ) ) {
				return;
		}
        
        // Check hidden field first (most reliable)
        if (isset($_POST['wallet_partial_state'])) {
            $use_wallet = sanitize_text_field(wp_unslash($_POST['wallet_partial_state'])) == '1';
        }
        // Check direct POST (when submitting checkout)
        elseif (isset($_POST['use_wallet_partial'])) {
            $use_wallet = sanitize_text_field(wp_unslash($_POST['use_wallet_partial'])) == '1';
        }
        // Check session (for persistence)
        elseif (WC()->session) {
            $use_wallet = WC()->session->get('wallet_partial_checked', false);
        }
        
        if ($use_wallet) {
            update_post_meta($order_id, '_wallet_partial_payment', 'yes');
            
            // Save the wallet amount used
            $wallet_amount = WC()->session->get('wallet_partial_payment_amount', 0);
            if ($wallet_amount > 0) {
                update_post_meta($order_id, '_wallet_partial_payment_amount', $wallet_amount);
            }
        }
        
        // Always clear session data after order is placed
        if (WC()->session) {
            WC()->session->__unset('wallet_partial_checked');
            WC()->session->__unset('wallet_partial_payment_amount');
        }
    }
    
    /**
     * Process wallet payment after order is placed
     *
     * @param int $order_id Order ID
     * @param array $posted_data Posted data
     * @param WC_Order $order Order object
     */
    public function process_wallet_payment($order_id, $posted_data, $order) {
        // Check if partial payment was used
        $wallet_partial_payment = get_post_meta($order_id, '_wallet_partial_payment', true);
        
        if ($wallet_partial_payment === 'yes') {
            $wallet_amount = get_post_meta($order_id, '_wallet_partial_payment_amount', true);
            
            if ($wallet_amount > 0) {
                $user_id = $order->get_user_id();
                $wallet_manager = Wallet_Manager::get_instance();
                
                // Deduct from wallet
                $debit_result = $wallet_manager->add_debit([
                    'user_id' => $user_id,
                    'amount' => $wallet_amount,
                    'reference_id' => $order_id,
                    'reference_type' => 'order',
                    
                    'note' => sprintf(
                        // translators: %s: Order number
                        __('Partial payment for order #%s', 'advanced-wallet-for-woocommerce'),
                        $order->get_order_number()
                    )
                ]);
                
                if ($debit_result) {
                    // Add order note
                    
                    $order->add_order_note(
                        sprintf(
                            // translators: %s: Formatted amount
                            __('%s paid from wallet balance.', 'advanced-wallet-for-woocommerce'),
                            wc_price($wallet_amount)
                        )
                    );
                    
                    // Clear session data
                    WC()->session->__unset('wallet_partial_payment_amount');
                    WC()->session->__unset('wallet_partial_checked');
                }
            }
        }
    }
    
    /**
     * Check if user has sufficient wallet balance for the order
     *
     * @param bool $has_balance Default value
     * @param float $amount Amount to check
     * @return bool True if user has sufficient balance
     */
    public function check_sufficient_balance($has_balance, $amount) {
        if (!is_user_logged_in()) {
            return false;
        }
        
        $user_id = get_current_user_id();
        $wallet_manager = Wallet_Manager::get_instance();
        
        return $wallet_manager->has_sufficient_balance($user_id, $amount);
    }
}