<?php 

namespace PISOL\AWW\FRONT;

use PISOL\AWW\CLASSES\Wallet_Manager;
use PISOL\AWW\CLASSES\Install;
use PISOL\AWW\CLASSES\Email;


defined('ABSPATH') || exit;

class Add_Fund {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // Add action to handle form submission
        add_action('wp_loaded', [$this, 'handle_add_fund']);

        add_filter( 'woocommerce_is_purchasable', array( $this, 'make_purchasable' ), 10, 2 );

        add_action( 'woocommerce_before_calculate_totals', array( $this, 'set_recharge_product_price' ) );
        
        // Add action to remove wallet product when regular product is added
        add_action( 'woocommerce_add_to_cart', array( $this, 'maybe_remove_wallet_product_from_cart' ), 10, 6 );

        foreach ( apply_filters( 'pisol_aww_purchase_order_status', array( 'processing', 'completed' ) ) as $status ) {
			add_action( 'woocommerce_order_status_' . $status, array( $this, 'credit_purchase' ) );
		}

        add_action('woocommerce_cart_calculate_fees' , array($this,'addfees'), PHP_INT_MAX);
    }

    function handle_add_fund(){
        if (
            !isset($_POST['pisol_add_fund_nonce']) ||
            !wp_verify_nonce(sanitize_text_field($_POST['pisol_add_fund_nonce']), 'pisol_add_fund_action')
        ) {
            return;
        }        
        
        if(Install::wallet_recharge_product()){
            $amount = isset($_POST['pisol_fund_amount']) ? sanitize_text_field($_POST['pisol_fund_amount']) : 0;

            $min_amount = get_option('pisol_aww_min_recharge_amount');
            $max_amount = get_option('pisol_aww_max_recharge_amount');
            if ($amount < $min_amount || (!empty($max_amount) && $amount > $max_amount)) {
                wc_add_notice(sprintf(__('Please enter an amount between %s and %s.', 'advanced-wallet-for-woocommerce'), wc_price($min_amount), wc_price($max_amount)), 'error');
                return;
            }

            if($amount <= 0 ){
                wc_add_notice(__('Please enter a amount greater than 0.', 'advanced-wallet-for-woocommerce'), 'error');
                return;
            }

            $product_id = get_option('_add_wallet_recharge_product');
            $product = wc_get_product($product_id);
            WC()->cart->empty_cart();
            WC()->cart->add_to_cart($product_id, 1, '', array(), array('topup_amount' => $amount));
            // Redirect to checkout page after adding product to cart
            wp_safe_redirect(wc_get_checkout_url());
            exit;
        } else {
            wc_add_notice(__('Wallet product not found. Please contact support.', 'advanced-wallet-for-woocommerce'), 'error');
        }
    }

    public function make_purchasable( $is_purchasable, $product ) {
        $wallet_product = Install::wallet_recharge_product();
        if ( $wallet_product ) {
            if ( $wallet_product === $product->get_id() ) {
                $is_purchasable = true;
            }
        }
        return $is_purchasable;
    }

    static function is_wallet_rechargeable_cart(){
        $wallet_product = Install::wallet_recharge_product();
        if ( WC()->cart && ! empty( WC()->cart->get_cart() ) ) {
            foreach ( WC()->cart->get_cart() as $cart_item ) {
                if (  $cart_item['product_id'] === $wallet_product ) {
                    return true;
                }
            }
        }
        return false;
    }

    public function set_recharge_product_price( $cart ) {
        $product_id = Install::wallet_recharge_product();
        if ( ! $product_id && empty( $cart->cart_contents ) ) {
            return;
        }
        foreach ( $cart->cart_contents as $key => $value ) {
            if ( isset( $value['topup_amount'] ) && $value['topup_amount'] && $product_id == $value['product_id'] ) {
                $value['data']->set_price( $value['topup_amount'] );
            }
        }
    }
    
    /**
     * Remove wallet recharge product from cart when a regular product is added
     *
     * @param string $cart_item_key Cart item key of the newly added product
     * @param int    $product_id    ID of the product that was added to the cart
     * @param int    $quantity      Quantity of the item added
     * @param int    $variation_id  Variation ID of the product (if applicable)
     * @param array  $variation     Variation data (if applicable)
     * @param array  $cart_item_data Cart item data passed when adding the product
     */
    public function maybe_remove_wallet_product_from_cart( $cart_item_key, $product_id, $quantity, $variation_id, $variation, $cart_item_data ) {
        // Get wallet recharge product ID
        $wallet_product_id = Install::wallet_recharge_product();
        
        // If no wallet product exists or the added product is the wallet product, do nothing
        if ( ! $wallet_product_id || $product_id == $wallet_product_id ) {
            return;
        }
        
        // Check if wallet recharge product is in the cart
        if ( WC()->cart && ! empty( WC()->cart->get_cart() ) ) {
            foreach ( WC()->cart->get_cart() as $cart_key => $cart_item ) {
                // If this cart item is the wallet recharge product
                if ( $cart_item['product_id'] == $wallet_product_id ) {
                    // Remove the wallet recharge product from cart
                    WC()->cart->remove_cart_item( $cart_key );
                    
                    // Add notice to inform the user
                    wc_add_notice( 
                        __( 'Wallet recharge product has been removed from your cart as it cannot be purchased with other products.', 'advanced-wallet-for-woocommerce' ), 
                        'notice' 
                    );
                    
                    // No need to continue checking
                    break;
                }
            }
        }
    }

    function credit_purchase( $order_id ) {
        $order          = wc_get_order( $order_id );
        $wallet_product = Install::wallet_recharge_product();

        if( !self::is_wallet_rechargeable_order( $order ) ) {
            return;
        }
       
        if ( $order->get_meta( '_aww_wallet_topup_done' ) || ! $wallet_product ) {
            if (function_exists('pisol_aww_error_log')) {
                pisol_aww_error_log(sprintf('Topup amount already credited %s', $order->get_order_number()), 'wallet');
            }
            return;
        }

        $recharge_amount = $order->get_subtotal( 'edit' );
       
        $transaction_id = $this->add_credit($recharge_amount, $order);

        if ( $transaction_id ) {
            Email::send_email(
                $order->get_billing_email(),
                sprintf( __( 'Wallet Topup for Order #%s', 'advanced-wallet-for-woocommerce' ), $order->get_order_number() ),
                sprintf( __( 'Your wallet has been topped up with %s.', 'advanced-wallet-for-woocommerce' ), wc_price($recharge_amount) ),
                __( 'Wallet Topup', 'advanced-wallet-for-woocommerce' )
            );
            $order->update_meta_data( '_aww_wallet_topup_done', true);
            $order->update_meta_data( '_aww_wallet_topup_transaction_id', $transaction_id );
        }

        $order->save();
    }

    function add_credit($amount, $order){
        $user_id = get_current_user_id();
        $order_id = $order->get_id();
        $currency = $order->get_currency();
        $topup_expiry_date = self::purchased_credit_expiry_date();
        

        $credit_params = array(
            "user_id" => $user_id,
            "amount" => $amount,
            "currency" => $currency,
            "source" => "purchase",
            /* translators: %s: order number */
            "note" => sprintf(__("Wallet topup by order #%s", "advanced-wallet-for-woocommerce"), $order->get_order_number())
        );

        if ($topup_expiry_date) {
            $credit_params['expiry_date'] = $topup_expiry_date;
        }

        $wallet_manager = Wallet_Manager::get_instance();
        $credit_id = $wallet_manager->add_credit($credit_params);
        if (!$credit_id) {
            if (function_exists('pisol_aww_error_log')) {
                pisol_aww_error_log(sprintf('Failed to add credit for topup of order %s', $order->get_order_number()), 'wallet');
            }
            return false;
        }

        return $credit_id;
    }

    static function is_wallet_rechargeable_order( $order ) {
		$is_wallet_rechargeable_order = false;
        $recharge_product_id = Install::wallet_recharge_product();

        if ( ! $recharge_product_id ) return false;

        foreach ( $order->get_items( 'line_item' ) as $item ) {
            $product_id = $item['product_id'];
            if ( $recharge_product_id === $product_id ) {
                $is_wallet_rechargeable_order = true;
                break;
            }
        }
		
		return $is_wallet_rechargeable_order;
	}

    function addfees(){
        if (self::is_wallet_rechargeable_cart()) {
            $cart = WC()->cart;
            $extra_charge_enabled = get_option('pisol_aww_enable_gateway_charge',0);

            if(empty($extra_charge_enabled)){
                return;
            }

            $extra_charge = get_option('pisol_aww_payment_gateway_charges',[]);
            if(!is_array($extra_charge) || empty($extra_charge)){
                return;
            }
            
            $selected_method = self::getUserSelectedPaymentMethod();
            if(empty($selected_method)){
                return;
            }

            $extra_charge = isset($extra_charge[$selected_method]) ? $extra_charge[$selected_method] : 0;
            if(empty($extra_charge)){
                return;
            }

            $extra_charge_type = get_option('pisol_aww_gateway_charge_type','fixed');

            if($extra_charge_type == 'percentage'){
                $extra_charge = ($cart->subtotal * $extra_charge) / 100;
            }else{
                $extra_charge = $extra_charge;
            } 
            

            if ($extra_charge) {
                $fee_arg = array(
                        'id' => 'pisol-aww-fees',
                        'name'=> __('Payment Gateway Charges', 'advanced-wallet-for-woocommerce'),
                        'amount' => $extra_charge,
                        'taxable' =>  false,
                        'tax_class' => '' 
                );
                $cart->fees_api()->add_fee( $fee_arg );
            }
        }
    }

    static function purchased_credit_expiry_date(){
        $days = absint( get_option('pisol_aww_purhase_expiry_in_days', 0));
        $expiry_date = false;
        if($days > 0){
            $expiry_date = gmdate('Y-m-d', strtotime('+' . $days . ' days'));
        }

        return apply_filters('pisol_aww_purchased_credit_expiry_date', $expiry_date);
    }

    static function getUserSelectedPaymentMethod(){

        if(function_exists('WC') && isset(WC()->session) && is_object(WC()->session)) {
            
            $chosen_payment_method = WC()->session->get('chosen_payment_method');

            if(!empty($chosen_payment_method)){
                return $chosen_payment_method;
            }
        }
        
        if(!isset($_POST['post_data']) && !isset($_POST['payment_method'])) return false;
        
        if(isset($_POST['payment_method'])){
            $values['payment_method'] = sanitize_text_field( wp_unslash( $_POST['payment_method'] ) );
        }else{
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
            parse_str($_POST['post_data'], $values);
        }
        
        if(!empty($values['payment_method'])){
            $selected_method = $values['payment_method'];
            return $selected_method;
        }
        return false;
    }
}