<?php
/**
 * Wallet Manager Class
 * Handles all database operations for the WooCommerce wallet tables
 *
 * @package Advanced Wallet for WooCommerce
 * @since 1.0.0
 */

namespace PISOL\AWW\CLASSES;

defined('ABSPATH') || exit;

class Wallet_Manager {

    /**
     * The single instance of the class
     *
     * @var Wallet_Manager
     */
    private static $instance = null;

    /**
     * WordPress database object
     *
     * @var \wpdb
     */
    private $wpdb;

    /**
     * Cache group for wallet data
     *
     * @var string
     */
    private $cache_group = 'pisol_aww_wallet';

    /**
     * Singleton pattern - ensures only one instance is created
     *
     * @return Wallet_Manager
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        global $wpdb;
        $this->wpdb = $wpdb;
    }

    /**
     * Get table names with prefixes
     *
     * @return array Array of table names
     */
    private function get_table_names() {
        return [
            'credits' => $this->wpdb->base_prefix . 'pi_aww_wallet_credits',
            'debits' => $this->wpdb->base_prefix . 'pi_aww_wallet_debits',
            'map' => $this->wpdb->base_prefix . 'pi_aww_wallet_credit_debit_map'
        ];
    }

    /**
     * Clear cache for a specific user's wallet data
     *
     * @param int $user_id User ID
     * @param int $blog_id Blog ID (default: current blog)
     * @return void
     */
    private function clear_user_cache($user_id, $blog_id = 1) {
        $user_id = absint($user_id);
        $blog_id = absint($blog_id);
        
        $cache_keys = [
            "wallet_balance_{$user_id}_{$blog_id}",
            "wallet_credits_{$user_id}_{$blog_id}",
            "wallet_debits_{$user_id}_{$blog_id}",
        ];

        foreach ($cache_keys as $key) {
            wp_cache_delete($key, $this->cache_group);
        }
    }

    /**
     * Add credit to user wallet
     *
     * @param array $args {
     *     Required. An array of arguments.
     *     @type int    $user_id     Required. User ID to add credit to
     *     @type float  $amount      Required. Amount to credit
     *     @type string $currency    Optional. Currency code (default: site currency)
     *     @type string $expiry_date Optional. Expiry date in Y-m-d format
     *     @type string $source      Optional. Source of credit (e.g. 'admin', 'purchase')
     *     @type string $note        Optional. Note for this credit
     *     @type int    $blog_id     Optional. Blog ID for multisite (default: 1)
     * }
     * @return int|false Credit ID on success, false on failure
     */
    public function add_credit($args) {
        // Required fields
        if (empty($args['user_id']) || empty($args['amount']) || $args['amount'] <= 0) {
            return false;
        }
        
        $tables = $this->get_table_names();
        
        // Default values and validation
        $user_id = absint($args['user_id']);
        $amount = floatval($args['amount']);
        $currency = isset($args['currency']) ? sanitize_text_field($args['currency']) : get_woocommerce_currency();
        $blog_id = isset($args['blog_id']) ? absint($args['blog_id']) : 1;
        
        $data = [
            'user_id' => $user_id,
            'blog_id' => $blog_id,
            'amount' => $amount,
            'currency' => $currency,
            'created_at' => current_time('mysql', true),
            'is_deleted' => 0 // Set default value for is_deleted
        ];

        // Format specifiers for the base data
        $formats = [
            '%d', // user_id
            '%d', // blog_id
            '%f', // amount
            '%s', // currency
            '%s', // created_at
            '%d'  // is_deleted
        ];

        // Optional fields
        if (!empty($args['expiry_date'])) {
            $data['expiry_date'] = sanitize_text_field($args['expiry_date']);
            $formats[] = '%s'; // expiry_date format
        }

        if (!empty($args['source'])) {
            $data['source'] = sanitize_text_field($args['source']);
            $formats[] = '%s'; // source format
        }

        if (!empty($args['note'])) {
            $data['note'] = sanitize_textarea_field($args['note']);
            $formats[] = '%s'; // note format
        }

        // Insert credit into database
        $result = $this->wpdb->insert(
            $tables['credits'],
            $data,
            $formats
        );

        if ($result === false) {
            if (function_exists('pisol_aww_error_log')) {
                pisol_aww_error_log('Failed to add credit: ' . $this->wpdb->last_error, 'wallet');
            }
            return false;
        }

        $credit_id = $this->wpdb->insert_id;
        
        // Clear user cache
        $this->clear_user_cache($user_id, $blog_id);
        
        /**
         * Action triggered after adding credit to wallet
         *
         * @param int   $credit_id  The credit ID
         * @param array $data       Credit data that was inserted
         */
        do_action('pisol_aww_after_add_credit', $credit_id, $data);
        
        return $credit_id;
    }

    /**
     * Add debit to user wallet
     *
     * @param array $args {
     *     Required. An array of arguments.
     *     @type int    $user_id       Required. User ID to debit from
     *     @type float  $amount        Required. Amount to debit
     *     @type string $currency      Optional. Currency code (default: site currency)
     *     @type int    $reference_id  Optional. Reference ID (e.g. order ID)
     *     @type string $reference_type Optional. Reference type (e.g. 'order', 'manual')
     *     @type string $note          Optional. Note for this debit
     *     @type int    $blog_id       Optional. Blog ID for multisite (default: 1)
     * }
     * @return int|false Debit ID on success, false on failure
     */
    public function add_debit($args) {
        // Required fields
        if (empty($args['user_id']) || empty($args['amount']) || $args['amount'] <= 0) {
            return false;
        }

        // Check if user has sufficient balance
        $balance = $this->get_user_balance($args['user_id'], isset($args['currency']) ? $args['currency'] : null);
        if ($balance < $args['amount']) {
            return false;
        }
        
        $tables = $this->get_table_names();
        
        // Default values and validation
        $user_id = absint($args['user_id']);
        $amount = floatval($args['amount']);
        $currency = isset($args['currency']) ? sanitize_text_field($args['currency']) : get_woocommerce_currency();
        $blog_id = isset($args['blog_id']) ? absint($args['blog_id']) : 1;
        
        $data = [
            'user_id' => $user_id,
            'blog_id' => $blog_id,
            'amount' => $amount,
            'currency' => $currency,
            'created_at' => current_time('mysql', true),
            'is_deleted' => 0 // Set default value for is_deleted
        ];

        // Format specifiers for the base data
        $formats = [
            '%d', // user_id
            '%d', // blog_id
            '%f', // amount
            '%s', // currency
            '%s', // created_at
            '%d'  // is_deleted
        ];

        // Optional fields
        if (!empty($args['reference_id'])) {
            $data['reference_id'] = absint($args['reference_id']);
            $formats[] = '%d'; // reference_id format
        }

        if (!empty($args['reference_type'])) {
            $data['reference_type'] = sanitize_text_field($args['reference_type']);
            $formats[] = '%s'; // reference_type format
        }

        if (!empty($args['note'])) {
            $data['note'] = sanitize_textarea_field($args['note']);
            $formats[] = '%s'; // note format
        }

        // Insert debit into database
        $result = $this->wpdb->insert(
            $tables['debits'],
            $data,
            $formats
        );

        if ($result === false) {
            if (function_exists('pisol_aww_error_log')) {
                pisol_aww_error_log('Failed to add debit: ' . $this->wpdb->last_error, 'wallet');
            }
            return false;
        }

        $debit_id = $this->wpdb->insert_id;
        
        // Map this debit to available credits
        $mapped = $this->map_debit_to_credits($debit_id, $user_id, $amount, $currency, $blog_id);
        
        if (!$mapped) {
            // Roll back the debit if mapping failed
            $this->wpdb->delete($tables['debits'], ['debit_id' => $debit_id], ['%d']);
            return false;
        }
        
        // Clear user cache
        $this->clear_user_cache($user_id, $blog_id);
        
        /**
         * Action triggered after adding debit to wallet
         *
         * @param int   $debit_id  The debit ID
         * @param array $data      Debit data that was inserted
         */
        do_action('pisol_aww_after_add_debit', $debit_id, $data);
        
        return $debit_id;
    }

    /**
     * Map a debit to available credits
     *
     * @param int    $debit_id Debit ID
     * @param int    $user_id  User ID
     * @param float  $amount   Amount to debit
     * @param string $currency Currency code
     * @param int    $blog_id  Blog ID
     * @return bool Success or failure
     */
    private function map_debit_to_credits($debit_id, $user_id, $amount, $currency, $blog_id = 1) {
        $tables = $this->get_table_names();
        $remaining_amount = $amount;
        
        // Get available credits ordered by expiry date (earliest first), then creation date
        // Include is_deleted=0 in the query to only get active credits
        $credits = $this->wpdb->get_results($this->wpdb->prepare(
            "SELECT credit_id, amount, used_amount 
            FROM {$tables['credits']} 
            WHERE user_id = %d 
            AND blog_id = %d 
            AND currency = %s 
            AND (amount - used_amount) > 0 
            AND (expiry_date IS NULL OR expiry_date >= %s)
            AND is_deleted = 0
            ORDER BY CASE WHEN expiry_date IS NULL THEN 1 ELSE 0 END, expiry_date ASC, created_at ASC",
            $user_id, $blog_id, $currency, current_time('Y-m-d')
        ));
        
        if (empty($credits)) {
            return false;
        }
        
        foreach ($credits as $credit) {
            $available = $credit->amount - $credit->used_amount;
            $amount_to_use = min($available, $remaining_amount);
            
            // Update the credit's used amount
            $this->wpdb->update(
                $tables['credits'],
                ['used_amount' => $credit->used_amount + $amount_to_use],
                ['credit_id' => $credit->credit_id],
                ['%f'],
                ['%d']
            );
            
            // Create the mapping record
            $this->wpdb->insert(
                $tables['map'],
                [
                    'blog_id' => $blog_id,
                    'credit_id' => $credit->credit_id,
                    'debit_id' => $debit_id,
                    'amount_used' => $amount_to_use
                ],
                [
                    '%d', // blog_id
                    '%d', // credit_id
                    '%d', // debit_id
                    '%f'  // amount_used
                ]
            );
            
            $remaining_amount -= $amount_to_use;
            
            // Break if we've covered the full debit amount
            if ($remaining_amount <= 0.001) {
                break;
            }
        }
        
        // Check if we've mapped the full amount
        return ($remaining_amount <= 0.001);
    }

    /**
     * Get user balance
     *
     * @param int    $user_id  User ID
     * @param string $currency Currency code (null for site default)
     * @param int    $blog_id  Blog ID
     * @return float Balance amount
     */
    public function get_user_balance($user_id, $currency = null, $blog_id = 1) {
        $user_id = absint($user_id);
        $blog_id = absint($blog_id);
        
        if (empty($currency)) {
            $currency = get_woocommerce_currency();
        }

        // Try to get from cache first
        $cache_key = "wallet_balance_{$user_id}_{$blog_id}_{$currency}";
        $balance = wp_cache_get($cache_key, $this->cache_group);
        
        if ($balance !== false) {
            return (float)$balance;
        }
        
        $tables = $this->get_table_names();
        
        // Current date for expiry check
        $current_date = current_time('Y-m-d');
        
        // Calculate balance using get_var() with prepared query
        $query = "SELECT COALESCE(SUM(amount - used_amount), 0) 
                 FROM {$tables['credits']} 
                 WHERE user_id = %d 
                 AND blog_id = %d 
                 AND currency = %s 
                 AND (expiry_date IS NULL OR expiry_date >= %s)
                 AND is_deleted = 0";
        
        $balance = $this->wpdb->get_var($this->wpdb->prepare(
            $query, 
            $user_id, $blog_id, $currency, $current_date
        ));
        
        $balance = (float)$balance;
        
        // Cache the result
        wp_cache_set($cache_key, $balance, $this->cache_group, 3600); // 1 hour cache
        
        return $balance;
    }

    /**
     * Get detailed user credits
     *
     * @param int    $user_id  User ID
     * @param string $currency Currency code (null for site default)
     * @param int    $blog_id  Blog ID
     * @param array  $args     Optional arguments (limit, offset, order)
     * @return array Credits data
     */
    public function get_user_credits($user_id, $currency = null, $blog_id = 1, $args = []) {
        $user_id = absint($user_id);
        $blog_id = absint($blog_id);
        
        if (empty($currency)) {
            $currency = get_woocommerce_currency();
        }
        
        // Default args
        $defaults = [
            'limit' => 10,
            'offset' => 0,
            'order' => 'DESC', // DESC or ASC
            'include_expired' => false,
            'include_deleted' => false, // New parameter to control if deleted records are included
        ];
        
        $args = wp_parse_args($args, $defaults);
        $tables = $this->get_table_names();
        
        // Current date for expiry check
        $current_date = current_time('Y-m-d');
        
        $where = [
            'user_id = %d',
            'blog_id = %d',
            'currency = %s'
        ];
        
        $query_args = [$user_id, $blog_id, $currency];
        
        if (!$args['include_expired']) {
            $where[] = '(expiry_date IS NULL OR expiry_date >= %s)';
            $query_args[] = $current_date;
        }
        
        // Add condition for deleted records
        if (!$args['include_deleted']) {
            $where[] = 'is_deleted = 0';
        }
        
        $where_clause = implode(' AND ', $where);
        $order_clause = sanitize_sql_orderby("created_at {$args['order']}");
        $limit_clause = '';
        
        if ($args['limit'] > 0) {
            $limit_clause = $this->wpdb->prepare('LIMIT %d OFFSET %d', $args['limit'], $args['offset']);
        }
        
        $query = $this->wpdb->prepare(
            "SELECT * FROM {$tables['credits']} WHERE {$where_clause} ORDER BY {$order_clause} {$limit_clause}",
            ...$query_args
        );
        
        return $this->wpdb->get_results($query, ARRAY_A);
    }

    /**
     * Get detailed user debits
     *
     * @param int    $user_id  User ID
     * @param string $currency Currency code (null for site default)
     * @param int    $blog_id  Blog ID
     * @param array  $args     Optional arguments (limit, offset, order)
     * @return array Debits data
     */
    public function get_user_debits($user_id, $currency = null, $blog_id = 1, $args = []) {
        $user_id = absint($user_id);
        $blog_id = absint($blog_id);
        
        if (empty($currency)) {
            $currency = get_woocommerce_currency();
        }
        
        // Default args
        $defaults = [
            'limit' => 10,
            'offset' => 0,
            'order' => 'DESC', // DESC or ASC
            'include_deleted' => false, // New parameter to control if deleted records are included
        ];
        
        $args = wp_parse_args($args, $defaults);
        $tables = $this->get_table_names();
        
        $where = [
            'user_id = %d',
            'blog_id = %d',
            'currency = %s'
        ];
        
        $query_args = [$user_id, $blog_id, $currency];
        
        // Add condition for deleted records
        if (!$args['include_deleted']) {
            $where[] = 'is_deleted = 0';
        }
        
        $where_clause = implode(' AND ', $where);
        $order_clause = sanitize_sql_orderby("created_at {$args['order']}");
        $limit_clause = '';
        
        if ($args['limit'] > 0) {
            $limit_clause = $this->wpdb->prepare('LIMIT %d OFFSET %d', $args['limit'], $args['offset']);
        }
        
        $query = $this->wpdb->prepare(
            "SELECT * FROM {$tables['debits']} WHERE {$where_clause} ORDER BY {$order_clause} {$limit_clause}",
            ...$query_args
        );
        
        return $this->wpdb->get_results($query, ARRAY_A);
    }

    /**
     * Get credit details by ID
     *
     * @param int $credit_id Credit ID
     * @param bool $include_deleted Whether to include deleted credits
     * @return array|false Credit data or false if not found
     */
    public function get_credit($credit_id, $include_deleted = false) {
        $credit_id = absint($credit_id);
        $tables = $this->get_table_names();
        
        $where = ['credit_id = %d'];
        $params = [$credit_id];
        
        if (!$include_deleted) {
            $where[] = 'is_deleted = 0';
        }
        
        $where_clause = implode(' AND ', $where);
        
        $credit = $this->wpdb->get_row(
            $this->wpdb->prepare(
                "SELECT * FROM {$tables['credits']} WHERE $where_clause",
                ...$params
            ),
            ARRAY_A
        );
        
        return $credit ? $credit : false;
    }

    /**
     * Get debit details by ID
     *
     * @param int $debit_id Debit ID
     * @param bool $include_deleted Whether to include deleted debits
     * @return array|false Debit data or false if not found
     */
    public function get_debit($debit_id, $include_deleted = false) {
        $debit_id = absint($debit_id);
        $tables = $this->get_table_names();
        
        $where = ['debit_id = %d'];
        $params = [$debit_id];
        
        if (!$include_deleted) {
            $where[] = 'is_deleted = 0';
        }
        
        $where_clause = implode(' AND ', $where);
        
        $debit = $this->wpdb->get_row(
            $this->wpdb->prepare(
                "SELECT * FROM {$tables['debits']} WHERE $where_clause",
                ...$params
            ),
            ARRAY_A
        );
        
        return $debit ? $debit : false;
    }

    /**
     * Update credit details
     *
     * @param int   $credit_id Credit ID
     * @param array $data      Data to update
     * @return bool Success or failure
     */
    public function update_credit($credit_id, $data) {
        $credit_id = absint($credit_id);
        $tables = $this->get_table_names();
        
        // Get current credit to retain user_id and blog_id
        $current_credit = $this->get_credit($credit_id, true);
        if (!$current_credit) {
            return false;
        }

        // Fields that can be updated
        $allowed_fields = [
            'amount' => '%f',
            'used_amount' => '%f',
            'currency' => '%s',
            'expiry_date' => '%s',
            'source' => '%s',
            'note' => '%s',
            'is_deleted' => '%d'
        ];
        
        $update_data = [];
        $formats = [];
        
        foreach ($allowed_fields as $field => $format) {
            if (isset($data[$field])) {
                $update_data[$field] = $data[$field];
                $formats[] = $format;
            }
        }
        
        if (empty($update_data)) {
            return false;
        }
        
        $result = $this->wpdb->update(
            $tables['credits'],
            $update_data,
            ['credit_id' => $credit_id],
            $formats,
            ['%d']
        );
        
        if ($result !== false) {
            $this->clear_user_cache($current_credit['user_id'], $current_credit['blog_id']);
        }
        
        return ($result !== false);
    }

    /**
     * Update debit details
     *
     * @param int   $debit_id Debit ID
     * @param array $data     Data to update
     * @return bool Success or failure
     */
    public function update_debit($debit_id, $data) {
        $debit_id = absint($debit_id);
        $tables = $this->get_table_names();
        
        // Get current debit to retain user_id and blog_id
        $current_debit = $this->get_debit($debit_id, true);
        if (!$current_debit) {
            return false;
        }

        // Fields that can be updated
        $allowed_fields = [
            'amount' => '%f',
            'currency' => '%s',
            'reference_id' => '%d',
            'reference_type' => '%s',
            'note' => '%s',
            'is_deleted' => '%d'
        ];
        
        $update_data = [];
        $formats = [];
        
        foreach ($allowed_fields as $field => $format) {
            if (isset($data[$field])) {
                $update_data[$field] = $data[$field];
                $formats[] = $format;
            }
        }
        
        if (empty($update_data)) {
            return false;
        }
        
        // Don't allow amount change if there are mappings
        if (isset($update_data['amount'])) {
            $mappings_count = $this->wpdb->get_var($this->wpdb->prepare(
                "SELECT COUNT(*) FROM {$tables['map']} WHERE debit_id = %d",
                $debit_id
            ));
            
            if ($mappings_count > 0) {
                unset($update_data['amount']);
            }
        }
        
        if (empty($update_data)) {
            return false;
        }
        
        $result = $this->wpdb->update(
            $tables['debits'],
            $update_data,
            ['debit_id' => $debit_id],
            $formats,
            ['%d']
        );
        
        if ($result !== false) {
            $this->clear_user_cache($current_debit['user_id'], $current_debit['blog_id']);
        }
        
        return ($result !== false);
    }

    /**
     * Soft delete a credit record
     *
     * @param int $credit_id Credit ID to soft delete
     * @return bool Success or failure
     */
    public function soft_delete_credit($credit_id) {
        $credit_id = absint($credit_id);
        
        // Get credit details before deletion
        $credit = $this->get_credit($credit_id, true);
        if (!$credit) {
            return false;
        }
        
        // Check if credit has been used in mappings
        $mappings_count = $this->wpdb->get_var($this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->get_table_names()['map']} WHERE credit_id = %d",
            $credit_id
        ));
        
        if ($mappings_count > 0 && $credit['used_amount'] > 0) {
            // Credit is already in use, can't delete
            return false;
        }
        
        // Update is_deleted flag
        $result = $this->update_credit($credit_id, ['is_deleted' => 1]);
        
        if ($result) {
            $this->clear_user_cache($credit['user_id'], $credit['blog_id']);
            
            do_action('pisol_aww_after_soft_delete_credit', $credit_id, $credit);
        }
        
        return $result;
    }

    /**
     * Soft delete a debit record
     *
     * @param int $debit_id Debit ID to soft delete
     * @return bool Success or failure
     */
    public function soft_delete_debit($debit_id) {
        $debit_id = absint($debit_id);
        
        // Get debit details before deletion
        $debit = $this->get_debit($debit_id, true);
        if (!$debit) {
            return false;
        }
        
        // Update is_deleted flag
        $result = $this->update_debit($debit_id, ['is_deleted' => 1]);
        
        if ($result) {
            $this->clear_user_cache($debit['user_id'], $debit['blog_id']);
            
            do_action('pisol_aww_after_soft_delete_debit', $debit_id, $debit);
        }
        
        return $result;
    }

    /**
     * Restore a soft deleted credit record
     *
     * @param int $credit_id Credit ID to restore
     * @return bool Success or failure
     */
    public function restore_credit($credit_id) {
        $credit_id = absint($credit_id);
        
        // Get credit details
        $credit = $this->get_credit($credit_id, true);
        if (!$credit || $credit['is_deleted'] == 0) {
            return false;
        }
        
        // Update is_deleted flag
        $result = $this->update_credit($credit_id, ['is_deleted' => 0]);
        
        if ($result) {
            $this->clear_user_cache($credit['user_id'], $credit['blog_id']);
            
            do_action('pisol_aww_after_restore_credit', $credit_id, $credit);
        }
        
        return $result;
    }

    /**
     * Restore a soft deleted debit record
     *
     * @param int $debit_id Debit ID to restore
     * @return bool Success or failure
     */
    public function restore_debit($debit_id) {
        $debit_id = absint($debit_id);
        
        // Get debit details
        $debit = $this->get_debit($debit_id, true);
        if (!$debit || $debit['is_deleted'] == 0) {
            return false;
        }
        
        // Update is_deleted flag
        $result = $this->update_debit($debit_id, ['is_deleted' => 0]);
        
        if ($result) {
            $this->clear_user_cache($debit['user_id'], $debit['blog_id']);
            
            do_action('pisol_aww_after_restore_debit', $debit_id, $debit);
        }
        
        return $result;
    }

    /**
     * Delete credit permanently (hard delete)
     *
     * @param int $credit_id Credit ID
     * @return bool Success or failure
     */
    public function delete_credit($credit_id) {
        $credit_id = absint($credit_id);
        $tables = $this->get_table_names();
        
        // Get credit details before deletion
        $credit = $this->get_credit($credit_id, true);
        if (!$credit) {
            return false;
        }
        
        // Check if credit has been used in mappings
        $mappings_count = $this->wpdb->get_var($this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$tables['map']} WHERE credit_id = %d",
            $credit_id
        ));
        
        if ($mappings_count > 0) {
            // Credit is already in use, can't delete
            return false;
        }
        
        $result = $this->wpdb->delete(
            $tables['credits'],
            ['credit_id' => $credit_id],
            ['%d']
        );
        
        if ($result) {
            $this->clear_user_cache($credit['user_id'], $credit['blog_id']);
            do_action('pisol_aww_after_hard_delete_credit', $credit_id, $credit);
        }
        
        return (bool)$result;
    }

    /**
     * Delete debit permanently (hard delete)
     *
     * @param int $debit_id Debit ID
     * @return bool Success or failure
     */
    public function delete_debit($debit_id) {
        $debit_id = absint($debit_id);
        $tables = $this->get_table_names();
        
        // Get debit details before deletion
        $debit = $this->get_debit($debit_id, true);
        if (!$debit) {
            return false;
        }
        
        // Check if debit has been used in mappings
        $mappings_count = $this->wpdb->get_var($this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$tables['map']} WHERE debit_id = %d",
            $debit_id
        ));
        
        if ($mappings_count > 0) {
            // First delete all mappings
            $this->wpdb->delete(
                $tables['map'],
                ['debit_id' => $debit_id],
                ['%d']
            );
        }
        
        $result = $this->wpdb->delete(
            $tables['debits'],
            ['debit_id' => $debit_id],
            ['%d']
        );
        
        if ($result) {
            $this->clear_user_cache($debit['user_id'], $debit['blog_id']);
            do_action('pisol_aww_after_hard_delete_debit', $debit_id, $debit);
        }
        
        return (bool)$result;
    }

    /**
     * Get wallet transaction history for a user
     *
     * @param int    $user_id  User ID
     * @param string $currency Currency code
     * @param int    $blog_id  Blog ID
     * @param array  $args     Arguments for filtering and pagination
     * @return array Transaction history
     */
    public function get_user_transactions($user_id, $currency = null, $blog_id = 1, $args = []) {
        $user_id = absint($user_id);
        $blog_id = absint($blog_id);
        
        if (empty($currency)) {
            $currency = get_woocommerce_currency();
        }
        
        // Default args
        $defaults = [
            'limit' => 20,
            'offset' => 0,
            'order' => 'DESC',
            'transaction_type' => 'all', // 'all', 'credit', or 'debit'
            'include_deleted' => false, // Whether to include soft-deleted transactions
        ];
        
        $args = wp_parse_args($args, $defaults);
        $tables = $this->get_table_names();
        
        // Base conditions for both credits and debits
        $conditions = [
            "user_id = %d",
            "blog_id = %d", 
            "currency = %s"
        ];
        
        if (!$args['include_deleted']) {
            $conditions[] = "is_deleted = 0";
        }
        
        $conditions_str = implode(' AND ', $conditions);
        $query_params = [$user_id, $blog_id, $currency];
        
        $transactions = [];

        // Get credits
        if ($args['transaction_type'] === 'all' || $args['transaction_type'] === 'credit') {
            $credit_query = $this->wpdb->prepare(
                "SELECT 
                credit_id AS id,
                'credit' AS type,
                amount,
                used_amount,
                currency,
                expiry_date,
                source,
                note,
                created_at,
                is_deleted
                FROM {$tables['credits']}
                WHERE {$conditions_str}
                ORDER BY created_at {$args['order']}
                LIMIT %d OFFSET %d",
                array_merge($query_params, [$args['limit'], $args['offset']])
            );
            
            $credits = $this->wpdb->get_results($credit_query, ARRAY_A);
            $transactions = array_merge($transactions, $credits ?: []);
        }

        // Get debits
        if ($args['transaction_type'] === 'all' || $args['transaction_type'] === 'debit') {
            $debit_query = $this->wpdb->prepare(
                "SELECT 
                debit_id AS id,
                'debit' AS type,
                amount,
                0 AS used_amount,
                currency,
                NULL AS expiry_date,
                reference_type AS source,
                note,
                created_at,
                is_deleted
                FROM {$tables['debits']}
                WHERE {$conditions_str}
                ORDER BY created_at {$args['order']}
                LIMIT %d OFFSET %d",
                array_merge($query_params, [$args['limit'], $args['offset']])
            );
            
            $debits = $this->wpdb->get_results($debit_query, ARRAY_A);
            $transactions = array_merge($transactions, $debits ?: []);
        }

        // Sort transactions by date
        if ($args['transaction_type'] === 'all') {
            usort($transactions, function($a, $b) use ($args) {
                $order = ($args['order'] === 'DESC') ? -1 : 1;
                return $order * (strtotime($a['created_at']) - strtotime($b['created_at']));
            });
            
            // Apply limit after merging
            $transactions = array_slice($transactions, 0, $args['limit']);
        }
        
        return $transactions;
    }

    /**
     * Check if a user has sufficient balance for a specified amount
     *
     * @param int    $user_id  User ID
     * @param float  $amount   Amount to check
     * @param string $currency Currency code
     * @param int    $blog_id  Blog ID
     * @return bool True if sufficient balance, false otherwise
     */
    public function has_sufficient_balance($user_id, $amount, $currency = null, $blog_id = 1) {
        $balance = $this->get_user_balance($user_id, $currency, $blog_id);
        return ($balance >= $amount);
    }

    /**
     * Get expired credits for cleanup or notification purposes
     *
     * @param string $before_date Date string in Y-m-d format (credits expired before this date)
     * @param bool $include_deleted Whether to include soft-deleted credits
     * @return array Expired credits
     */
    public function get_expired_credits($before_date = '', $include_deleted = false) {
        $tables = $this->get_table_names();
        
        if (empty($before_date)) {
            $before_date = current_time('Y-m-d');
        } else {
            $before_date = sanitize_text_field($before_date);
        }
        
        $where = [
            "expiry_date IS NOT NULL", 
            "expiry_date < %s",
            "(amount - used_amount) > 0"
        ];
        
        $params = [$before_date];
        
        if (!$include_deleted) {
            $where[] = "is_deleted = 0";
        }
        
        $where_clause = implode(' AND ', $where);
        
        return $this->wpdb->get_results($this->wpdb->prepare(
            "SELECT * FROM {$tables['credits']} WHERE {$where_clause}",
            ...$params
        ), ARRAY_A);
    }

    /**
     * Handle expired credits (mark as expired or remove them)
     *
     * @param string $before_date Date string in Y-m-d format
     * @param string $action      'mark', 'soft_delete', or 'hard_delete'
     * @return int Number of affected credits
     */
    public function handle_expired_credits($before_date = '', $action = 'mark') {
        $expired_credits = $this->get_expired_credits($before_date);
        $count = 0;
        
        if (empty($expired_credits)) {
            return 0;
        }
        
        foreach ($expired_credits as $credit) {
            switch ($action) {
                case 'mark':
                    // Update note to indicate expiration
                    $note = !empty($credit['note']) ? $credit['note'] . ' | ' : '';
                    $note .= 'Expired on ' . current_time('Y-m-d');
                    
                    $this->update_credit($credit['credit_id'], ['note' => $note]);
                    $count++;
                    break;
                    
                case 'soft_delete':
                    // Soft delete the credit
                    if ($this->soft_delete_credit($credit['credit_id'])) {
                        $count++;
                    }
                    break;
                    
                case 'hard_delete':
                    // Only delete credits that have not been used
                    if ($credit['used_amount'] <= 0) {
                        if ($this->delete_credit($credit['credit_id'])) {
                            $count++;
                        }
                    }
                    break;
            }
        }
        
        return $count;
    }

    /**
     * Process expired cashbacks
     * 
     * This method is intended to be run on a scheduled basis to handle expired cashback credits.
     * It will mark expired credits as such and optionally add order notes.
     * 
     * @param string $action How to handle expired credits: 'mark' (default), 'soft_delete', or 'notify'
     * @return array Report of processed expired cashbacks
     */
    public function process_expired_cashbacks($action = 'mark') {
        // Get current date in Y-m-d format
        $current_date = current_time('Y-m-d');
        
        // Get expired cashback credits
        $expired_credits = $this->get_expired_credits($current_date);
        
        if (empty($expired_credits)) {
            return array(
                'status' => 'success',
                'processed' => 0,
                'message' => __('No expired cashbacks to process.', 'advanced-wallet-for-woocommerce')
            );
        }
        
        $processed_count = 0;
        $failed_count = 0;
        $processed_ids = array();
        
        foreach ($expired_credits as $credit) {
            // Only process cashback credits
            if ($credit['source'] !== 'cashback') {
                continue;
            }
            
            $credit_id = $credit['credit_id'];
            $user_id = $credit['user_id'];
            $amount = $credit['amount'] - $credit['used_amount'];
            
            // Skip if already fully used
            if ($amount <= 0) {
                continue;
            }
            
            $success = false;
            
            switch ($action) {
                case 'soft_delete':
                    // Soft delete the credit
                    $success = $this->soft_delete_credit($credit_id);
                    break;
                    
                case 'notify':
                    // Just update the note to indicate expiration without deleting
                    $note = !empty($credit['note']) ? $credit['note'] . ' | ' : '';
                    $note .= __('Expired on', 'advanced-wallet-for-woocommerce') . ' ' . $current_date;
                    
                    $success = $this->update_credit($credit_id, array('note' => $note));
                    
                    // Notify the user about expired cashback if email notifications are enabled
                    if ($success) {
                        do_action('pisol_aww_cashback_expired', $credit_id, $user_id, $amount);
                    }
                    break;
                    
                case 'mark':
                default:
                    // Mark as expired by updating the note and setting is_deleted = 1
                    $note = !empty($credit['note']) ? $credit['note'] . ' | ' : '';
                    $note .= __('Expired on', 'advanced-wallet-for-woocommerce') . ' ' . $current_date;
                    
                    $success = $this->update_credit($credit_id, array(
                        'note' => $note,
                        'is_deleted' => 1
                    ));
                    break;
            }
            
            if ($success) {
                $processed_count++;
                $processed_ids[] = $credit_id;
                
                // Trigger an action that can be used for notifications or other processing
                do_action('pisol_aww_cashback_expired_processed', $credit_id, $user_id, $amount, $action);
            } else {
                $failed_count++;
            }
        }
        
        
        $message = sprintf(
            // translators: %1$d is the number of processed expired cashbacks, %2$d is the number of failures
            __('Processed %1$d expired cashbacks (%2$d failed).', 'advanced-wallet-for-woocommerce'),
            $processed_count,
            $failed_count
        );
        
        return array(
            'status' => ($failed_count === 0) ? 'success' : 'partial',
            'processed' => $processed_count,
            'failed' => $failed_count,
            'processed_ids' => $processed_ids,
            'message' => $message
        );
    }

    /**
     * Combined function to get paginated transactions for either specific user or all users
     * 
     * @param int $user_id User ID (0 for all users)
     * @param int $current_page Current page number
     * @param int $per_page Items per page
     * @return array Associative array with transactions, pagination data, and total count
     */
    static function get_paginated_transactions_combined($user_id = 0, $current_page = 1, $per_page = 10) {
        global $wpdb;
        $tables = [
            'credits' => $wpdb->base_prefix . 'pi_aww_wallet_credits',
            'debits' => $wpdb->base_prefix . 'pi_aww_wallet_debits'
        ];
        
        // Calculate offset
        $offset = ($current_page - 1) * $per_page;
        
        // Build the WHERE clause for filtering by user_id if needed
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- $user_where_credits is safely prepared above
        $user_where_credits = $user_id > 0 ? $wpdb->prepare("WHERE user_id = %d", $user_id) : "";
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- $user_where_debits is safely prepared above
        $user_where_debits = $user_id > 0 ? $wpdb->prepare("WHERE user_id = %d", $user_id) : "";
        
        // Prepare the table names - we can't use placeholders for table names
        $credits_table = esc_sql($tables['credits']);
        $debits_table = esc_sql($tables['debits']);
        
        // Count total transactions using a UNION query
        if ($user_id > 0) {
            // If filtering by user, use a prepared statement with the user ID
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table names can't be parameterized
            $total_count = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM (
                SELECT credit_id FROM {$credits_table} WHERE user_id = %d
                UNION ALL
                SELECT debit_id FROM {$debits_table} WHERE user_id = %d
            ) AS all_transactions", $user_id, $user_id));
        } else {
            // If not filtering by user
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table names can't be parameterized
            $total_count = $wpdb->get_var("SELECT COUNT(*) FROM (
                SELECT credit_id FROM {$credits_table}
                UNION ALL
                SELECT debit_id FROM {$debits_table}
            ) AS all_transactions");
        }
                
        // Calculate total pages
        $total_pages = ceil($total_count / $per_page);
        
        // Build and execute query safely        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table names can't be parameterized
        $transactions = $wpdb->get_results(
            $wpdb->prepare("SELECT * FROM (
            SELECT 
                credit_id as id, 
                user_id, 
                amount, 
                source, 
                note, 
                used_amount, 
                is_deleted, 
                created_at, 
                expiry_date, 
                'credit' as type 
            FROM {$credits_table}
            " . $user_where_credits . "
            
            UNION ALL
            
            SELECT 
                debit_id as id, 
                user_id, 
                amount, 
                COALESCE(reference_type, '') as source, 
                note, 
                0 as used_amount, 
                is_deleted, 
                created_at, 
                NULL as expiry_date, 
                'debit' as type 
            FROM {$debits_table}
            " . $user_where_debits . "
        ) AS all_transactions 
        ORDER BY created_at DESC
        LIMIT %d OFFSET %d", $per_page, $offset),
            ARRAY_A
        );
        
        return [
            'transactions' => $transactions ?: [],
            'total_count' => $total_count,
            'total_pages' => $total_pages,
            'current_page' => $current_page,
            'per_page' => $per_page
        ];
    }

    static function get_expiry_date($expiry_days) {
        if ($expiry_days <= 0) {
            return null; // No expiry
        }
        
        // Calculate expiry date based on current date and expiry days
        return date('Y-m-d', strtotime('+' . intval($expiry_days) . ' days', current_time('timestamp')));
    }
}