<?php 
namespace PISOL\AWW\CLASSES;

defined('ABSPATH') || exit;

class Wallet_Coupon {

    private static $instance = null;

    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_filter('woocommerce_coupon_get_discount_amount', [$this, 'make_coupon_zero_and_store_original_discount'], 10, 5);

        add_filter('woocommerce_cart_totals_coupon_html', [$this, 'custom_coupon_display_message'], 10, 2);

        add_action('woocommerce_checkout_create_order_coupon_item', [$this, 'add_wallet_coupon_meta_to_order_item'], 10, 4);

        add_action('woocommerce_order_status_changed', [$this, 'credit_wallet_coupon'], 10, 4);
    }

    /**
     * Filter out cart items that are either not in cart anymore or not eligible for this coupon
     *
     * @param array $original_discount
     * @param array $valid_keys
     * @param array $cart
     * @param WC_Coupon $coupon
     * @return array
     */
    private function filter_valid_cart_items($original_discount, $valid_keys, $cart, $coupon) {
        $type = $coupon->get_discount_type();
        $check_product_validity = in_array($type, ['percent', 'fixed_product'], true);
        return array_filter(
            $original_discount,
            function ($k) use ($valid_keys, $cart, $coupon, $check_product_validity) {
                if (!in_array($k, $valid_keys, true)) return false;

                $cart_item = $cart[$k];
                $product = $cart_item['data'];

                if ($check_product_validity) {
                    /**
                     * is valid for product check is only valid for percent and fixed_product types coupons only
                     */
                    return $coupon->is_valid_for_product($product, $cart_item);
                }

                return true; // for fixed_cart and others
            },
            ARRAY_FILTER_USE_KEY
        );
    }

    static function get_supported_coupon_types() {
        return apply_filters(
            'pi_aww_supported_wallet_coupon_types',
            [
            'fixed_cart',
            'percent',
            'fixed_product'
            ]
        );
    }

    static function supported_coupon_type_for_wallet_coupon($coupon) {
        $supported_types = self::get_supported_coupon_types();

        // Check if the coupon is a wallet coupon
        if ('yes' === $coupon->get_meta('_pi_aww_is_wallet_coupon', true)  && in_array($coupon->get_discount_type(), $supported_types, true)) {
            return true;
        }

        return false;
    }

    function make_coupon_zero_and_store_original_discount($discount, $discounting_amount, $cart_item, $single, $coupon) {
        if (is_admin() && !defined('DOING_AJAX')) return $discount;

        $coupon_code = $coupon->get_code();

        if ( !self::supported_coupon_type_for_wallet_coupon($coupon) ) {
            WC()->session->set('original_discount_' . $coupon_code, []);
            return $discount;
        }

        $key = $cart_item['key'] ?? '';

        // Get current cart keys
        $cart = WC()->cart->get_cart();
        $valid_keys = array_keys($cart);

        // Retrieve original discount data
        $original_discount = WC()->session->get('original_discount_' . $coupon_code);
        if (!is_array($original_discount)) {
            $original_discount = [];
        }

        $original_discount[$key] = $discount;

        // Filter out cart items that are either not in cart anymore or not eligible for this coupon
        $original_discount = $this->filter_valid_cart_items($original_discount, $valid_keys, $cart, $coupon);

        WC()->session->set('original_discount_' . $coupon_code, $original_discount);

        return 0;
    }

    function custom_coupon_display_message($value, $coupon) {
        $coupon_code = $coupon->get_code();
        

        if (self::supported_coupon_type_for_wallet_coupon($coupon)) {
            $wallet_coupon_amount = self::get_wallet_coupon_amount($coupon);
            $wallet_coupon_message = __('Get Wallet credit %s', 'advanced-wallet-for-woocommerce');
            $message = '<span class="custom-coupon-message" style="color: green;">' . sprintf($wallet_coupon_message, wc_price($wallet_coupon_amount)) . '</span>';
            // Keep the [remove] button if present in $value
            if (strpos($value, 'class="woocommerce-remove-coupon"') !== false) {
                // Try to extract the [remove] button
                if (preg_match('/(<a[^>]*class=\"woocommerce-remove-coupon\"[^>]*>.*?<\/a>)/i', $value, $matches)) {
                    $remove_button = $matches[1];
                    return $message . ' ' . $remove_button;
                }
            }
            return $message;
        }

        return $value;
    }

    static function get_wallet_coupon_amount($coupon) {
        
        $original_discount = WC()->session->get('original_discount_' . $coupon->get_code());
        if (is_array($original_discount) && !empty($original_discount)) {
            return array_sum($original_discount);
        }
        
        return 0;
    }

    function add_wallet_coupon_meta_to_order_item($item, $code, $coupon, $order) {
        if (self::supported_coupon_type_for_wallet_coupon($coupon)) {
            $wallet_coupon_amount = self::get_wallet_coupon_amount($coupon);
            $item->add_meta_data('_pi_aww_is_wallet_coupon', 'yes', true);
            $item->add_meta_data('_pi_aww_wallet_coupon_amount', $wallet_coupon_amount, true);
            $item->add_meta_data('_pi_aww_wallet_coupon_credited', false, true);

            $order->add_meta_data('_pi_aww_has_wallet_coupon', 'yes');
        }
    }

    function credit_wallet_coupon($order_id, $old_status, $new_status, $order) {
        $state_to_credit_balance = $this->order_state_to_credit_wallet_coupon_amt_to_wallet();
        if (!in_array($new_status, $state_to_credit_balance)) {
            return;
        }

        $has_wallet_coupon = $order->get_meta('_pi_aww_has_wallet_coupon', true);
        if ($has_wallet_coupon !== 'yes') {
            return;
        }

        $user_id = $order->get_user_id();
        if (!$user_id) {
            // Log error or handle case where user ID is not available
            pisol_aww_error_log(sprintf(__('Wallet coupon cant be credited as Order #%s does not have a valid user ID', 'advanced-wallet-for-woocommerce'), $order->get_order_number()));
            return;
        }

        $wallet_coupons = $order->get_items('coupon');
        foreach ($wallet_coupons as $item_id => $item) {
            if ($item->get_meta('_pi_aww_is_wallet_coupon', true) === 'yes') {
                $amount = $item->get_meta('_pi_aww_wallet_coupon_amount', true);
                if ($amount > 0 && !$item->get_meta('_pi_aww_wallet_coupon_credited', true)) {

                    $credit_id = $this->add_amount_to_wallet($amount, $order);

                    if ($credit_id) {
                        Email::send_email(
                            $order->get_billing_email(),
                            sprintf(__('Wallet Coupon Credited for Order #%s', 'advanced-wallet-for-woocommerce'), $order->get_order_number()),
                            sprintf(__('A wallet credit of %s has been credited to your wallet for order #%s.', 'advanced-wallet-for-woocommerce'), wc_price($amount, ['currency' => $order->get_currency()]), $order->get_order_number()),
                            __('Wallet coupon credited', 'advanced-wallet-for-woocommerce')
                        );
                        // Add credit ID to order item meta
                        $item->add_meta_data('_pi_aww_wallet_coupon_credit_id', $credit_id, true);
                        $item->update_meta_data('_pi_aww_wallet_coupon_credited', true);
                    } else {
                        // Log error or handle failure to credit wallet
                        pisol_aww_error_log(sprintf(__('Failed to credit wallet for order #%s with coupon %s', 'advanced-wallet-for-woocommerce'), $order->get_order_number(), $item->get_name()), 'wallet_coupon_credit_error');
                    }
                    
                    $item->save();
                }else{
                    pisol_aww_error_log(sprintf(__('Wallet coupon already credited to wallet for order #%s with coupon %s', 'advanced-wallet-for-woocommerce'), $order->get_order_number(), $item->get_name()), 'wallet_coupon_credit_error');
                }
            }
        }
    }

    function add_amount_to_wallet($amount, $order) {
        $credit_params = array(
                        "user_id" => $order->get_user_id(),
                        "amount" => $amount,
                        "currency" => $order->get_currency(),
                        "source" => "wallet_coupon",
                        /* translators: %s: order number */
                        "note" => sprintf(__("Wallet coupon credited for order #%s", "advanced-wallet-for-woocommerce"), $order->get_order_number())
                    ); 
        $wallet_manager = Wallet_Manager::get_instance();
        $credit_id = $wallet_manager->add_credit($credit_params);
        return $credit_id;
    }

    function order_state_to_credit_wallet_coupon_amt_to_wallet(){
        $order_states = get_option('pisol_aww_wallet_coupon_order_states', ['completed', 'processing']);
        return apply_filters('pi_aww_credit_wallet_coupon_on_order_status', $order_states);
    }
}