<?php
/**
 * Review Reward Class
 * Handles giving wallet rewards to customers when their product review is approved
 *
 * @package Advanced Wallet for WooCommerce
 * @since 1.0.0
 */

namespace PISOL\AWW\CLASSES;

defined('ABSPATH') || exit;

class Review_Reward {
    /**
     * The single instance of the class
     *
     * @var Review_Reward
     */
    private static $instance = null;

    /**
     * Wallet manager instance
     *
     * @var Wallet_Manager
     */
    private $wallet_manager;

    /**
     * Singleton pattern - ensures only one instance is created
     *
     * @return Review_Reward
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->wallet_manager = Wallet_Manager::get_instance();
        // Hook into comment status change (for admin approval)
        add_action('wp_set_comment_status', array($this, 'maybe_reward_on_approval'), 10, 2);
        // Hook into auto-approved comments
        add_action('comment_post', array($this, 'maybe_reward_on_auto_approve'), 20, 2);
    }

    /**
     * Check if review reward is enabled
     *
     * @return bool
     */
    public function is_review_reward_enabled() {
        return (bool) get_option('pisol_aww_enable_review_reward', false);
    }

    /**
     * Get review reward amount
     *
     * @return float
     */
    public function get_reward_amount() {
        return floatval(get_option('pisol_aww_review_reward_amount', 0));
    }

    /**
     * Get review reward expiry days
     *
     * @return int
     */
    public function get_expiry_days() {
        return absint(get_option('pisol_aww_review_reward_expires_in_days', 0));
    }

    /**
     * Maybe reward when review is approved by admin
     *
     * @param int $comment_id
     * @param string $comment_status
     */
    public function maybe_reward_on_approval($comment_id, $comment_status) {
        if ($comment_status !== 'approve') {
            return;
        }
        $this->maybe_give_review_reward($comment_id);
    }

    /**
     * Maybe reward when review is auto-approved
     *
     * @param int $comment_id
     * @param int $approved
     */
    public function maybe_reward_on_auto_approve($comment_id, $approved) {
        if ($approved == 1) { // 1 means auto-approved
            $this->maybe_give_review_reward($comment_id);
        }
    }

    /**
     * Check if user has already received a review reward for this product
     *
     * @param int $user_id
     * @param int $product_id
     * @return bool
     */
    private function has_already_rewarded_for_product($user_id, $product_id) {
        $enable_single_reward = get_option('pisol_aww_single_reward_per_product', false);
        if (!$enable_single_reward) {
            return false; // If single reward per product is not enabled, no need to check
        }

        $args = array(
            'user_id' => $user_id,
            'post_id' => $product_id,
            'type'    => 'review',
            'meta_query' => array(
                array(
                    'key'   => '_pisol_aww_review_reward_given',
                    'value' => '1',
                ),
            ),
            'fields' => 'ids',
            'number' => 1,
        );
        $existing_rewarded = get_comments($args);
        return !empty($existing_rewarded);
    }

    /**
     * Give review reward if not already given and all conditions met
     *
     * @param int $comment_id
     */
    private function maybe_give_review_reward($comment_id) {
        // Only for product reviews
        $comment = get_comment($comment_id);
        if (!$comment || $comment->comment_type !== 'review') {
            return;
        }
        // Only for registered users
        $user_id = intval($comment->user_id);
        if ($user_id <= 0) {
            return;
        }
        // Ensure only one reward per product per user
        $product_id = intval($comment->comment_post_ID);
        if ($this->has_already_rewarded_for_product($user_id, $product_id)) {
            pisol_aww_error_log(sprintf('Reward already given for the product review to user: %s for product: %s', $user_id, $product_id), 'review_reward');
            return;
        }

        // Check if already rewarded
        if (get_comment_meta($comment_id, '_pisol_aww_review_reward_given', true)) {
            pisol_aww_error_log(sprintf('Reward already given for comment id: %s', $comment_id), 'review_reward');
            return;
        }
        // Check if reward is enabled
        if (!$this->is_review_reward_enabled()) {
            return;
        }

        $amount = $this->get_reward_amount();
        if ($amount <= 0) {
            return;
        }

        $expiry_days = $this->get_expiry_days();
        $expiry_date = '';
        if ($expiry_days > 0) {
            $expiry_date = $this->wallet_manager::get_expiry_date($expiry_days);
        }

        $currency = get_woocommerce_currency();
        $credit_args = array(
            'user_id'     => $user_id,
            'amount'      => $amount,
            'currency'    => $currency,
            'expiry_date' => $expiry_date,
            'source'      => 'review_reward',
            'note'        => sprintf(__('Reward for product review (Comment ID: %d)', 'advanced-wallet-for-woocommerce'), $comment_id),
        );

        $credit_id = $this->wallet_manager->add_credit($credit_args);
        if ($credit_id) {
            
            Email::send_email(
                get_userdata($user_id)->user_email,
                __('Your Product Review Reward', 'advanced-wallet-for-woocommerce'),
                sprintf(__('You have received a wallet credit of %s for your product review. Thank you!', 'advanced-wallet-for-woocommerce'), wc_price($amount)),
                __('Product Review Reward', 'advanced-wallet-for-woocommerce')
            );

            // Mark as rewarded
            update_comment_meta($comment_id, '_pisol_aww_review_reward_given', 1);
            // Add marker for reference
            update_comment_meta($comment_id, '_pisol_aww_review_reward_credit_id', $credit_id);
        }
    }
}

