<?php
/**
 * Registration Reward Class
 * Handles giving wallet rewards to customers when they register on the website
 *
 * @package Advanced Wallet for WooCommerce
 * @since 1.0.0
 */

namespace PISOL\AWW\CLASSES;

defined('ABSPATH') || exit;

class Registration_Reward {
    /**
     * The single instance of the class
     *
     * @var Registration_Reward
     */
    private static $instance = null;

    /**
     * Wallet manager instance
     *
     * @var Wallet_Manager
     */
    private $wallet_manager;

    /**
     * Singleton pattern - ensures only one instance is created
     *
     * @return Registration_Reward
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->wallet_manager = Wallet_Manager::get_instance();
        // Hook into user registration
        add_action('user_register', array($this, 'maybe_give_registration_reward'), 10, 1);
    }

    /**
     * Check if registration reward is enabled
     *
     * @return bool
     */
    public function is_registration_reward_enabled() {
        return (bool) get_option('pisol_aww_enable_registration_reward', false);
    }

    /**
     * Get registration reward amount
     *
     * @return float
     */
    public function get_reward_amount() {
        return floatval(get_option('pisol_aww_registration_reward_amount', 0));
    }

    /**
     * Get registration reward expiry days
     *
     * @return int
     */
    public function get_expiry_days() {
        return absint(get_option('pisol_aww_registration_reward_expires_in_days', 0));
    }

    /**
     * Maybe give registration reward to user
     *
     * @param int $user_id
     */
    public function maybe_give_registration_reward($user_id) {
        $user_id = absint($user_id);
        if ($user_id <= 0) {
            return;
        }
        // Check if already rewarded
        if (get_user_meta($user_id, '_pisol_aww_registration_reward_given', true)) {
            return;
        }
        // Check if reward is enabled
        if (!$this->is_registration_reward_enabled()) {
            return;
        }
        $amount = $this->get_reward_amount();
        if ($amount <= 0) {
            return;
        }
        $expiry_days = $this->get_expiry_days();
        $expiry_date = '';
        if ($expiry_days > 0) {
            $expiry_date = $this->wallet_manager::get_expiry_date($expiry_days);
        }
        $currency = function_exists('get_woocommerce_currency') ? get_woocommerce_currency() : 'USD';
        $credit_args = array(
            'user_id'     => $user_id,
            'amount'      => $amount,
            'currency'    => $currency,
            'expiry_date' => $expiry_date,
            'source'      => 'registration_reward',
            'note'        => sprintf(__('Reward for registration (User ID: %d)', 'advanced-wallet-for-woocommerce'), $user_id),
        );
        $credit_id = $this->wallet_manager->add_credit($credit_args);
        if ($credit_id) {
            Email::send_email(
                get_userdata($user_id)->user_email,
                __('Registration Reward', 'advanced-wallet-for-woocommerce'),
                sprintf(__('You have received a wallet credit of %s for registering on our site.', 'advanced-wallet-for-woocommerce'), wc_price($amount)),
                __('Registration Reward', 'advanced-wallet-for-woocommerce')
            );
            // Mark as rewarded
            update_user_meta($user_id, '_pisol_aww_registration_reward_given', 1);
            update_user_meta($user_id, '_pisol_aww_registration_reward_credit_id', $credit_id);
        }
    }
}
