<?php
/**
 * Database Manager Class
 * Handles database table creation and updates for the Advanced Wallet for WooCommerce plugin
 *
 * @package Advanced Wallet for WooCommerce
 * @since 1.0.0
 */

namespace PISOL\AWW\CLASSES;

defined('ABSPATH') || exit;

class Database_Manager {

    /**
     * The single instance of the class.
     *
     * @var Database_Manager
     */
    private static $instance = null;

    /**
     * The current database version
     *
     * @var string
     */
    private $db_version = '1.0.0';

    /**
     * Option name to store the database version
     *
     * @var string
     */
    private $db_version_option = 'pisol_aww_db_version';

    /**
     * Singleton pattern - ensures only one instance is created
     *
     * @return Database_Manager
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        register_activation_hook(PISOL_AWW_WOOCOMMERCE_PLUGIN_FILE, array($this, 'install_database'));
        add_action('plugins_loaded', array($this, 'check_db_version'));
    }

    /**
     * Check if database needs updating
     *
     * @return void
     */
    public function check_db_version() {
        $installed_version = get_option($this->db_version_option);
        
        // If the installed version doesn't match the current version, update the database
        if ($installed_version !== $this->db_version) {
            $this->install_database();
        }
    }

    /**
     * Create/update database tables
     *
     * @return void
     */
    public function install_database() {
        global $wpdb;

        // Get collation
        $collate = '';
        if ($wpdb->has_cap('collation')) {
            $collate = $wpdb->get_charset_collate();
        }

        // Required for dbDelta function
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

        // Create wallet credits table
        $sql_wallet_credits = "CREATE TABLE IF NOT EXISTS {$wpdb->base_prefix}pi_aww_wallet_credits (
            credit_id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            blog_id BIGINT UNSIGNED NOT NULL DEFAULT 1,
            user_id BIGINT UNSIGNED NOT NULL,
            amount DECIMAL(16,8) NOT NULL,
            used_amount DECIMAL(16,8) NOT NULL DEFAULT 0,
            currency VARCHAR(10) NOT NULL DEFAULT 'USD',
            expiry_date DATE DEFAULT NULL,
            source VARCHAR(100) DEFAULT NULL,
            note TEXT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            is_deleted TINYINT(1) NOT NULL DEFAULT 0,
            PRIMARY KEY (credit_id),
            KEY blog_user (blog_id, user_id),
            KEY expiry_date (expiry_date),
            KEY currency (currency),
            KEY is_deleted (is_deleted)
        ) $collate;";
        dbDelta($sql_wallet_credits);
        
        // Create wallet debits table
        $sql_wallet_debits = "CREATE TABLE IF NOT EXISTS {$wpdb->base_prefix}pi_aww_wallet_debits (
            debit_id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            blog_id BIGINT UNSIGNED NOT NULL DEFAULT 1,
            user_id BIGINT UNSIGNED NOT NULL,
            amount DECIMAL(16,8) NOT NULL,
            currency VARCHAR(10) NOT NULL DEFAULT 'USD',
            reference_id BIGINT DEFAULT NULL,
            reference_type VARCHAR(50) DEFAULT NULL,
            note TEXT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            is_deleted TINYINT(1) NOT NULL DEFAULT 0,
            PRIMARY KEY (debit_id),
            KEY blog_user (blog_id, user_id),
            KEY currency (currency),
            KEY is_deleted (is_deleted)
        ) $collate;";
        dbDelta($sql_wallet_debits);
        
        // Create wallet credit-debit mapping table
        $sql_credit_debit_map = "CREATE TABLE IF NOT EXISTS {$wpdb->base_prefix}pi_aww_wallet_credit_debit_map (
            map_id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            blog_id BIGINT UNSIGNED NOT NULL DEFAULT 1,
            credit_id BIGINT UNSIGNED NOT NULL,
            debit_id BIGINT UNSIGNED NOT NULL,
            amount_used DECIMAL(16,8) NOT NULL,
            PRIMARY KEY (map_id),
            KEY credit_id (credit_id),
            KEY debit_id (debit_id),
            KEY blog_id (blog_id)
        ) $collate;";
        dbDelta($sql_credit_debit_map);
        
        // Log database installation/update
        if (function_exists('pisol_aww_error_log')) {
            pisol_aww_error_log('Database tables created or updated to version: ' . $this->db_version, 'database');
        }
        
        // Update database version in options
        update_option($this->db_version_option, $this->db_version);
    }
}