<?php
namespace PISOL\AWW\ADMIN;

use PISOL\AWW\CLASSES\Wallet_Manager;

if ( ! defined( 'ABSPATH' ) ) exit;

class View_Transactions{

    public $plugin_name = PISOL_AWW_FOLDER_NAME;

    private $settings = array();

    private $active_tab;

    public $tab_name = 'Transactions';

    private $this_tab = 'pisol_aww_transactions';

    private $setting_key = 'pisol_aww_transactions';
    
    protected static $instance = null;

    public static function get_instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

    function __construct(){
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- No sensitive action; tab switching only.        
        $this->active_tab = (isset($_GET['tab'])) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'default';
        $this->setting_key = 'pisol_aww_transactions';

        if($this->this_tab == $this->active_tab){
            add_action('pisol_'.$this->plugin_name.'_tab_content', array($this,'tab_content'));
        }
        // Add AJAX handler for user search
        add_action('wp_ajax_pisol_aww_search_users', array($this, 'ajax_search_users'));
        // Add AJAX handler for soft delete transaction
        add_action('wp_ajax_pisol_aww_soft_delete_transaction', array($this, 'ajax_soft_delete_transaction'));

        add_action('pisol_'.$this->plugin_name.'_tab', array($this,'tab'), 10);
                
    }
    
    /**
     * Add transactions tab to plugin settings page
     *
     * @param array $tabs Existing tabs
     * @return array Modified tabs
     */
    function tab(){
        $this->tab_name = __("Transactions",'advanced-wallet-for-woocommerce');
        ?>
        <a class=" pi-side-menu  <?php echo ($this->active_tab == $this->this_tab ? 'bg-primary' : 'bg-secondary'); ?>" href="<?php echo esc_url( admin_url( 'admin.php?page='.PISOL_AWW_FOLDER_NAME.'&tab='.$this->this_tab ) ); ?>">
        <img class="mr-1 pisol-icon" src="<?php echo esc_url( plugin_dir_url( __FILE__ ) ); ?>img/transaction-icon.svg">  <?php  echo esc_html( $this->tab_name ); ?> 
        </a>
        <?php
    }

    /**
     * Display tab content with wallet transactions
     */
    function tab_content(){
        // Get wallet manager instance
        $wallet_manager = Wallet_Manager::get_instance();
        
        // Check if user ID is provided
        $user_id = 0;
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just retrieving view parameter
        if(isset($_GET['user_id']) && !empty($_GET['user_id'])) {
            $user_id = absint($_GET['user_id']);
        }
        // Nonce verification for user profile transaction view link
        if ($user_id > 0 && isset($_GET['transactions_nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_GET['transactions_nonce']));
            if (!wp_verify_nonce($nonce, 'pisol_aww_transactions_view_nonce_' . $user_id)) {
                wp_die(esc_html__('Invalid or expired nonce. Please access the transaction history from the user profile page.', 'advanced-wallet-for-woocommerce'));
            }
        }
        
        // Get users for filter
        $users_with_transactions = $this->get_users_with_transactions();
        
        // Display the filter form
        $this->display_filter_form($user_id, $users_with_transactions);
        
        // Get transactions
        $transactions = $this->get_transactions($wallet_manager, $user_id);
        
        // Display transactions
        $this->display_transactions($transactions, $user_id, $wallet_manager);
    }
    
    /**
     * Get list of users with wallet transactions
     *
     * @return array List of users with transactions
     */
    private function get_users_with_transactions() {
        // Fetch all users using WordPress functions instead of direct DB queries
        $args = array(
            'orderby' => 'display_name',
            'order'   => 'ASC',
            'fields'  => array('ID', 'user_login', 'display_name'),
        );
        $user_objects = get_users($args);
        $users = array();
        foreach ($user_objects as $user) {
            $users[] = array(
                'ID'           => $user->ID,
                'user_login'   => $user->user_login,
                'display_name' => $user->display_name,
            );
        }
        return $users;
    }
    
    /**
     * Display user filter form
     *
     * @param int $selected_user_id Currently selected user ID
     * @param array $users List of users to display
     */
    private function display_filter_form($selected_user_id, $users) {
        // Prepare selected user label
        $selected_user_name = '';
        if ($selected_user_id) {
            $user_data = get_userdata($selected_user_id);
            if ($user_data) {
                $selected_user_name = $user_data->display_name . ' (' . $user_data->user_email . ')';
            }
        }
        
        $current_url = admin_url('admin.php?page=advanced-wallet-for-woocommerce&tab=pisol_aww_transactions');
        
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Wallet Transactions', 'advanced-wallet-for-woocommerce'); ?></h1>
            
            <?php 
            // Add nonce field for transaction deletion
            wp_nonce_field('pisol_aww_soft_delete_transaction', 'pisol_aww_soft_delete_nonce'); 
            ?>
            
            <form method="get" action="">
                <input type="hidden" name="page" value="advanced-wallet-for-woocommerce">
                <input type="hidden" name="tab" value="pisol_aww_transactions">
                <?php wp_nonce_field('pisol_aww_search_users', 'pisol_aww_search_users_nonce'); ?>
                <div class="tablenav top">
                    <div class="alignleft actions">
                        <label for="filter-user-select" class="screen-reader-text"><?php esc_html_e('Search by email', 'advanced-wallet-for-woocommerce'); ?></label>
                        <select id="filter-user-select" name="user_id" class="regular-text" style="width:300px;">
                            <?php if($selected_user_id): ?>
                                <option value="<?php echo esc_attr($selected_user_id); ?>" selected><?php echo esc_html($selected_user_name); ?></option>
                            <?php endif; ?>
                        </select>
                        <input type="submit" class="button" value="<?php esc_attr_e('Filter', 'advanced-wallet-for-woocommerce'); ?>">
                        <?php if($selected_user_id): ?>
                            <a href="<?php echo esc_url($current_url); ?>" class="button action"><?php esc_html_e('Reset', 'advanced-wallet-for-woocommerce'); ?></a>
                        <?php endif; ?>
                    </div>
                </div>
            </form>
            <!-- Autocomplete initialized via admin/js/script.js -->
        <?php
        
        // Display user balance if a user is selected
        if($selected_user_id) {
            $user_data = get_userdata($selected_user_id);
            if($user_data) {
                $wallet_manager = Wallet_Manager::get_instance();
                $wallet_balance = $wallet_manager->get_user_balance($selected_user_id);
                echo '<div class="notice notice-info inline"><p>';
                printf(
                    /* translators: 1: user display name, 2: wallet balance */
                    esc_html__('Showing transactions for %1$s. Current balance: %2$s', 'advanced-wallet-for-woocommerce'),
                    '<strong>' . esc_html($user_data->display_name) . '</strong>',
                    '<strong>' . wp_kses_post( wc_price( $wallet_balance ) ) . '</strong>'
                );
                
                // Add link to edit user
                echo ' <a href="' . esc_url(admin_url('user-edit.php?user_id=' . $selected_user_id)) . '" class="button button-small">';
                esc_html_e('Edit User', 'advanced-wallet-for-woocommerce');
                echo '</a>';
                
                echo '</p></div>';
            }
        }
        ?>
        <?php
    }
    
    /**
     * Get wallet transactions based on filters
     *
     * @param Wallet_Manager $wallet_manager Wallet manager instance
     * @param int $user_id Optional user ID filter
     * @return array Transactions
     */
    private function get_transactions($wallet_manager, $user_id = 0) {
        // Get current page number
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- No sensitive action; page switching only.
        $current_page = isset($_GET['paged']) ? absint($_GET['paged']) : 1;
        if ($current_page < 1) $current_page = 1;
        
        // Items per page
        $per_page = 10;
        
        // Get paginated transactions directly using the combined function
        return Wallet_Manager::get_paginated_transactions_combined($user_id, $current_page, $per_page);
    }
    
    /**
     * Get all transactions across all users (legacy method - replaced by pagination)
     * 
     * @param Wallet_Manager $wallet_manager Wallet manager instance
     * @param array $args Query arguments
     * @return array All transactions
     */
    private function get_all_users_transactions($wallet_manager, $args) {
        global $wpdb;
        $tables = [
            'credits' => $wpdb->base_prefix . 'pi_aww_wallet_credits',
            'debits' => $wpdb->base_prefix . 'pi_aww_wallet_debits'
        ];
        
        // Properly escape table names
        $credits_table = esc_sql($tables['credits']);
        $debits_table = esc_sql($tables['debits']);
        
        // Get all user IDs with wallet activity using properly escaped table names
        // Table names can't be parameters in prepared statements
        // We have already escaped the table names earlier with esc_sql()
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table names can't be parameterized
        $user_ids = $wpdb->get_col(
            "SELECT DISTINCT user_id FROM (
                SELECT user_id FROM {$credits_table}
                UNION 
                SELECT user_id FROM {$debits_table}
            ) as users"
        );
        
        $all_transactions = [];
        
        // Loop through each user and get their transactions
        foreach ($user_ids as $user_id) {
            // Using the existing Wallet_Manager method to get user transactions
            $user_transactions = $wallet_manager->get_user_transactions($user_id, null, 1, $args);
            $all_transactions = array_merge($all_transactions, $user_transactions ?: []);
        }
        
        // Sort transactions by date
        usort($all_transactions, function($a, $b) use ($args) {
            $order = ($args['order'] === 'DESC') ? -1 : 1;
            return $order * (strtotime($a['created_at']) - strtotime($b['created_at']));
        });
        
        // Limit after merging
        return array_slice($all_transactions, 0, $args['limit']);
    }
    
    /**
     * Display transactions in a table
     *
     * @param array $transactions List of transactions
     * @param int $user_id Current user ID filter
     * @param Wallet_Manager $wallet_manager Wallet manager instance
     */
    private function display_transactions($transactions_data, $user_id, $wallet_manager) {
        // Extract data from the transaction results
        $transactions = isset($transactions_data['transactions']) ? $transactions_data['transactions'] : [];
        $total_count = isset($transactions_data['total_count']) ? $transactions_data['total_count'] : 0;
        $total_pages = isset($transactions_data['total_pages']) ? $transactions_data['total_pages'] : 1;
        $current_page = isset($transactions_data['current_page']) ? $transactions_data['current_page'] : 1;
        $per_page = isset($transactions_data['per_page']) ? $transactions_data['per_page'] : 10;
        
        if(empty($transactions)) {
            echo '<div class="notice notice-warning inline"><p>';
            esc_html_e('No transactions found.', 'advanced-wallet-for-woocommerce');
            echo '</p></div>';
            echo '</div>'; // Close wrap div
            return;
        }
        
        // Build the pagination base URL
        $base_url = admin_url('admin.php?page=advanced-wallet-for-woocommerce&tab=pisol_aww_transactions');
        if ($user_id > 0) {
            $base_url .= '&user_id=' . $user_id;
        }
        
        ?>
        <table class="wp-list-table widefat fixed striped posts">
            <thead>
                <tr>
                    <th scope="col"><?php esc_html_e('ID', 'advanced-wallet-for-woocommerce'); ?></th>
                    <?php if(!$user_id): ?>
                        <th scope="col"><?php esc_html_e('User', 'advanced-wallet-for-woocommerce'); ?></th>
                    <?php endif; ?>
                    <th scope="col"><?php esc_html_e('Date', 'advanced-wallet-for-woocommerce'); ?></th>
                    <th scope="col"><?php esc_html_e('Type', 'advanced-wallet-for-woocommerce'); ?></th>
                    <th scope="col"><?php esc_html_e('Amount', 'advanced-wallet-for-woocommerce'); ?></th>
                    <th scope="col"><?php esc_html_e('Source', 'advanced-wallet-for-woocommerce'); ?></th>
                    <th scope="col"><?php esc_html_e('Description', 'advanced-wallet-for-woocommerce'); ?></th>
                    <th scope="col"><?php esc_html_e('Expiry', 'advanced-wallet-for-woocommerce'); ?></th>
                    <th scope="col"><?php esc_html_e('Status', 'advanced-wallet-for-woocommerce'); ?></th>
                    <th scope="col"><?php esc_html_e('Actions', 'advanced-wallet-for-woocommerce'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach($transactions as $transaction): ?>
                    <?php 
                    // Format the date
                    $date = date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($transaction['created_at']));
                    
                    // Determine CSS class based on transaction type
                    $type_class = ($transaction['type'] === 'credit') ? 'credit' : 'debit';
                    
                    // Format the type label
                    $type_label = ($transaction['type'] === 'credit') 
                        ? esc_html__('Credit', 'advanced-wallet-for-woocommerce') 
                        : esc_html__('Debit', 'advanced-wallet-for-woocommerce');
                    
                    // Format the amount
                    if($transaction['type'] === 'credit') {
                        $amount = wc_price($transaction['amount'] - $transaction['used_amount']);
                        if($transaction['used_amount'] > 0) {
                            $amount .= ' <small>(' . wc_price($transaction['amount']) . ' ' . esc_html__('total', 'advanced-wallet-for-woocommerce') . ')</small>';
                        }
                    } else {
                        $amount = wc_price($transaction['amount']);
                    }
                    
                    // Format the description
                    $description = $transaction['note'] ?: '&mdash;';
                    
                    // Format the source
                    $source = $transaction['source'] ?: '&mdash;';
                    
                    // Reference links for orders
                    if($transaction['type'] === 'debit' && $transaction['source'] === 'order' && !empty($transaction['note'])) {
                        if(preg_match('/\#(\d+)/', $transaction['note'], $matches)) {
                            $order_id = $matches[1];
                            $source = '<a href="' . esc_url(admin_url('post.php?post=' . $order_id . '&action=edit')) . '">';
                            $source .= esc_html__('Order', 'advanced-wallet-for-woocommerce') . ' #' . esc_html($order_id);
                            $source .= '</a>';
                        }
                    }
                    
                    // Special format for cashback credits
                    if($transaction['type'] === 'credit' && $transaction['source'] === 'cashback') {
                        $source = esc_html__('Cashback', 'advanced-wallet-for-woocommerce');
                    }
                    
                    // Get and format expiry date (only for credits)
                    $expiry_html = '&mdash;';
                    if($transaction['type'] === 'credit' && !empty($transaction['expiry_date'])) {
                        $today = current_time('Y-m-d');
                        $expiry_date = $transaction['expiry_date'];
                        
                        // Format the date for display
                        $formatted_expiry = date_i18n(get_option('date_format'), strtotime($expiry_date));
                        
                        // Show different formatting based on expiry status
                        if($expiry_date < $today) {
                            // Expired
                            $expiry_html = '<span style="color: #e74c3c;">' . $formatted_expiry . ' (' . esc_html__('Expired', 'advanced-wallet-for-woocommerce') . ')</span>';
                        } elseif($expiry_date === $today) {
                            // Expires today
                            $expiry_html = '<span style="color: #e67e22;">' . $formatted_expiry . ' (' . esc_html__('Today', 'advanced-wallet-for-woocommerce') . ')</span>';
                        } else {
                            // Future expiry
                            $days_left = (int)((strtotime($expiry_date) - strtotime($today)) / DAY_IN_SECONDS);
                            if($days_left <= 7) {
                                // Highlight if expiring within 7 days
                                $expiry_html = '<span style="color: #f39c12;">' . $formatted_expiry . ' (' . sprintf(
                                    /* translators: %d: number of days */
                                    _n('%d day left', '%d days left', $days_left, 'advanced-wallet-for-woocommerce'), 
                                    $days_left) . ')</span>';
                            } else {
                                $expiry_html = $formatted_expiry;
                            }
                        }
                    }
                    
                    // Status
                    $status_html = ($transaction['is_deleted'] == 1) 
                        ? '<span style="color: #e74c3c;">' . esc_html__('Deleted', 'advanced-wallet-for-woocommerce') . '</span>' 
                        : '<span style="color: #27ae60;">' . esc_html__('Active', 'advanced-wallet-for-woocommerce') . '</span>';
                        
                    // Get user info if showing all users
                    $user_html = '';
                    if(!$user_id && isset($transaction['user_id'])) {
                        $user_data = get_userdata($transaction['user_id']);
                        if($user_data) {
                            $user_html = '<a href="' . esc_url(admin_url('admin.php?page=advanced-wallet-for-woocommerce&tab=pisol_aww_transactions&user_id=' . $transaction['user_id'])) . '">';
                            $user_html .= esc_html($user_data->display_name);
                            $user_html .= '</a>';
                        } else {
                            $user_html = esc_html__('Unknown User', 'advanced-wallet-for-woocommerce') . ' (ID: ' . esc_html($transaction['user_id']) . ')';
                        }
                    }
                    ?>
                    <tr>
                        <td><?php echo esc_html($transaction['id']); ?> 
                            <span class="row-actions">
                                <span class="id"><?php echo esc_html($transaction['type']); ?>-<?php echo esc_html($transaction['id']); ?></span>
                            </span>
                        </td>
                        <?php if(!$user_id): ?>
                            <td><?php echo wp_kses_post($user_html); ?></td>
                        <?php endif; ?>
                        <td><?php echo esc_html($date); ?></td>
                        <td>
                            <?php if($transaction['type'] === 'credit'): ?>
                                <span style="color: #27ae60;"><?php echo esc_html($type_label); ?></span>
                            <?php else: ?>
                                <span style="color: #e74c3c;"><?php echo esc_html($type_label); ?></span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if($transaction['type'] === 'credit'): ?>
                                <span style="color: #27ae60;"><?php echo wp_kses_post($amount); ?></span>
                            <?php else: ?>
                                <span style="color: #e74c3c;"><?php echo wp_kses_post($amount); ?></span>
                            <?php endif; ?>
                        </td>
                        <td><?php echo wp_kses_post($source); ?></td>
                        <td><?php echo wp_kses_post($description); ?></td>
                        <td><?php echo wp_kses_post($expiry_html); ?></td>
                        <td><?php echo wp_kses_post($status_html); ?></td>
                        <td>
                            <?php if ($transaction['is_deleted'] == 1): ?>
                                <button class="button undelete-transaction" data-transaction-id="<?php echo esc_attr($transaction['id']); ?>" data-transaction-type="<?php echo esc_attr($transaction['type']); ?>">
                                    <?php esc_html_e('Undelete', 'advanced-wallet-for-woocommerce'); ?>
                                </button>
                            <?php elseif ($transaction['type'] === 'credit'): ?>
                                <button class="button delete-transaction" data-transaction-id="<?php echo esc_attr($transaction['id']); ?>" data-transaction-type="<?php echo esc_attr($transaction['type']); ?>">
                                    <?php esc_html_e('Delete', 'advanced-wallet-for-woocommerce'); ?>
                                </button>
                            <?php else: ?>
                                <?php esc_html_e('No action available', 'advanced-wallet-for-woocommerce'); ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <?php if ($total_pages > 1): ?>
            <div class="tablenav bottom">
                <div class="tablenav-pages">
                    <span class="displaying-num">
                        <?php 
                        
                        printf(
                            // translators: %d: number of items
                            esc_html( _n('%d item', '%d items', $total_count, 'advanced-wallet-for-woocommerce') ),
                            esc_html( $total_count )
                        ); 
                        ?>
                    </span>
                    
                    <nav class="pagination-container">
                        <ul class="pagination">
                            <?php
                            // Previous page
                            if ($current_page > 1) {
                                echo '<li class="page-item"><a class="page-link" href="' . esc_url($base_url . '&paged=' . ($current_page - 1)) . '">&laquo; ';
                                esc_html_e('Previous', 'advanced-wallet-for-woocommerce');
                                echo '</a></li>';
                            } else {
                                echo '<li class="page-item disabled"><span class="page-link">&laquo; ';
                                esc_html_e('Previous', 'advanced-wallet-for-woocommerce');
                                echo '</span></li>';
                            }
                            
                            // Page numbers
                            $start_page = max(1, $current_page - 2);
                            $end_page = min($total_pages, $current_page + 2);
                            
                            // First page
                            if ($start_page > 1) {
                                echo '<li class="page-item"><a class="page-link" href="' . esc_url($base_url . '&paged=1') . '">1</a></li>';
                                if ($start_page > 2) {
                                    echo '<li class="page-item disabled"><span class="page-link">&hellip;</span></li>';
                                }
                            }
                            
                            // Page links
                            for ($i = $start_page; $i <= $end_page; $i++) {
                                if ($i == $current_page) {
                                    echo '<li class="page-item active"><span class="page-link">' . esc_html( $i ) . '</span></li>';
                                } else {
                                    echo '<li class="page-item"><a class="page-link" href="' . esc_url( $base_url . '&paged=' . $i ) . '">' . esc_html( $i ) . '</a></li>';
                                }
                            }
                            
                            // Last page
                            if ($end_page < $total_pages) {
                                if ($end_page < $total_pages - 1) {
                                    echo '<li class="page-item disabled"><span class="page-link">&hellip;</span></li>';
                                }
                                echo '<li class="page-item"><a class="page-link" href="' . esc_url( $base_url . '&paged=' . $total_pages ) . '">' . esc_html( $total_pages ) . '</a></li>';
                            }
                            
                            // Next page
                            if ($current_page < $total_pages) {
                                echo '<li class="page-item"><a class="page-link" href="' . esc_url($base_url . '&paged=' . ($current_page + 1)) . '">';
                                esc_html_e('Next', 'advanced-wallet-for-woocommerce');
                                echo ' &raquo;</a></li>';
                            } else {
                                echo '<li class="page-item disabled"><span class="page-link">';
                                esc_html_e('Next', 'advanced-wallet-for-woocommerce');
                                echo ' &raquo;</span></li>';
                            }
                            ?>
                        </ul>
                    </nav>
                </div>
            </div>
        <?php endif; ?>

        </div> <!-- Close wrap div -->
        <?php
    }

    /**
     * Search users by term using direct SQL query for better performance
     * Handles AJAX request for user search in admin
     */
    public function ajax_search_users() {
        // Verify nonce for security
        check_admin_referer('pisol_aww_search_users', 'pisol_aww_search_users_nonce');
        
        // Get and sanitize search term
        $term = isset($_GET['term']) ? sanitize_text_field(wp_unslash($_GET['term'])) : '';
        
        if (empty($term) || strlen($term) < 3) {
            wp_send_json(array());
            return;
        }
        
        global $wpdb;
        
        // Get current blog ID for multisite support
        $blog_id = get_current_blog_id();
        
        // Prepare the LIKE term for SQL query
        $like_term = '%' . $wpdb->esc_like($term) . '%';
        
        // For multisite: join with usermeta to ensure we only get users from current blog
        $join_statement = '';
        $where_statement = '';
        
        if (is_multisite()) {
            $cap_meta_key = $wpdb->prefix . 'capabilities';
            $join_statement = $wpdb->prepare(
                " INNER JOIN {$wpdb->usermeta} AS um 
                  ON u.ID = um.user_id 
                  AND um.meta_key = %s
                  AND um.meta_value != ''", // Ensure user has some role
                $cap_meta_key
            );
        }
        
        // SQL query to find users by email, login or display name
        // Run the query
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $join_statement is already prepared
        $users = $wpdb->get_results($wpdb->prepare(
            "SELECT DISTINCT u.ID, u.user_email, u.display_name
            FROM {$wpdb->users} AS u
            {$join_statement}
            WHERE (
                u.user_email LIKE %s
                OR u.user_login LIKE %s
                OR u.display_name LIKE %s
            )
            ORDER BY 
                CASE 
                    WHEN u.user_email LIKE %s THEN 1
                    WHEN u.user_login LIKE %s THEN 2 
                    ELSE 3
                END,
                u.display_name ASC
            LIMIT 20",
            $like_term,
            $like_term,
            $like_term,
            $like_term,
            $like_term
        ));
        
        // Format the results
        $results = array();
        foreach ($users as $user) {
            $results[] = array(
                'id'   => $user->ID,
                'text' => sprintf('%s (%s)', $user->display_name, $user->user_email),
            );
        }
        
        wp_send_json($results);
    }

    /**
     * Handle AJAX request to soft delete or restore a transaction
     */
    public function ajax_soft_delete_transaction() {
        // Verify nonce for security
        check_ajax_referer('pisol_aww_soft_delete_transaction', 'security');
        
        // Check user capability
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array(
                'message' => __('You do not have permission to perform this action.', 'advanced-wallet-for-woocommerce')
            ));
            return;
        }
        
        // Get transaction details and properly unslash before sanitizing
        $transaction_id = isset($_POST['transaction_id']) ? absint($_POST['transaction_id']) : 0;
        $transaction_type = isset($_POST['transaction_type']) ? sanitize_text_field(wp_unslash($_POST['transaction_type'])) : '';
        $action = isset($_POST['action_type']) ? sanitize_text_field(wp_unslash($_POST['action_type'])) : 'delete'; // 'delete' or 'undelete'
        
        if (!$transaction_id || empty($transaction_type)) {
            wp_send_json_error(array(
                'message' => __('Invalid transaction details.', 'advanced-wallet-for-woocommerce')
            ));
            return;
        }
        
        // Get wallet manager
        $wallet_manager = \PISOL\AWW\CLASSES\Wallet_Manager::get_instance();
        
        $success = false;
        
        if ($action === 'delete') {
            // Soft delete based on type
            if ($transaction_type === 'credit') {
                $success = $wallet_manager->soft_delete_credit($transaction_id);
            } elseif ($transaction_type === 'debit') {
                $success = $wallet_manager->soft_delete_debit($transaction_id);
            }
            
            $message = $success 
                ? __('Transaction has been soft deleted successfully.', 'advanced-wallet-for-woocommerce')
                : __('Failed to delete the transaction. It may be in use or already deleted.', 'advanced-wallet-for-woocommerce');
        } else {
            // Restore based on type
            if ($transaction_type === 'credit') {
                $success = $wallet_manager->restore_credit($transaction_id);
            } elseif ($transaction_type === 'debit') {
                $success = $wallet_manager->restore_debit($transaction_id);
            }
            
            $message = $success 
                ? __('Transaction has been restored successfully.', 'advanced-wallet-for-woocommerce')
                : __('Failed to restore the transaction.', 'advanced-wallet-for-woocommerce');
        }
        
        if ($success) {
            wp_send_json_success(array(
                'message' => $message
            ));
        } else {
            wp_send_json_error(array(
                'message' => $message
            ));
        }
    }
}

