<?php
/**
 * User Profile Class
 * Handles displaying wallet balance and adding funds in user profile
 *
 * @package Advanced Wallet for WooCommerce
 * @since 1.0.0
 */

namespace PISOL\AWW\ADMIN;

use PISOL\AWW\CLASSES\Wallet_Manager;
use PISOL\AWW\CLASSES\Email;

defined('ABSPATH') || exit;

class User_Profile {
    /**
     * The single instance of the class
     *
     * @var User_Profile
     */
    private static $instance = null;

    /**
     * Singleton pattern - ensures only one instance is created
     *
     * @return User_Profile
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    public function __construct() {
        // Add fields to user profile
        add_action('show_user_profile', array($this, 'add_wallet_fields'));
        add_action('edit_user_profile', array($this, 'add_wallet_fields'));
        
        // Save wallet fields
        add_action('personal_options_update', array($this, 'save_wallet_fields'));
        add_action('edit_user_profile_update', array($this, 'save_wallet_fields'));

        // Enqueue admin CSS for wallet profile
        add_action('admin_enqueue_scripts', array($this, 'enqueue_wallet_admin_styles'));
    }

    /**
     * Enqueue admin CSS for wallet profile using wp_add_inline_style
     */
    public function enqueue_wallet_admin_styles($hook) {
        // Only load on user profile pages
        if ($hook !== 'user-edit.php' && $hook !== 'profile.php') {
            return;
        }
        $inline_css = '.credit-amount { color: green; }\n.debit-amount { color: red; }\n.transactions-view-all { margin-top: 10px; }';
        // Use core admin style handle
        wp_add_inline_style('wp-admin', $inline_css);
    }

    /**
     * Add wallet balance and fund management fields to user profile
     *
     * @param WP_User $user User object
     */
    public function add_wallet_fields($user) {
        // Check if current user has permission to edit users
        if (!current_user_can('edit_users')) {
            return;
        }

        // Get the wallet manager instance
        $wallet_manager = Wallet_Manager::get_instance();
        
        // Get user's balance
        $balance = $wallet_manager->get_user_balance($user->ID);
        
        // Get currency symbol
        $currency_symbol = get_woocommerce_currency_symbol();
        
        // Create nonce for security
        wp_nonce_field('pisol_aww_wallet_action', 'pisol_aww_wallet_nonce');
        
        ?>
        <h2><?php esc_html_e('Wallet Information', 'advanced-wallet-for-woocommerce'); ?></h2>
        <table class="form-table">
            <tr>
                <th><label for="wallet_balance"><?php esc_html_e('Current Balance', 'advanced-wallet-for-woocommerce'); ?></label></th>
                <td>
                    <strong><?php echo esc_html($currency_symbol . number_format($balance, 2)); ?></strong>
                </td>
            </tr>
            <?php if (current_user_can('manage_woocommerce')) : ?>
            <tr>
                <th><label for="wallet_add_balance"><?php esc_html_e('Add Balance', 'advanced-wallet-for-woocommerce'); ?></label></th>
                <td>
                    <input type="number" step="0.01" min="0" name="wallet_add_balance" id="wallet_add_balance" value="" class="regular-text" />
                    <p class="description"><?php esc_html_e('Enter amount to add to user wallet', 'advanced-wallet-for-woocommerce'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="wallet_expiry_date"><?php esc_html_e('Expiry Date', 'advanced-wallet-for-woocommerce'); ?></label></th>
                <td>
                    <input type="date" name="wallet_expiry_date" id="wallet_expiry_date" value="" class="regular-text" min="<?php echo esc_attr(gmdate('Y-m-d', strtotime('+1 day'))); ?>" />
                    <p class="description"><?php esc_html_e('Optional: Set an expiry date for this balance (leave empty for no expiry)', 'advanced-wallet-for-woocommerce'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="wallet_note"><?php esc_html_e('Note', 'advanced-wallet-for-woocommerce'); ?></label></th>
                <td>
                    <input type="text" name="wallet_note" id="wallet_note" value="" class="regular-text" />
                    <p class="description"><?php esc_html_e('Optional note about this transaction', 'advanced-wallet-for-woocommerce'); ?></p>
                </td>
            </tr>
            <?php endif; ?>
        </table>

        <?php if (current_user_can('manage_woocommerce')) : ?>
        <h3><?php esc_html_e('Recent Transactions', 'advanced-wallet-for-woocommerce'); ?></h3>
        <?php
        // Get recent transactions
        $transactions = $wallet_manager->get_user_transactions($user->ID, null, 1, array('limit' => 5));
        
        if (!empty($transactions)) : ?>
            <table class="widefat striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Date', 'advanced-wallet-for-woocommerce'); ?></th>
                        <th><?php esc_html_e('Type', 'advanced-wallet-for-woocommerce'); ?></th>
                        <th><?php esc_html_e('Amount', 'advanced-wallet-for-woocommerce'); ?></th>
                        <th><?php esc_html_e('Source', 'advanced-wallet-for-woocommerce'); ?></th>
                        <th><?php esc_html_e('Expiry Date', 'advanced-wallet-for-woocommerce'); ?></th>
                        <th><?php esc_html_e('Note', 'advanced-wallet-for-woocommerce'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($transactions as $transaction) : ?>
                    <tr>
                        <td><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($transaction['created_at']))); ?></td>
                        <td><?php echo esc_html(ucfirst($transaction['type'])); ?></td>
                        <td>
                            <?php 
                            $amount_class = $transaction['type'] === 'credit' ? 'credit-amount' : 'debit-amount';
                            $prefix = $transaction['type'] === 'credit' ? '+' : '-';
                            echo '<span class="' . esc_attr($amount_class) . '">' . esc_html($prefix . ' ' . $currency_symbol . number_format($transaction['amount'], 2)) . '</span>';
                            ?>
                        </td>
                        <td><?php echo esc_html($transaction['source'] ?? '—'); ?></td>
                        <td><?php echo esc_html($transaction['expiry_date'] ?? '—'); ?></td>
                        <td><?php echo esc_html($transaction['note'] ?? '—'); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
            <p class="transactions-view-all">
                <?php 
                $transactions_nonce = wp_create_nonce('pisol_aww_transactions_view_nonce_' . $user->ID); 
                ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=advanced-wallet-for-woocommerce&tab=pisol_aww_transactions&user_id=' . $user->ID . '&transactions_nonce=' . $transactions_nonce)); ?>" class="button">
                    <?php esc_html_e('View All Transactions', 'advanced-wallet-for-woocommerce'); ?>
                </a>
            </p>
        <?php else : ?>
            <p><?php esc_html_e('No transactions found.', 'advanced-wallet-for-woocommerce'); ?></p>
            
            <?php if (current_user_can('manage_woocommerce')) : ?>
                <p class="transactions-view-all">
                    <?php 
                    $transactions_nonce = wp_create_nonce('pisol_aww_transactions_view_nonce_' . $user->ID); 
                    ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=advanced-wallet-for-woocommerce&tab=pisol_aww_transactions&user_id=' . $user->ID . '&transactions_nonce=' . $transactions_nonce)); ?>" class="button">
                        <?php esc_html_e('View Transaction History', 'advanced-wallet-for-woocommerce'); ?>
                    </a>
                </p>
            <?php endif; ?>
        <?php endif; ?>
        
        <?php endif; ?>
        <?php
    }

    /**
     * Save wallet fields from user profile
     *
     * @param int $user_id User ID
     */
    public function save_wallet_fields($user_id) {
        // Check nonce for security
        if (!isset($_POST['pisol_aww_wallet_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['pisol_aww_wallet_nonce'])), 'pisol_aww_wallet_action')) {
            return;
        }
        
        // Check permissions
        if (!current_user_can('edit_user', $user_id) || !current_user_can('manage_woocommerce')) {
            return;
        }
        
        // Check if add balance is set
        if (isset($_POST['wallet_add_balance']) && !empty($_POST['wallet_add_balance'])) {
            $amount = floatval(sanitize_text_field(wp_unslash($_POST['wallet_add_balance'])));
            
            if ($amount <= 0) {
                return;
            }
            
            // Get note if available
            $note = '';
            if (isset($_POST['wallet_note']) && !empty($_POST['wallet_note'])) {
                $note = sanitize_text_field(wp_unslash($_POST['wallet_note']));
            } else {
                /* translators: %s: Admin username */
                $note = sprintf(
                    /* translators: %s: Admin username */
                    __('Added by admin: %s', 'advanced-wallet-for-woocommerce'), 
                    wp_get_current_user()->user_login
                );
            }
            
            // Get expiry date if set
            $credit_params = array(
                'user_id' => $user_id,
                'amount' => $amount,
                'source' => 'admin',
                'note' => $note,
            );
            
            // Add expiry date if set
            if (isset($_POST['wallet_expiry_date']) && !empty($_POST['wallet_expiry_date'])) {
                $expiry_date = sanitize_text_field(wp_unslash($_POST['wallet_expiry_date']));
                // Validate date format (YYYY-MM-DD)
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $expiry_date)) {
                    // Make sure expiry date is in the future
                    if (strtotime($expiry_date) > time()) {
                        $credit_params['expiry_date'] = $expiry_date;
                        
                        // Append expiry information to note
                        $formatted_date = date_i18n(get_option('date_format'), strtotime($expiry_date));
                        /* translators: %s: Formatted expiry date */
                        $credit_params['note'] .= ' | ' . sprintf(__('Expires on %s', 'advanced-wallet-for-woocommerce'), $formatted_date);
                    }
                }
            }
            
            // Get wallet manager instance
            $wallet_manager = Wallet_Manager::get_instance();
            
            // Add credit to user wallet
            $result = $wallet_manager->add_credit($credit_params);
            
            // Add admin notice for success or failure
            if ($result) {

                $currency_symbol = get_woocommerce_currency_symbol();
                Email::send_email(
                    get_userdata($user_id)->user_email,
                    __('Wallet Funds Added', 'advanced-wallet-for-woocommerce'),
                    sprintf(
                        __('Hello %s, your wallet has been credited with %s.', 'advanced-wallet-for-woocommerce'),
                        get_userdata($user_id)->display_name,
                        $currency_symbol . number_format($amount, 2)
                    ),
                    __('Wallet Funds Added', 'advanced-wallet-for-woocommerce')
                );

                add_action('admin_notices', function() use ($amount, $credit_params, $currency_symbol) {
                    /* translators: %s: Amount with currency symbol */
                    $message = sprintf(__('Wallet funds added successfully: %s', 'advanced-wallet-for-woocommerce'), $currency_symbol . number_format($amount, 2));
                    
                    // Add expiry message if applicable
                    if (!empty($credit_params['expiry_date'])) {
                        $formatted_date = date_i18n(get_option('date_format'), strtotime($credit_params['expiry_date']));
                        /* translators: %s: Formatted expiry date */
                        $message .= ' ' . sprintf(__('(Expires on: %s)', 'advanced-wallet-for-woocommerce'), $formatted_date);
                    }
                    ?>
                    <div class="notice notice-success is-dismissible">
                        <p><?php echo esc_html($message); ?></p>
                    </div>
                    <?php
                });
            } else {
                add_action('admin_notices', function() {
                    ?>
                    <div class="notice notice-error is-dismissible">
                        <p><?php esc_html_e('Failed to add wallet funds. Please try again.', 'advanced-wallet-for-woocommerce'); ?></p>
                    </div>
                    <?php
                });
            }
        }
    }
}