<?php

namespace FBTWOO;

defined( 'ABSPATH' ) || exit;



if ( ! class_exists( 'ProductTab' ) ) {
	class ProductTab {

		public function __construct() {
			add_filter( 'woocommerce_product_data_tabs', array( $this, 'add_tab' ) );
			add_action( 'woocommerce_product_data_panels', array( $this, 'tab_content' ) );

			add_action( 'woocommerce_admin_process_product_object', array( $this, 'save_options' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_script' ) );

			add_action( 'wp_ajax_fbtwoo_custom_product_search', array( $this, 'custom_product_search' ) );
			add_action( 'wp_ajax_nopriv_fbtwoo_custom_product_search', array( $this, 'custom_product_search' ) );
		}

		public function add_tab( $tabs ) {

			$tabs['frequently_bought_together'] = array(
				'label'    => __( 'Frequently Bought Together', 'frequently-bought-together-woo' ),
				'target'   => 'fbtwoo_product_options',
				'class'    => array( 'hide_if_grouped', 'hide_if_external', 'hide_if_bundle' ),
				'priority' => 70,
			);

			return $tabs;
		}

		public function tab_content() {
			global $product_object, $post;
			$product_id      = $post->ID;
			$currency_symbol = get_woocommerce_currency_symbol(); // Get WooCommerce currency symbol

			// Get saved discount type and value
			$apply_discount = intval( $product_object->get_meta( '_fbtwoo_apply_discount' ) );

			$discount_type  = $product_object->get_meta( '_fbtwoo_discount_type', true ) ?: 'percentage';
			$discount_value = $product_object->get_meta( '_fbtwoo_discount_value', true ) ?: '';

			?>
			<div id="fbtwoo_product_options" class="panel woocommerce_options_panel">
				<div class="options_group">
					<p class="form-field frequently_bought_together_field">
							<label for="fbtwoo_product_ids"><?php esc_html_e( 'Add Products', 'frequently-bought-together-woo' ); ?></label>
							<select 
								class="wc-product-search" 
								multiple="multiple" 
								style="width: 90%;" 
								id="fbtwoo_product_ids" 
								name="fbtwoo_product_ids[]" 
								data-placeholder="<?php esc_attr_e( 'Search for a product&hellip;', 'frequently-bought-together-woo' ); ?>" 
								data-action="fbtwoo_custom_product_search" 
								data-exclude="<?php echo intval( $post->ID ); ?>" 
							>
								<?php
								$product_ids = $product_object->get_meta( '_fbtwoo_product_ids' );
								if ( ! is_array( $product_ids ) ) {
									$product_ids = array();
								}

								foreach ( $product_ids as $product_id ) {
									$product = wc_get_product( $product_id );
									if ( is_object( $product ) ) {
										echo '<option value="' . esc_attr( $product_id ) . '"' . selected( true, true, false ) . '>' . wp_kses_post( $product->get_formatted_name() ) . '</option>';
									}
								}
								?>
							</select>	
							<?php echo wp_kses_post( wc_help_tip( 'Add Products to show in Frequently Bought Together Box.', false ) ); ?>				
					</p>

					<!-- Apply Discount Checkbox -->
					<p class="form-field frequently_bought_together_field">
						<label for="fbtwoo_apply_discount"><?php esc_html_e( 'Apply Discount', 'frequently-bought-together-woo' ); ?></label>
						<input type="checkbox" id="fbtwoo_apply_discount" name="fbtwoo_apply_discount" value="1" <?php checked( $apply_discount, 1 ); ?>>	
					</p>

					<!-- Discount Type Select -->
					<p class="form-field fbtwoo_discount_type_field">
						<label for="fbtwoo_discount_type"><?php esc_html_e( 'Discount Type', 'frequently-bought-together-woo' ); ?></label>
						<select id="fbtwoo_discount_type" name="fbtwoo_discount_type">
							<option value="percentage" <?php selected( $discount_type, 'percentage' ); ?>><?php esc_html_e( 'Percentage (%)', 'frequently-bought-together-woo' ); ?></option>
							<?php // translators: %s represents the currency symbol ?>
							<option value="fixed" <?php selected( $discount_type, 'fixed' ); ?>><?php printf( esc_html__( 'Fixed (%s)', 'frequently-bought-together-woo' ), esc_html( $currency_symbol ) ); ?></option>
						</select>
					</p>

					<!-- Discount Value Input -->
					<p class="form-field fbtwoo_discount_value_field">
						<label for="fbtwoo_discount_value"><?php esc_html_e( 'Discount Value', 'frequently-bought-together-woo' ); ?></label>
						<input type="number" id="fbtwoo_discount_value" name="fbtwoo_discount_value" style="margin-right:5px;" value="<?php echo esc_attr( $discount_value ); ?>" step="0.01" min="0">
						<span id="fbtwoo_discount_label"><?php echo ( $discount_type === 'percentage' ) ? '%' : esc_html( $currency_symbol ); ?></span>
					</p>
					
				</div>
			</div>
			<?php
		}

		/**
		 * Handles custom AJAX product search for the WooCommerce select field.
		 */
		public function custom_product_search() {

			// Verify security nonce
			if ( ! isset( $_GET['security'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['security'])), 'fbtwoo_search_nonce' ) ) {
				wp_send_json_error( array( 'message' => 'Invalid security token' ) );
				wp_die();
			}

			// Get search term
			$term = isset( $_GET['term'] ) ? sanitize_text_field( wp_unslash($_GET['term'] )) : '';

			if ( empty( $term ) ) {
				wp_send_json_error( array( 'message' => 'No search term provided' ) );
				wp_die();
			}

			// Query WooCommerce products, excluding variable parents
			$args = array(
				'post_type'      => array( 'product', 'product_variation' ),
				'posts_per_page' => 10,
				's'              => $term,
				'post_status'    => 'publish',
			);

			$products = get_posts( $args );

			$results = array();
			foreach ( $products as $product ) {
				$wc_product = wc_get_product( $product->ID );
				if ( $wc_product->is_type( 'variation' ) || $wc_product->is_type( 'simple' ) ) {
					$results[] = array(
						'id'   => $wc_product->get_id(),
						'text' => $wc_product->get_name(),
					);
				}
			}

			wp_send_json_success( $results );
			wp_die();
		}

		public function save_options( $product ) {

			if ( ! isset( $_POST['woocommerce_meta_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['woocommerce_meta_nonce']) ), 'woocommerce_save_data' ) ) {
				return;
			}

			// Sanitize and validate product IDs
			$product_ids = isset( $_POST['fbtwoo_product_ids'] ) ? array_map( 'absint', (array) $_POST['fbtwoo_product_ids'] ) : array();

			// Sanitize discount apply checkbox (convert to boolean)
			$apply_discount = isset( $_POST['fbtwoo_apply_discount'] ) ? 1 : 0;

			// Sanitize discount type (allow only 'fixed' or 'percentage')
			$allowed_discount_types = array( 'fixed', 'percentage' );
			$discount_type          = isset( $_POST['fbtwoo_discount_type'] )
			? sanitize_text_field( wp_unslash( $_POST['fbtwoo_discount_type'] ) )
			: 'fixed';

			// Ensure it's an allowed type
			if ( ! in_array( $discount_type, $allowed_discount_types, true ) ) {
				$discount_type = 'fixed';
			}

			// Sanitize discount value (ensure it's a float number)
			$discount_value = isset( $_POST['fbtwoo_discount_value'] ) ? floatval( $_POST['fbtwoo_discount_value'] ) : 0;

			// Store data using WooCommerce's update_meta_data()
			$product->update_meta_data( '_fbtwoo_product_ids', $product_ids );
			$product->update_meta_data( '_fbtwoo_apply_discount', $apply_discount );
			$product->update_meta_data( '_fbtwoo_discount_type', $discount_type );
			$product->update_meta_data( '_fbtwoo_discount_value', $discount_value );
		}

		function enqueue_script( $hook_suffix ) {
			global $post;

			// Check if we are on a WooCommerce product edit page
			if ( $hook_suffix !== 'post.php' && $hook_suffix !== 'post-new.php' ) {
				return;
			}

			// Ensure it's a product post type
			if ( isset( $post ) && get_post_type( $post->ID ) !== 'product' ) {
				return;
			}

			wp_register_script( 'fbtwoo-discount-manager-script', Constants::$plugin_dir_url . 'assets/src/admin/js/discount-manager.js', array( 'jquery' ), Constants::$plugin_version, true );
				wp_localize_script(
					'fbtwoo-discount-manager-script',
					'fbtwoo_discount_manager_scripts',
					array(
						'currencySymbol' => get_woocommerce_currency_symbol(),
					)
				);
			wp_enqueue_script( 'fbtwoo-discount-manager-script' );

			wp_register_script(
				'fbtwoo-search',
				Constants::$plugin_dir_url . 'assets/src/admin/js/ajax-search.js',
				array( 'jquery', 'selectWoo' ),
				'1.0.0',
				true
			);
			wp_localize_script(
				'fbtwoo-search',
				'fbtwoo_ajax',
				array(
					'ajax_url' => admin_url( 'admin-ajax.php' ),
					'security' => wp_create_nonce( 'fbtwoo_search_nonce' ),
				)
			);
			wp_enqueue_script( 'fbtwoo-search' );
		}
	}
}
