<?php

namespace FBTWOO;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use FBTWOO\Constants;
use FBTWOO\ProductTab;
use FBTWOO\Settings;
use FBTWOO\Frontend;
use FBTWOO\Cart;

if ( ! class_exists( 'Init' ) ) {
	class Init {

		public function __construct() {
			$this->define_constants();
			$this->includes();
		}


		public function define_constants() {
			Constants::init(); // When you declare a static property in a class, you can only assign constant values ​​to it, such as numbers or fixed strings. We need to use a separate method to initialize the static property
		}

		public function includes() {

			
			add_action( 'admin_init', array( $this, 'check_dependencies' ) );
			
			add_action( 'plugins_loaded', array( $this, 'on_plugins_loaded' ) );
			
			add_action( 'wp_enqueue_scripts', array( $this, 'maybe_enqueue_scripts' ) );
			
			add_filter( 'plugin_action_links_' . Constants::$plugin_basename, array( $this, 'add_settings_link' ) );
			
			add_action( 'wp_head', array( $this, 'fbtwoo_custom_css' ) );
			
	
		}


		/**
		 * Checks if WooCommerce is installed and activated.
		 *
		 * - Displays an admin notice if WooCommerce is missing.
		 * - Deactivates the plugin if WooCommerce is not active.
		 *
		 * @return void
		 */
		public function check_dependencies(): void {
			if ( ! class_exists( 'WooCommerce' ) ) {
				add_action(
					'admin_notices',
					function () {
						echo '<div class="notice notice-error"><p>' . esc_html__( 'Frequently Bought Together For WooCommerce requires WooCommerce to be installed and activated.', 'frequently-bought-together-woo' ) . '</p></div>';
					}
				);

				deactivate_plugins( Constants::$plugin_basename );
			}
		}

		

		public function on_plugins_loaded() {

			new Settings();
			if ( is_admin() && current_user_can( 'manage_options' ) ) {
				new ProductTab();
			}

			if ( ! is_admin() ) {
				new Frontend();
				new Cart();
			}
		}


		public function maybe_enqueue_scripts() {
			global $product;

			if ( is_product() ) {
				$product = wc_get_product( get_the_id() );

				wp_register_script( 'fbtwoo-script', Constants::$plugin_dir_url . 'assets/src/public/js/script.js', array( 'jquery' ), Constants::$plugin_version, true );
				wp_localize_script(
					'fbtwoo-script',
					'fbtwoo_scripts',
					array(
						'name'                             => $product->get_name(),
						'price'                            => $product->get_price(),
						'product_type'                     => $product->get_type(),
						'currencySymbol'                   => get_woocommerce_currency_symbol(),
						'currency'                         => get_woocommerce_currency(),
						'decimal_separator'                => wc_get_price_decimal_separator(),
						'thousand_separator'               => wc_get_price_thousand_separator(),
						'decimals'                         => wc_get_price_decimals(),
						'price_format'                     => get_woocommerce_price_format(),
						'fbtwoo_label_save'                => esc_html__( 'Save:', 'frequently-bought-together-woo' ),
						'fbtwoo_button_text_no_product'    => get_option( 'fbtwoo_button_text_no_product' ),
						'fbtwoo_button_text_one_product'   => get_option( 'fbtwoo_button_text_one_product' ),
						'fbtwoo_button_text_two_product'   => get_option( 'fbtwoo_button_text_two_product' ),
						'fbtwoo_button_text_three_product' => get_option( 'fbtwoo_button_text_three_product' ),
						'fbtwoo_apply_discount'            => $product->get_meta( '_fbtwoo_apply_discount' ),
						'fbtwoo_discount_type'             => $product->get_meta( '_fbtwoo_discount_type' ), // 'percentage' or 'fixed'
						'fbtwoo_discount_value'            => floatval( $product->get_meta( '_fbtwoo_discount_value' ) ),
					)
				);
				wp_enqueue_script( 'fbtwoo-script' );

				wp_register_style( 'fbtwoo-style', Constants::$plugin_dir_url . 'assets/src/public/css/style.css', array(), Constants::$plugin_version );
				wp_enqueue_style( 'fbtwoo-style' );
			}
		}

		function fbtwoo_custom_css() {
			global $product;
			if ( is_product() ) {
				$button_color            = get_option( 'fbtwoo_button_color' );
				$button_color_hover      = get_option( 'fbtwoo_button_color_hover' );
				$button_text_color       = get_option( 'fbtwoo_button_text_color' );
				$button_text_color_hover = get_option( 'fbtwoo_button_text_color_hover' );
				$save_text_color         = get_option( 'fbtwoo_save_text_color' );

				if ( $button_color || $button_color_hover || $button_text_color || $button_text_color_hover || $save_text_color ) {
					echo '<style type="text/css" id="fbtwoo-custom-css">';
					if ( $button_color ) {
						echo '#fbtwoo-form-container .fbtwoo-button { background-color: ' . esc_attr( sanitize_hex_color( $button_color ) ) . '; }';
					}
					if ( $button_color_hover ) {
						echo '#fbtwoo-form-container .fbtwoo-button:hover { background-color: ' . esc_attr( sanitize_hex_color( $button_color_hover ) ) . '; }';
					}
					if ( $button_text_color ) {
						echo '#fbtwoo-form-container .fbtwoo-button { color: ' . esc_attr( sanitize_hex_color( $button_text_color ) ) . '; }';
					}
					if ( $button_text_color_hover ) {
						echo '#fbtwoo-form-container .fbtwoo-button:hover { color: ' . esc_attr( sanitize_hex_color( $button_text_color_hover ) ) . '; }';
					}
					if ( $save_text_color ) {
						echo '#fbtwoo-form-container .fbtwoo-form__total-price--saved { color: ' . esc_attr( sanitize_hex_color( $save_text_color ) ) . '; }';
					}
					echo '</style>';
				}
			}
		}


		public function add_settings_link( $links ) {
			$url           = get_admin_url() . 'admin.php?page=fbtwoo-settings';
			$settings_link = array( '<a href="' . esc_url( $url ) . '">' . esc_html( __( 'Settings', 'frequently-bought-together-woo' ) ) . '</a>' );
			return array_merge( $settings_link, $links );
		}
	}
}
