<?php

namespace FBTWOO;

defined( 'ABSPATH' ) || exit;

use FBTWOO\Constants;
use FBTWOO\Settings;

if ( ! class_exists( 'Frontend' ) ) {
	class Frontend {

		public function __construct() {
			// The wp action runs after WordPress has determined the query (e.g., whether it's a product page or another type of page).
			add_action( 'wp', array( $this, 'maybe_display' ) );
		}

		public function maybe_display() {

			if ( ! is_product() ) {
				return; // Exit if not a single product page
			}

			$product = wc_get_product( get_the_ID() );

			if ( ! $product ) {
				return;
			}

			// If the shortcode should only work on product pages, keep it in maybe_display(). If you want flexibility to use it anywhere, move it to __construct().
			add_shortcode( 'fbtwoo_form', array( $this, 'fbtwoo_form_shortcode' ) );

			$fbt_products = $this->get_available_frequently_bought_together_products( $product );

			if ( ! empty( $fbt_products ) ) {
				$position            = get_option( 'fbtwoo_position', 'above_product_tabs' );
				$available_positions = Settings::get_position_options();

				// Default priority if not found
				$priority       = 10;
				$position_found = false; // Flag to check if the position exists

				// Check if the selected position exists in the available positions array
				foreach ( $available_positions as $key => $available_position ) {
					if ( $key === $position ) {
						$hook_name      = isset( $available_position['hook_name'] ) ? sanitize_text_field( $available_position['hook_name'] ) : 'woocommerce_after_single_product_summary';
						$priority       = isset( $available_position['priority'] ) ? absint( $available_position['priority'] ) : 10;
						$position_found = true;
						break;
					}
				}

				// Only add action if a valid position was found
				if ( $position_found ) {
					add_action( $hook_name, array( $this, 'add_frequently_bought_together_form' ), $priority );
				}
			}
		}

		public function add_frequently_bought_together_form() {
			echo do_shortcode( '[fbtwoo_form product_id="' . esc_attr( get_the_ID() ) . '"]' );
		}

		public function get_available_frequently_bought_together_products( $product ) {
			$fbt_product_ids = $product->get_meta( '_fbtwoo_product_ids' );

			$fbt_products = array();
			foreach ( $fbt_product_ids as $fbt_product_id ) {
				$current = wc_get_product( $fbt_product_id );
				if ( $current && $current->is_purchasable() && $current->is_in_stock() ) {
					$fbt_products[] = $current;
				}
			}
			return $fbt_products;
		}

		// add shortcode [fbtwoo_form product_id="XXX"]
		public function fbtwoo_form_shortcode( $atts ) {

			if ( defined( 'REST_REQUEST' ) ) {
				return;
			}

			if ( ! is_product() ) {
				return;
			}

			$attributes = shortcode_atts(
				array(
					'product_id' => is_product() ? get_the_ID() : 0,
					'title'      => esc_html( get_option( 'fbtwoo_title' ) ), // __( 'Frequently Bought Together', 'frequently-bought-together-woo' ),
					'class'      => '',
					'style'      => '',
				),
				$atts
			);

			$product_id = $attributes['product_id'];
			$title      = $attributes['title'];
			$class      = $attributes['class'];
			$style      = $attributes['style'];

			$product_id = intval( $product_id );

			if ( ! $product_id ) {
				return;
			}

			$product = wc_get_product( $product_id );

			if ( ! $product ) {
				return;
			}

			if ( $product->is_type( array( 'grouped', 'external' ) ) ) {
				return;
			}

			$fbt_products = $this->get_available_frequently_bought_together_products( $product );

			ob_start();

			wc_get_template(
				'frequently-bought-together-form.php',
				array(
					'product'      => $product,
					'fbt_products' => $fbt_products,
					'title'        => $title,
					'class'        => $class,
					'style'        => $style,
				),
				'',
				Constants::$plugin_dir_path . 'templates/'
			);

			return ob_get_clean();
		}
	}
}
