<?php

namespace FBTWOO;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Cart' ) ) {
	class Cart {

		public function __construct() {
			add_action( 'wp_loaded', array( $this, 'add_frequently_bought_together_products_to_cart' ), 20 );
			add_action( 'woocommerce_cart_calculate_fees', array( $this, 'apply_bundle_discount' ) );
		}

		public function add_frequently_bought_together_products_to_cart() {

			if ( ! ( isset( $_REQUEST['action'] ) && sanitize_text_field( wp_unslash( $_REQUEST['action'] ) ) === 'fbtwoo_add_to_cart' && ( isset( $_REQUEST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) ), 'fbtwoo_add_to_cart' ) ) ) ) {
				return;
			}

			if ( ! isset( $_REQUEST['fbtwoo-parent-product'] ) || empty( $_REQUEST['fbtwoo-parent-product'] ) ) {
				return;
			}

			$parent_product = isset( $_REQUEST['fbtwoo-parent-product'] ) ? absint( wp_unslash( $_REQUEST['fbtwoo-parent-product'] ) ) : '';

			if ( ! isset( $_REQUEST['fbtwoo-products'] ) || empty( $_REQUEST['fbtwoo-products'] ) ) {
				return;
			}

			$products = isset( $_REQUEST['fbtwoo-products'] ) ? array_map( 'absint', wp_unslash( (array) $_REQUEST['fbtwoo-products'] ) ) : array();

			$products_added = true;

			foreach ( $products as $product_id ) {
				$product = wc_get_product( $product_id );

				if ( ! $product ) {
					continue;
				}

				if ( $product->is_type( 'variable' ) ) {
					continue;
				}

				$variation_id   = 0;
				$variation_attr = array();

				if ( $product->is_type( 'variation' ) ) {
					$variation_id   = $product->get_id();
					$variation_attr = $product->get_variation_attributes();
				}

				$custom_data = array(
					'frequently_bought_together' => true,
					'ftbwoo_parent_product'      => $parent_product,
				);

				$added = WC()->cart->add_to_cart( $product_id, 1, $variation_id, $variation_attr, $custom_data );

				if ( ! $added ) {
					$products_added = false;
				}
			}

			if ( $products_added ) {
				wc_add_notice( 'Products added to cart successfully!', 'success' );

			} else {
				wc_add_notice( 'Error adding products to cart.', 'error' );
			}

			$url = remove_query_arg( array( 'action', '_wpnonce' ) );

			wp_safe_redirect( esc_url( $url ) );
			exit;
		}


		public function apply_bundle_discount() {
			$discount        = 0;
			$bundles_in_cart = array();

			foreach ( WC()->cart->get_cart() as $cart_item_key => $cart_item ) {
				if (
					isset( $cart_item['frequently_bought_together'] ) &&
					$cart_item['frequently_bought_together'] == true &&
					isset( $cart_item['ftbwoo_parent_product'] ) &&
					! empty( $cart_item['ftbwoo_parent_product'] )
				) {
					$parent_product_id = absint( $cart_item['ftbwoo_parent_product'] );

					// Check if it's a variation and use variation_id instead
					if ( isset( $cart_item['variation_id'] ) && $cart_item['variation_id'] != 0 ) {
						$bundles_in_cart[ $parent_product_id ][] = $cart_item['variation_id'];
					} else {
						$bundles_in_cart[ $parent_product_id ][] = $cart_item['product_id'];
					}
				}
			}

			// Process each bundle in cart
			foreach ( $bundles_in_cart as $parent_id => $bundle_items ) {
				$parent_product = wc_get_product( $parent_id );

				if ( ! $parent_product ) {
					continue;
				}

				// Get the bundle products from parent product meta
				$fbt_product_ids = $parent_product->get_meta( '_fbtwoo_product_ids' );
				$fbt_product_ids = is_array( $fbt_product_ids ) ? $fbt_product_ids : explode( ',', $fbt_product_ids );

				// Ensure all bundle items are in the cart
				if ( count( array_intersect( $fbt_product_ids, $bundle_items ) ) !== count( $fbt_product_ids ) ) {
					continue; // Skip discount if all bundled products aren't in the cart
				}

				// Get discount settings from the parent product
				$apply_discount = intval( $parent_product->get_meta( '_fbtwoo_apply_discount' ) );
				$discount_type  = $parent_product->get_meta( '_fbtwoo_discount_type' ); // 'percentage' or 'fixed'
				$discount_value = floatval( $parent_product->get_meta( '_fbtwoo_discount_value' ) );

				if ( $apply_discount && $discount_value > 0 ) {
					$bundle_total = 0;

					// Calculate total price of the bundle items
					foreach ( WC()->cart->get_cart() as $cart_item ) {
						if ( in_array( $cart_item['product_id'], $bundle_items ) ) {
							$bundle_total += floatval( $cart_item['line_total'] );
						}
					}

					// Apply discount
					if ( $discount_type === 'percentage' ) {
						$discount_amount = $bundle_total * ( $discount_value / 100 );
					} elseif ( $discount_type === 'fixed' ) {
						$discount_amount = min( $discount_value, $bundle_total ); // Ensure discount doesn't exceed price
					}

					$discount += $discount_amount;
				}
			}

			if ( $discount > 0 ) {
				WC()->cart->add_fee( __( 'Boundle Discount', 'frequently-bought-together-woo' ), -$discount );
			}
		}
	}
}
