<?php
namespace PISOL\ANFW\ADMIN;

/**
 * Product Presence Condition
 *
 * @package Advance_Notification_For_WooCommerce
 */

defined('ABSPATH') || exit;

/**
 * Class for checking if specific products are present in an order
 */
class Product extends Base_Condition {
    
    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();
        
        // Enqueue required scripts
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    /**
     * Enqueue necessary scripts for product search
     */
    public function enqueue_scripts() {
        if (!is_admin() || !function_exists('get_current_screen')) {
            return;
        }
        
        $screen = get_current_screen();
        if (isset($screen->id) && strpos($screen->id, 'anfw') !== false) {
            wp_enqueue_script('wc-enhanced-select');
            wp_enqueue_style('woocommerce_admin_styles');
        }
    }
    
    /**
     * Get condition ID
     *
     * @return string
     */
    public function get_id() {
        return 'product_presence';
    }
    
    /**
     * Get condition name
     *
     * @return string
     */
    public function get_name($trigger_type= '') {
        return __('Product', 'advance-notification-for-woocommerce');
    }
    
    /**
     * Get group for this condition
     *
     * @return string
     */
    public function get_group() {
        return 'Product';
    }
    
    /**
     * Get available operators
     *
     * @return array
     */
    public function get_operators($logic, $trigger_type = '') {
        if($trigger_type == 'new_order') {
            return $this->get_operators_for_new_order();
        } elseif($trigger_type == 'low_stock' || $trigger_type == 'out_of_stock' || $trigger_type == 'backorder') {
            return $this->get_operators_for_low_stock();
        }
        return array();
    }

    public function get_operators_for_new_order() {
        return array(
            'in' => __('Order Contains product', 'advance-notification-for-woocommerce'),
            'not_in' => __('Order Does not contain product', 'advance-notification-for-woocommerce'),
            'all_in' => __('Order Contains all this products', 'advance-notification-for-woocommerce'),
        );
    }

    public function get_operators_for_low_stock() {
        return array(
            'in' => __('Only For this product', 'advance-notification-for-woocommerce'),
            'not_in' => __('Any product other than this', 'advance-notification-for-woocommerce'),
        );
    }

    function supported_trigger(){
        return array(
            'new_order',
            'low_stock',
            'out_of_stock',
            'backorder'
        );
    }
    
    /**
     * Get HTML for value field
     *
     * @param string $html Current HTML.
     * @param string $condition_id Condition ID.
     * @param string $unused_param Unused parameter.
     * @param mixed $current_value Current value.
     * @return string
     */
    public function get_value_field($html, $condition_id, $unused_param, $current_value) {
        $current_product_ids = array();
        
        if (!empty($current_value)) {
            $current_product_ids = is_array($current_value) ? $current_value : array($current_value);
        }
        
        ob_start();
        ?>
        <div class="pisol-anfw-product-wrapper">
            <select 
                name="pisol_anfw_conditions[<?php echo esc_attr($condition_id); ?>][value][]"
                class="wc-product-search"
                data-placeholder="<?php esc_attr_e('Search for products...', 'advance-notification-for-woocommerce'); ?>"
                data-action="woocommerce_json_search_products_and_variations"
                multiple="multiple"
                style="width: 100%;"
            >
                <?php if (!empty($current_product_ids)) : 
                    foreach ($current_product_ids as $product_id) :
                        $product = wc_get_product($product_id);
                        if ($product) : ?>
                            <option value="<?php echo esc_attr($product_id); ?>" selected="selected">
                                <?php echo esc_html($product->get_formatted_name()); ?>
                            </option>
                        <?php endif;
                    endforeach;
                endif; ?>
            </select>
            <p class="description"><?php esc_html_e('Select one or more products', 'advance-notification-for-woocommerce'); ?></p>
        </div>
        <?php
        return ob_get_clean();
    }
    
    public function is_match($return, $data, $trigger_type, $operator, $value) {
        if($trigger_type == 'new_order') {
            return $this->is_match_for_new_order($return, $data, $trigger_type, $operator, $value);
        }

        if($trigger_type == 'low_stock' || $trigger_type == 'out_of_stock' || $trigger_type == 'backorder') {
            return $this->is_match_for_low_stock($return, $data, $trigger_type, $operator, $value);
        }

        return false;
    }

    public function is_match_for_low_stock($return, $data, $trigger_type, $operator, $value) {
        // For low stock trigger, we assume the value is a product ID or an array of product IDs
        if (empty($value)) {
            return false;
        }
        
        // Ensure we have an array of product IDs
        $triggering_low_stock_for_product = $data['product'] && is_object($data['product']) ? $data['product']->get_id() : null;

        if( !$triggering_low_stock_for_product ) {
            return false;
        }
        $product_ids = is_array($value) ? $value : array($value);
        
        switch ($operator) {
            case 'in':
                return $triggering_low_stock_for_product && in_array($triggering_low_stock_for_product, $product_ids);
            case 'not_in':
                return $triggering_low_stock_for_product && !in_array($triggering_low_stock_for_product, $product_ids);
            case 'all_in':
                return $triggering_low_stock_for_product && in_array($triggering_low_stock_for_product, $product_ids);
            default:
                return false;
        }
    }
    
    public function is_match_for_new_order($return, $data, $trigger_type, $operator, $value) {
        $order = isset($data['order']) ? $data['order'] : null;
        
        if (!$order || !is_a($order, 'WC_Order') || empty($value)) {
            return false;
        }
        
        // Ensure we have an array of product IDs
        $product_ids = is_array($value) ? $value : array($value);
        
        // Get all product IDs from order
        $order_product_ids = array();
        foreach ($order->get_items() as $item) {
            $order_product_ids[] = $item->get_product_id();
            $variation_id = $item->get_variation_id();
            if ($variation_id) {
                $order_product_ids[] = $variation_id;
            }
        }
        
        // Check if any of the specified products are in the order
        $found_products = array_intersect($product_ids, $order_product_ids);
        $found_count = count($found_products);
        
        switch ($operator) {
            case 'in':
                return $found_count > 0; // Any product is in order
            case 'not_in':
                return $found_count === 0; // No product is in order
            case 'all_in':
                return $found_count === count($product_ids); // All products are in order
            default:
                return false;
        }
    }
}