<?php
namespace PISOL\ANFW\ADMIN;

/**
 * Product Category Condition
 *
 * @package Auto_Assign_Order_Tags_For_WooCommerce
 */

defined('ABSPATH') || exit;

/**
 * Class for product category condition
 */
class Product_Category extends Base_Condition {
    
    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();
    }

    function supported_trigger(){
        return array(
            'new_order',
            'low_stock',
            'out_of_stock',
            'backorder',
        );
    }
    
    /**
     * Get condition ID
     *
     * @return string
     */
    public function get_id() {
        return 'product_category';
    }
    
    /**
     * Get condition name
     *
     * @return string
     */
    public function get_name($trigger_type= '') {
        return __('Product Category', 'advance-notification-for-woocommerce');
    }
    
    /**
     * Get group for this condition
     *
     * @return string
     */
    public function get_group() {
        return 'Product';
    }
    
    /**
     * Get available operators
     *
     * @return array
     */
    public function get_operators($logic, $trigger_type = '') {
        if($trigger_type == 'new_order') {
            return $this->get_operators_for_new_order();
        } elseif($trigger_type == 'low_stock' || $trigger_type == 'out_of_stock' || $trigger_type == 'backorder') {
            return $this->get_operators_for_low_stock();
        }
        return array();
    }

    public function get_operators_for_new_order() {
        return array(
            'in' => __('Contains product from categories', 'advance-notification-for-woocommerce'),
            'not_in' => __('Does not contain product from categories', 'advance-notification-for-woocommerce'),
        );
    }

    public function get_operators_for_low_stock() {
        return array(
            'in' => __('Product from this category', 'advance-notification-for-woocommerce'),
            'not_in' => __('Product not from this category', 'advance-notification-for-woocommerce'),
        );
    }
    
    /**
     * Get HTML for value field
     *
     * @param string $html Current HTML.
     * @param string $condition_id Condition ID.
     * @param string $unused_param Unused parameter.
     * @param string $current_value Current value.
     * @return string
     */
    public function get_value_field($html, $condition_id, $unused_param, $current_value) {
        $categories = get_terms(array(
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
        ));
        
        ob_start();
        ?>
        <div class="pisol-anfw-category-value">
            <select 
                name="pisol_anfw_conditions[<?php echo esc_attr($condition_id); ?>][value][]"
                class="pisol-anfw-multi-select"
                multiple="multiple"
                style="width: 300px;"
                data-condition-id="<?php echo esc_attr($condition_id); ?>"
            >
                <?php 
                $current_values = !empty($current_value) ? $current_value : array();
                
                if (!is_wp_error($categories) && !empty($categories)) {
                    foreach ($categories as $category) : 
                    ?>
                        <option value="<?php echo esc_attr($category->term_id); ?>" <?php echo in_array((string)$category->term_id, $current_values) ? 'selected="selected"' : ''; ?>><?php echo esc_html($category->name); ?></option>
                    <?php 
                    endforeach;
                }
                ?>
            </select>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Check if condition is met
     *
     * @param mixed $order WC_Order object.
     * @param string $operator Operator.
     * @param mixed $value Value to compare against.
     * @return bool
     */
    public function is_match($return, $data, $trigger_type, $operator, $value) {
        if($trigger_type == 'new_order') {
            return $this->is_match_for_new_order($return, $data, $trigger_type, $operator, $value);
        }

        if($trigger_type == 'low_stock' || $trigger_type == 'out_of_stock' || $trigger_type == 'backorder') {
            return $this->is_match_for_low_stock($return, $data, $trigger_type, $operator, $value);
        }

        return false;
    }

     public function is_match_for_low_stock($return, $data, $trigger_type, $operator, $value) {
        // For low stock trigger, we assume the value is a product ID or an array of product IDs
        if (empty($value)) {
            return false;
        }

        // Ensure we have an array of product IDs
        $triggering_low_stock_for_product = $data['product'] && is_object($data['product']) ? $data['product']->get_id() : null;

        if( !$triggering_low_stock_for_product ) {
            return false;
        }

        $product_categories = wc_get_product_term_ids( $triggering_low_stock_for_product, 'product_cat');

        if (empty($product_categories)) {
            return false;
        }

        $category_ids = !empty($value) && is_array($value) ? $value : array();

        if(empty($category_ids)) return false;

        $intersect = array_intersect($product_categories, $category_ids);
        // Check if the product belongs to the specified categories
        switch ($operator) {
            case 'in':
                return  count($intersect) > 0 ? true : false;
            case 'not_in':
                return  count($intersect) == 0 ? true : false;
            default:
                return false;
        }
    }

    public function is_match_for_new_order($return, $data, $trigger_type, $operator, $value) {
        $order = isset($data['order']) ? $data['order'] : null;

        if (!$order || !is_a($order, 'WC_Order') || empty($value)) {
            return false;
        }
        
        $items = $order->get_items();
        if (empty($items)) {
            return false;
        }
        
        // Always handle values as comma-separated list
        $category_ids = !empty($value) && is_array($value) ? $value : array();
        
        $found_categories = array();
        
        foreach ($items as $item) {
            $product_id = $item->get_product_id();
            $product_categories = wc_get_product_term_ids($product_id, 'product_cat');
            
            // Check each product category against our list of categories
            foreach ($product_categories as $product_cat) {
                if (in_array($product_cat, $category_ids)) {
                    $found_categories[] = $product_cat;
                }
            }
        }
        
        $contains_category = !empty($found_categories);
        
        switch ($operator) {
            case 'in':
                return $contains_category;
            case 'not_in':
                return !$contains_category;
            default:
                return false;
        }
    }
}
