<?php
namespace PISOL\ANFW\ADMIN;

defined('ABSPATH') || exit;

/**
 * Class for adding actions meta box to notification creation/edit screen
 */
class Notification_Form {
    
    /**
     * The single instance of this class
     *
     * @var Notification_Form
     */
    private static $instance = null;

    /**
     * Main instance
     * 
     * Ensures only one instance is loaded or can be loaded.
     *
     * @return Order_Tag_Actions
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    public function __construct() {
        // Add meta box to tag edit screen
        add_action('add_meta_boxes', array($this, 'add_actions_meta_box'));
        
        // Save action settings when tag is saved
        add_action('save_post_pisol_anfw_notice', array($this, 'save_actions'), 10, 3);
        
        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    /**
     * Enqueue scripts and styles for the admin area
     *
     * @param string $hook The current admin page.
     */
    public function enqueue_scripts($hook) {
        global $post;
        
        // Only enqueue on tag edit screen
        if (!($hook == 'post.php' && isset($post) && $post->post_type == 'pisol_anfw_notice') && 
            !($hook == 'post-new.php' && isset($_GET['post_type']) && $_GET['post_type'] == 'pisol_anfw_notice')) {
            return;
        }
        
        // Enqueue the actions.js script
        wp_enqueue_script(
            'pisol-anfw-actions',
            plugin_dir_url(__FILE__) . 'assets/js/actions.js',
            array('jquery'),
            filemtime(plugin_dir_path(__FILE__) . 'assets/js/actions.js'),
            true
        );
    }

    /**
     * Add meta box for tag actions
     */
    public function add_actions_meta_box() {
        add_meta_box(
            'pisol_anfw_tag_actions',
            __('Notification types', 'advance-notification-for-woocommerce'),
            array($this, 'render_actions_meta_box'),
            'pisol_anfw_notice',
            'normal',
            'default'
        );
    }

    /**
     * Render the actions meta box content
     *
     * @param \WP_Post $post The post object.
     */
    public function render_actions_meta_box($post) {
        // Add nonce for security
        wp_nonce_field('pisol_anfw_save_actions', 'pisol_anfw_actions_nonce');
        
        do_action('pisol_anfw_action_field', $post);
    }

    /**
     * Save actions when the tag is saved
     *
     * @param int $post_id The post ID.
     * @param \WP_Post $post The post object.
     * @param bool $update Whether this is an existing post being updated.
     */
    public function save_actions($post_id, $post, $update) {
        // Check if nonce is valid
        if (!isset($_POST['pisol_anfw_actions_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['pisol_anfw_actions_nonce'])), 'pisol_anfw_save_actions')) {
            return;
        }

        // Check if user has permission
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Don't save during autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Check if correct post type
        if ('pisol_anfw_notice' !== $post->post_type) {
            return;
        }

        do_action('pisol_anfw_action_field_save', $post_id, $post, $update);
    }
}