/**
 * JavaScript for handling dynamic condition form with groups
 */
(function($) {
    'use strict';
    
    var PISOL_ANFW_Conditions = {
        init: function() {
            this.bindEvents();
            this.initExistingConditions();
            this.initSelectWooFields();
        },
        
        bindEvents: function() {
            // Group related events
            $(document).on('click', '.pisol-anfw-add-group', this.addGroup);
            $(document).on('click', '.pisol-anfw-remove-group', this.removeGroup);
            $(document).on('click', '.pisol-anfw-add-condition-to-group', this.addConditionToGroup);
            
            // Condition related events
            $(document).on('click', '.pisol-anfw-remove-condition', this.removeCondition);
            $(document).on('change', '.pisol-anfw-condition-type', this.handleConditionTypeChange);
        },
        
        initExistingConditions: function() {
            // Make sure any existing conditions are initialized properly
            $('.pisol-anfw-condition-type').each(function() {
                if ($(this).val()) {
                    var $condition = $(this).closest('.pisol-anfw-condition');
                    var $conditionData = $condition.find('.pisol-anfw-condition-data');

                    // Store any existing values from data attributes for debugging
                    console.log('Initializing condition:', {
                        conditionId: $condition.data('condition-id'),
                        type: $conditionData.data('condition-type'),
                        operator: $conditionData.data('condition-operator'),
                        value: $conditionData.data('condition-value')
                    });
                    
                    // Now trigger the change to load the operator and value fields
                    $(this).trigger('change');
                }
            });
        },
        
        // Initialize SelectWoo on all select fields that need it
        initSelectWooFields: function() {
            // Initialize SelectWoo on country select fields
            $('.pisol-anfw-multi-select').each(function() {
                if (!$(this).data('select2')) {
                    $(this).selectWoo();
                }
            });
            
            // Initialize WooCommerce enhanced select for product search
            $(document.body).trigger('wc-enhanced-select-init');
            
            // Handle dynamic fields loaded via AJAX
            $(document).on('pisol_anfw_condition_fields_loaded', function() {
                $('.pisol-anfw-multi-select').each(function() {
                    if (!$(this).data('select2')) {
                        $(this).selectWoo();
                    }
                });
                
                // Also re-initialize WooCommerce enhanced select for product search when fields are loaded
                $(document.body).trigger('wc-enhanced-select-init');
            });
        },
        
        // Add a new condition group
        addGroup: function(e) {
            e.preventDefault();
            
            var groupId = new Date().getTime();
            var $groupsList = $('#pisol-anfw-groups-list');
            var html = '';
            
            // Add logic divider if this is not the first group
            if ($groupsList.children('.pisol-anfw-group').length > 0) {
                if (typeof wp !== 'undefined' && wp.template) {
                    var dividerTemplate = wp.template('pisol-anfw-group-logic-divider');
                    html += dividerTemplate({ groupId: groupId });
                } else {
                    html += PISOL_ANFW_Conditions.generateGroupLogicDividerHTML(groupId);
                }
            }
            
            // Add the group
            if (typeof wp !== 'undefined' && wp.template) {
                var groupTemplate = wp.template('pisol-anfw-group');
                html += groupTemplate({ groupId: groupId });
            } else {
                html += PISOL_ANFW_Conditions.generateGroupHTML(groupId);
            }
            
            $groupsList.append(html);
            
            // Add a condition to the new group
            var $newGroup = $groupsList.find('.pisol-anfw-group[data-group-id="' + groupId + '"]');
            $newGroup.find('.pisol-anfw-add-condition-to-group').trigger('click');
        },
        
        // Remove a condition group
        removeGroup: function(e) {
            e.preventDefault();

            var $group = $(this).closest('.pisol-anfw-group');
            var $logicDivider = $group.prev('.pisol-anfw-group-logic-divider');

            // If this is the first group and there's more than one, remove the next logic divider
            if ($group.prev('.pisol-anfw-group-logic-divider').length === 0 && $group.next('.pisol-anfw-group-logic-divider').length > 0) {
                $group.next('.pisol-anfw-group-logic-divider').remove();
            } else if ($logicDivider.length > 0) {
                // Otherwise remove the previous logic divider
                $logicDivider.remove();
            }
            
            // Remove the group
            $group.remove();
        },
        
        // Add a condition to a specific group
        addConditionToGroup: function(e) {
            e.preventDefault();

            var $group = $(this).closest('.pisol-anfw-group');
            var groupId = $group.data('group-id');
            var $conditionsList = $group.find('.pisol-anfw-conditions-list');
            var conditionId = new Date().getTime();
            var html = '';
            
            // Add the condition (without any logic divider)
            if (typeof wp !== 'undefined' && wp.template) {
                var conditionTemplate = wp.template('pisol-anfw-condition');
                html += conditionTemplate({ groupId: groupId, conditionId: conditionId });
            } else {
                html += PISOL_ANFW_Conditions.generateConditionHTML(groupId, conditionId);
            }
            
            $conditionsList.append(html);
        },
        
        // Remove a condition from a group
        removeCondition: function(e) {
            e.preventDefault();

            var $condition = $(this).closest('.pisol-anfw-condition');
            
            // Simply remove the condition (no need to handle logic dividers anymore)
            $condition.remove();
        },
        
        // Generate HTML for a new group (fallback without wp.template)
        generateGroupHTML: function(groupId) {
            var html = '<div class="pisol-anfw-group" data-group-id="' + groupId + '">';
            html += '<div class="pisol-anfw-group-header">';
            html += '<div class="pisol-anfw-group-title">Condition Group</div>';
            html += '<div class="pisol-anfw-group-match-type">';
            html += '<select name="pisol_anfw_conditions[' + groupId + '][match_type]" class="pisol-anfw-match-type">';
            html += '<option value="all">All conditions match</option>';
            html += '<option value="any">Any condition matches</option>';
            html += '</select>';
            html += '</div>';
            html += '<div class="pisol-anfw-group-actions">';
            html += '<button type="button" class="button pisol-anfw-add-condition-to-group">Add Condition</button>';
            html += '<button type="button" class="button pisol-anfw-remove-group">Remove Group</button>';
            html += '</div>';
            html += '</div>';
            html += '<div class="pisol-anfw-group-content">';
            html += '<div class="pisol-anfw-conditions-list"></div>';
            html += '</div>';
            html += '</div>';
            
            return html;
        },
        
        // Generate HTML for a group logic divider (fallback without wp.template)
        generateGroupLogicDividerHTML: function(groupId) {
            return '<div class="pisol-anfw-group-logic-divider">' +
                '<select name="pisol_anfw_conditions[' + groupId + '][logic]" class="pisol-anfw-group-logic">' +
                '<option value="AND">AND</option>' +
                '<option value="OR">OR</option>' +
                '</select>' +
                '</div>';
        },
        
        // Generate HTML for a new condition (fallback without wp.template)
        generateConditionHTML: function(groupId, conditionId) {
            var html = '<div class="pisol-anfw-condition" data-condition-id="' + conditionId + '">';
            html += '<div class="pisol-anfw-condition-content">';
            html += '<select name="pisol_anfw_conditions[' + groupId + '][conditions][' + conditionId + '][type]" class="pisol-anfw-condition-type">';
            html += '<option value="">Select Condition</option>';
            
            // Add condition types if available in the localized data
            if (pisol_anfw_conditions && pisol_anfw_conditions.condition_types) {
                for (var group in pisol_anfw_conditions.condition_types) {
                    var groupConditions = pisol_anfw_conditions.condition_types[group];

                    // Check if it's a condition group (object) or a single condition
                    if (typeof groupConditions === 'object') {
                        html += '<optgroup label="' + group + '">';
                        for (var type in groupConditions) {
                            html += '<option value="' + type + '">' + groupConditions[type] + '</option>';
                        }
                        html += '</optgroup>';
                    } else {
                        html += '<option value="' + group + '">' + groupConditions + '</option>';
                    }
                }
            }
            
            html += '</select>';
            html += '<div class="pisol-anfw-condition-details" style="display:none;">';
            html += '<div class="pisol-anfw-operator-container"></div>';
            html += '<div class="pisol-anfw-value-container"></div>';
            html += '</div>';
            html += '<button type="button" class="button pisol-anfw-remove-condition">Remove</button>';
            html += '</div>';
            html += '</div>';
            
            return html;
        },
        
        // Generate HTML for a condition logic divider (fallback without wp.template)
        generateLogicDividerHTML: function(groupId, conditionId) {
            return '<div class="pisol-anfw-logic-divider">' +
                '<select name="pisol_anfw_conditions[' + groupId + '][conditions][' + conditionId + '][logic]" class="pisol-anfw-logic">' +
                '<option value="AND">AND</option>' +
                '<option value="OR">OR</option>' +
                '</select>' +
                '</div>';
        },
        
        // Handle condition type change to load operators and value fields
        handleConditionTypeChange: function() {
            var $condition = $(this).closest('.pisol-anfw-condition');
            var $group = $(this).closest('.pisol-anfw-group');
            var $details = $condition.find('.pisol-anfw-condition-details');
            var $operatorContainer = $condition.find('.pisol-anfw-operator-container');
            var $valueContainer = $condition.find('.pisol-anfw-value-container');
            var $conditionData = $condition.find('.pisol-anfw-condition-data');
            var conditionType = $(this).val();
            var conditionId = $condition.data('condition-id');
            var groupId = $group.data('group-id');
            
            // Get existing values from data attributes
            var currentOperator = $conditionData.data('condition-operator') || '';
            var currentValue = $conditionData.data('condition-value') || '';
            
            // Hide details if no condition type selected
            if (!conditionType) {
                $details.hide();
                return;
            }
            
            // Show loading indicators
            $operatorContainer.html('<span class="spinner is-active"></span>');
            $valueContainer.html('<span class="spinner is-active"></span>');
            $details.show();
            
            // AJAX request to get operator and value fields
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'pisol_anfw_get_condition_fields',
                    condition_type: conditionType,
                    condition_id: groupId + '_' + conditionId, // Include group ID
                    operator: currentOperator,
                    value: currentValue,
                    nonce: pisol_anfw_conditions.nonce,
                    trigger_type: pisol_anfw_conditions.trigger_type || ''
                },
                success: function(response) {
                    if (response.success) {
                        if (response.data.operator_html) {
                            $operatorContainer.html(response.data.operator_html);
                            
                            // Update the name attribute to include group ID
                            $operatorContainer.find('select').attr('name',
                                'pisol_anfw_conditions[' + groupId + '][conditions][' + conditionId + '][operator]');

                            // Trigger change event on the operator to initialize any dependent logic
                            var $operator = $operatorContainer.find('.pisol-anfw-operator');
                            if ($operator.length) {
                                $operator.trigger('change');
                            }
                        } else {
                            $operatorContainer.html('');
                        }
                        
                        if (response.data.value_html) {
                            $valueContainer.html(response.data.value_html);
                            
                            // Update the name attribute in any input/select elements
                            $valueContainer.find('select, input').each(function() {
                                var oldName = $(this).attr('name');
                                if (oldName) {
                                    // Handle multiple select fields with [] at the end
                                    var newName;
                                    if (oldName.indexOf('pisol_anfw_temp[value]') === 0) {
                                        // This is our special case for multiple selects
                                        newName = 'pisol_anfw_conditions[' + groupId + '][conditions][' + conditionId + '][value][]';
                                    } else {
                                        // Regular field name replacement
                                        newName = oldName.replace(
                                            /pisol_anfw_conditions\[([^\]]+)\]/,
                                            'pisol_anfw_conditions[' + groupId + '][conditions][' + conditionId + ']'
                                        );
                                    }
                                    $(this).attr('name', newName);
                                }
                            });
                            
                            // Initialize select2 on any new select fields
                            $valueContainer.find('select').each(function() {
                                if ($(this).data('select2')) {
                                    $(this).select2('destroy');
                                }
                                $(this).select2();
                            });
                            
                            // Trigger event for other components to hook into
                            $(document).trigger('pisol_anfw_condition_fields_loaded');
                        } else {
                            $valueContainer.html('');
                        }
                    } else {
                        $operatorContainer.html('<p class="error">' + response.data + '</p>');
                        $valueContainer.html('');
                    }
                },
                error: function() {
                    $operatorContainer.html('<p class="error">Error loading condition fields</p>');
                    $valueContainer.html('');
                }
            });
        }
    };
    
    $(document).ready(function() {
        PISOL_ANFW_Conditions.init();
    });
    
})(jQuery);
